/**
 * \title ddxcat
 * \author Peter Corke
 *
 * Utility to concatentate ddx logfiles into a single file.  The header
 * is copied only once.  Subsequent headers are checked against the first
 * header to ensure output file consistency.
 *
 * Like cat output is to stdout.
 */

/**
 * \page ddxcat ddxcat
 *
 * Utility to concatentate ddx logfiles into a single file.  The header
 * is copied only once.  Subsequent headers are checked against the first
 * header to ensure output file consistency.
 *
 * Like cat output is to stdout.
 */

#include	<stdlib.h>
#include	<stdio.h>
#include	<string.h>

char	header0[BUFSIZ], header[BUFSIZ];

int ddxcat(char *filename, char *header, int copyheader);

int main(int ac, char *av[])
{
	int	i;

	if (ac == 0) {
		fprintf(stderr, "Usage: ddxcat logfiles\n");
		exit(0);
	}

	for (i=1; i<ac; i++) {
		if (i == 1) 
			/* first file, keep the header */
			ddxcat(av[i], header0, 1);
		else {
			/* subsequent files, check the header */
			header[0] = '\0';
			ddxcat(av[i], header, 0);
			if (strcmp(header0, header) != 0) {
				fprintf(stderr, "header mismatch\n");
				fprintf(stderr, "%s\n----\n%s\n", header0, header);
				exit(1);
			}
		}
	}
	return 0;
}

int
ddxcat(char *filename, char *header, int copyheader)
{
	FILE	*fp;
	char	buf[BUFSIZ];
	int	i, count = 0;

	if ((fp = fopen(filename, "r")) == NULL) {
		fprintf(stderr, "cant open %s for reading\n", filename);
		exit(1);
	}
		

	/* read the header */
	while (1) {
		fgets(buf, BUFSIZ, fp);

		/* write the header line to stdout */
		if (copyheader)
			fputs(buf, stdout);

		if (strcmp(buf, "_EOH_\n") == 0)
			break;

		/* append non comment lines to header copy */
		if (buf[0] == '%')
			continue;

		if (header != NULL)
			strcat(header, buf);
	}

	/* skip the alignment bytes */
	for (i=0; i<4; i++) {
		int	c;

		c = fgetc(fp);
		if (copyheader)
			fputc(c, stdout);
	}

	/* cat the rest of the file to stdout */
	while ((i = fread(buf, 1, BUFSIZ, fp)) > 0) {
		fwrite(buf, i, 1, stdout);
		count += i;
	}
	fprintf(stderr, "file %s: %d bytes %s\n", filename, count,
		copyheader ? "written" : "appended");

	return 0;
}
