
/***********************************************************************
 * ddx.h - ddx client API
 *
 * CSIRO MANUFACTURING SCIENCE & TECHNOLOGY
 * HIGH-SPEED SERIAL INTERFACE PROJECT
 * 
 * $Id: ddx.h 3068 2008-05-15 03:58:27Z roy029 $
 * 
 * Copyright (c) CSIRO Manufacturing Science & Technology.
 *
 ***********************************************************************
 */

#ifndef __ddx_h__
#define __ddx_h__

#ifdef  __cplusplus
extern "C" {
#endif


#include <sys/types.h>
#include <unistd.h>

#include <rtx/time.h>
#include <rtx/mutex.h>
#include <rtx/inet.h>
#include <rtx/parse.h>

#include "store-ipc.h"
#include "store-header.h"
#include "store-arch.h"

#define DDX_MAGIC_NUM 0x44445801
/* Macros */

#ifdef __GNUC__
#define DDX_STATIC_ATTRIBUTE __attribute__((unused))
#else
#define DDX_STATIC_ATTRIBUTE 
#endif
#define DDX_STORE_TYPE(typename, typedefn) \
    typedef typedefn typename ; \
    static const char * _store_type_##typename DDX_STATIC_ATTRIBUTE = #typedefn " " #typename ";"

#define DDX_STORE_REGISTER_CONST(store, name) \
    ddx_store_register_const (store, #name, name)

#define DDX_STORE_REGISTER_TYPE(store, typename) \
    ddx_store_register_type (store, _store_type_##typename)
    
#define ddx_store_sync_read(item, d, ts, timeout) \
    ddx_store_read (item, d, ts, timeout, 1)

#define ddx_store_async_read(item, d, ts, staleness) \
    ddx_store_read (item, d, ts, staleness, 0)


/* Typedefs */

typedef struct _store_id {
    int pid;
    STORE_SHMEM_T * shmemP;
    STORE_SHMEM_T * semQShmemP;
    STORE_SEM_Q * semQ;
    RtxInet * server;
    double tmout;
    STORE_ARCH_TYPE myArchType;
    STORE_ARCH myArch;
    int verbose;
    RtxMutex * mutex;
    char buf[8192];
    char buf1[8192];
    unsigned int magicNum;
} DDX_STORE_ID;

typedef struct _store_item {
    DDX_STORE_ID * store;
    STORE_VAR_HEADER * headerP;
    STORE_MULTICAST_HEADER multiHeader;
    void * d;
    int varSize;
    int varId;
    char * varName;
    double timeout;
    double period;
    RtxMutex * mutex;
    RtxInet * multicastPort;
    STORE_SEM_T syncSem;
    STORE_SEM_T lockSem;
    char * varDefinition;
    char buf[8192];
    unsigned int magicNum;
} DDX_STORE_ITEM;

typedef struct _string_list {
	char * name;
	struct _string_list * next;
} DDX_STRING_LIST;

/* Functions */

/* Startup and shutdown */

int ddx_client_init (int verbose);
int ddx_client_done (void);

DDX_STORE_ID * ddx_store_open (const char * storeName, int storePortNumber, 
			       int timeout);
void ddx_store_check (DDX_STORE_ID * storeId,const char * header);
int ddx_store_close (DDX_STORE_ID * storeId);

/* String List management */
DDX_STRING_LIST * ddx_stringlist_pushfront(DDX_STRING_LIST * list, 
		const char * name);
DDX_STRING_LIST * ddx_stringlist_attachafter(DDX_STRING_LIST * list,
		const char * name);
char * ddx_stringlist_concatenate(DDX_STRING_LIST * list);
void ddx_stringlist_free(DDX_STRING_LIST * list);

/* General */

int ddx_store_register_const (DDX_STORE_ID * storeId, const char * name, int val);
int ddx_store_lookup_const (DDX_STORE_ID * storeId, const char * name, int * val);

int ddx_store_register_type (DDX_STORE_ID * storeId, const char * type);
char* ddx_store_lookup_type (DDX_STORE_ID * storeId, const char * typeName);

DDX_STRING_LIST * ddx_store_get_constant_list(DDX_STORE_ID * storeId);
DDX_STRING_LIST * ddx_store_get_type_list(DDX_STORE_ID * storeId);
DDX_STRING_LIST * ddx_store_get_variable_list(DDX_STORE_ID * storeId);

/* Clients */

DDX_STORE_ITEM * ddx_store_lookup_item (DDX_STORE_ID * storeId, 
					const char * itemDecl, const char * itemType,
					int itemSize);
void* ddx_store_var_pointer(DDX_STORE_ITEM *storeItem);

int ddx_store_read (DDX_STORE_ITEM * storeItem, void * d, RtxTime * ts, 
		    double timeArg, int skipCount);
int ddx_store_write (DDX_STORE_ITEM * storeItem, void * d, RtxTime * ts);
int ddx_store_read_direct (DDX_STORE_ITEM * storeItem, RtxTime * ts, 
			   double timeArg, int skipCount);
int ddx_store_write_direct (DDX_STORE_ITEM * storeItem, RtxTime * ts);
int ddx_store_read (DDX_STORE_ITEM * storeItem, void * d, RtxTime * ts, 
		    double timeArg, int skipCount);
int ddx_store_queue_read (DDX_STORE_ITEM * storeItem, void * d, RtxTime * ts, 
		    double timeArg, int *tailP);
int ddx_store_done_item (DDX_STORE_ITEM * storeItem);

/* General item-related */

int ddx_store_item_get_size (DDX_STORE_ITEM * storeItem);
int ddx_store_item_get_period (DDX_STORE_ITEM * storeItem);
int ddx_store_item_set_period (DDX_STORE_ITEM * storeItem, double period);
double ddx_store_item_get_timeout (DDX_STORE_ITEM * storeItem);
int ddx_store_item_set_timeout (DDX_STORE_ITEM * storeItem, double timeout);
RtxParseVar * ddx_store_item_get_parsed_var(DDX_STORE_ITEM * storeItem);
char * ddx_store_item_get_definition(DDX_STORE_ITEM * storeItem,int pretty);
char * ddx_store_item_get_type_definition(DDX_STORE_ITEM * storeItem, int pretty);

int ddx_store_item_set_queue (DDX_STORE_ITEM *storeItem, int queueLength);

/* For item serialisation */


int ddx_store_item_get_packsize(const DDX_STORE_ITEM * storeItem);
int ddx_store_item_pack(const DDX_STORE_ITEM * itemP, 
		const void * data, const RtxTime * ts, 
		unsigned char * buffer);
int ddx_store_item_unpack(DDX_STORE_ITEM * itemP, 
		void * data, RtxTime * ts, 
		RtxParseVar ** localVar, RtxParseVar ** remoteVar,
		unsigned char * buffer);


#ifdef  __cplusplus
}
#endif

#endif
