/*********************************************************************
 *
 * CSIRO Automation
 * Queensland Centre for Advanced Technologies
 * PO Box 883, Kenmore, QLD 4069, Australia
 * www.cat.csiro.au/cmst
 *
 * Copyright (c) CSIRO Manufacturing Science & Technology
 *
 *********************************************************************/

static char *rcsid = "$Id: catalog-main.c 2850 2008-04-01 04:37:35Z duf048 $";
#ifdef __GNUC__
#define DDX_STATIC_ATTRIBUTE __attribute__((unused))
#else
#define DDX_STATIC_ATTRIBUTE
#endif
static char *tagname DDX_STATIC_ATTRIBUTE = "$Name$";

/**
 * \file catalog-main.c
 * \brief DDX catalog
 * \author Pavan Sikka
 */

/**
 * \page catalog catalog
The catalog keeps track of all the stores that make up an
application. It also keeps track of all the data items that
are in the stores. The catalog is responsible to manage
 multicasting of data items by the store. The catalog supports the following options:
- -m multicast address to be used for the application (default 226.0). 
- -b port number to be used for multicasting (default 10000)
- -p port number on which the catalog listens for requests (default 6001)
- -v verbose level for diagnostics (default 0)
- -h print a help message

For the muticast address the remaining 2 bytes are taken from the localhost. 
For example, the default on 192.168.100.25 will be 226.0.100.25.

The catalog maintains two indexes: one for data items 
and one for data stores. The indexes are implemented as 
hashtables. 

Each data item is represented by a structure:

\code
struct _data_entry { 
Data                 *next; 
char                *name;
char                *type;
Store               *store; 
void                *pointer; 
Semaphore           sem;
int                 timeout;  
IPADDR              multicast_ip;
};

typdef struct _data_entry Data;
\endcode

The type is a string that describes the data structure 
in a similar fashion to the Logger3 configuration file.
Issues: does the catalog need to know the offset of the 
data item within the store, or is the store's own business.

Each store is represented by the structure:

\code
enum _endian {
    LITTLE, BIG
};

struct _store_entry {

Store        *next;
char         *storeName;
char         *hostName;
enum Endian  endian;
int          timeout;
IPADDR       ip_address;
int          port;
enum _endian endian;
unsigned int uptime;
};

typedef struct _store_entry Store;
\endcode

This information is provided by each store on a 
periodic basis at which point the timeout element is 
set to a small positive value. A periodic thread 
decrements the timeout element and if it reaches zero 
we assume that the store has failed.

\section Dialogs

The catalog listens for UDP packets on one well known 
port. Its main loop is simply

\code
for (;;) {
     recvfrom(udp_socket, &sender, &message, ...);
     deal_with(&message);
     sendto(sender, ...);
}
\endcode

Messages are sent back to the sender by UDP. All dialog 
strings have an initial command word followed by a list 
of tagged arguments. This format is readily extensible 
and also human readable. A small library to parse these 
strings will be written whose argument would be a table 
of valid tags and return token values.

\section StoreNet

This is a UDP 'network' that the catalog and the stores 
communicate via. The catalog is a well known address 
passed to each store through an environment variable. 
The stores do not know each other's IP address, but 
relay messages via the catalog. The catalog obtains the 
IP address of each store via the periodic heartbeat messages.

If the catalog informs a store that the variable lives 
in another store then it tells the requesting store the 
IP address of the store that can supply.

If a command string is sent to the catalog whose first 
word is relay then, that word is stripped, and the 
catalog sends the string to all stores except the one 
that sent the command to be relayed.

\section Store heartbeat

The store writes a string to the catalog:

\code
"store StoreName=store-name Endian=big|little"
\endcode

There is no reply.

\section Data item query

A store writes a string of the form:

\code
"query DataName=data-name"
\endcode

and the catalog responds with:

\code
"catalog DataName=data-name StoreIP=ip_addr DataStructure=data-type DataOffset=offset"
\endcode

or

\code 
"error Message=<not found> DataName=data-name"
\endcode

For the situation without a local store, wholesale 
access, the client writes directly to the catalog

\code
"query DataName=data-name"
\endcode

and the reply is

\code
"catalog DataName=data-name MultiCast=ip-addr DataStructure=data-type DataOffset=offset"
\endcode

or

\code
"error Message=<not found> DataName=data-name"
\endcode

\section Data item producer

A store declares a new data item to the catalog by 
writing a string of the form:

\code
"data DataName=data-name StoreName=store-name DataStructure=data-type Offset=offset"
\endcode

There is no reply. A store withdraws a data item by writing a string of the form:

\code
"delete DataName=data-name StoreName=store-name"
\endcode

There is no reply. The catalog will send a message to 
all stores informing them of this

\code
"delete DataName=data-name"
\endcode

and it is their responsibility to propogate this 
information to consumers.

\section  Dynamic data availability

A store may put data into the catalog that is currently 
not actually being produced. If a request comes to the 
store for that variable then it can launch a process 
that will produce it, before informing the consumer 
that it exists (by means of a configuration file). I 
don't think the catalog needs to know this particular detail.

\section Catalog threads
- 1. listen to StoreNet
- 2. timeout on stores
- 3. dialog with a store, created as needed after an enquiry on StoreNet
- 4. RTC server with commands to list top level variables, details of variables, set variable details, list the stores.
 *
 */

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <errno.h>
#include <signal.h>
#include <pthread.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <sys/stat.h>

#include <rtx/message.h>
#include <rtx/error.h>
#include <rtx/main.h>
#include <rtx/thread.h>
#include <rtx/signal.h>
#include <rtx/inet.h>
#include <rtx/getopt.h>

#include "store-msg.h"
#include "catalog-messages.h"
#include "catalog-main.h"
#include "catalog-handlers.h"
#include "ddx.h"

CATALOG mainCatalogStruct;
CATALOG * catalog = &mainCatalogStruct;

RtxGetopt catalogOpts[] = {
  {"port", "catalog port number",
   {
     {RTX_GETOPT_INT, &mainCatalogStruct.portNum, "port number"},
     RTX_GETOPT_END_ARG
   }
  },
  {"multicast", "catalog multicast address", 
   {
     {RTX_GETOPT_STR, &mainCatalogStruct.multiNet, "multicast address"},
     RTX_GETOPT_END_ARG
   }
  },
  {"mport", "catalog multicast port number",
   {
     {RTX_GETOPT_INT, &mainCatalogStruct.curMultiPort,
      "multicast port number"},
     RTX_GETOPT_END_ARG
   }
  },
  {"-daemon", "fork and run as a daemon", 
   {
     {RTX_GETOPT_SET, &mainCatalogStruct.forkFlag, ""},
     RTX_GETOPT_END_ARG
   }
  },
  RTX_GETOPT_END
};

char * catalogHelpStr = "DDX catalog";

/**
 * process request messages received from client stores
 *
 * @return -1 on error, 0 if OK
 */

int
catalog_process_request (
			 RtxInetConn * inetConn,    /**< Inet endpoint of
						     **  client store */
			 char * recvBuf,            /**< rx buffer */
			 char * sendBuf,            /**< tx buffer */
			 CATALOG * catalog          /**< global state */
			 )
{
    int numChars;
    MSG * mp;
    int ret;

    if ((mp = parse_message (catKeyTable, recvBuf)) == NULL) {
        rtx_error ("parse_message() failed on message from %s:%d [%s]", 
		   rtx_inet_print_net_addr (&inetConn->remote), 
		   rtx_inet_print_net_portnum (&inetConn->remote), recvBuf);
	strcpy (sendBuf, 
		"catalogErrorReply response = invalid message - "
		"parse failed\n");
	if ((numChars = rtx_inet_write (inetConn, sendBuf, 
					 strlen (sendBuf), NULL)) == -1) {
	    rtx_error ("rtx_inet_write()"
		       " failed in response to message [%s] "
		       "received "
		       "from %s:%d", recvBuf, 
		       rtx_inet_print_net_addr (&inetConn->remote), 
		       rtx_inet_print_net_portnum (&inetConn->remote));
	}
	return (-1);
    }
    if (identify_message (catMsgTable, mp) == -1) {
        rtx_error ("identify_message() failed on message from %s:%d [%s]", 
		   rtx_inet_print_net_addr (&inetConn->remote), 
		   rtx_inet_print_net_portnum (&inetConn->remote), recvBuf);
	strcpy (sendBuf, 
		"catalogErrorReply response = invalid message - "
		"identify failed\n");
	if ((numChars = rtx_inet_write (inetConn, sendBuf, 
					 strlen (sendBuf), NULL)) == -1) {
	    rtx_error ("rtx_inet_write()"
		       " failed in response to message [%s] "
		       "received "
		       "from %s:%d", recvBuf, 
		       rtx_inet_print_net_addr (&inetConn->remote), 
		       rtx_inet_print_net_portnum (&inetConn->remote));
	}
	return (-1);
    }
    catalog_debug (("catalog_request_handler_thread: message id = %d",
		    mp->id));
    ret = 0;
    switch (mp->id) {
        case MSG_CATALOG_REGISTER_STORE_REQUEST :
	    ret = catalog_handle_register_store_request (mp, recvBuf, sendBuf, 
							 inetConn, catalog);
	    break;
        case MSG_CATALOG_DEREGISTER_STORE_REQUEST :
	    ret = catalog_handle_deregister_store_request (mp, recvBuf, 
							   sendBuf,
							   inetConn, catalog);
	    break;
        case MSG_CATALOG_LOOKUP_VAR_REQUEST :
	    ret = catalog_handle_lookup_var_request (mp, recvBuf, sendBuf, 
						     inetConn, catalog);
	    break;
        case MSG_CATALOG_DONE_VAR_REQUEST :
	    ret = catalog_handle_done_var_request (mp, recvBuf, sendBuf, 
						   inetConn, catalog);
	    break;
        case MSG_CATALOG_GET_STORE_LIST_REQUEST :
	    ret = catalog_handle_get_store_list_request (mp, recvBuf, sendBuf, 
							 inetConn, catalog);
	    break;
        case MSG_CATALOG_GET_ITEM_LIST_REQUEST :
	    ret = catalog_handle_get_item_list_request (mp, recvBuf, sendBuf,
							inetConn, catalog);
	    break;
        case MSG_CATALOG_GET_STORE_INFO_REQUEST :
	    ret = catalog_handle_get_store_info_request (mp, recvBuf, sendBuf,
							 inetConn, catalog);
	    break;
        default :
	    rtx_error ("unknown message, id = %d from %s:%d [%s]", mp->id, 
		       rtx_inet_print_net_addr (&inetConn->remote), 
		       rtx_inet_print_net_portnum (&inetConn->remote),
		       recvBuf);
	    strcpy (sendBuf, "catalogErrorReply response = "
		    "unknown message id\n");
	    if ((numChars = rtx_inet_write (inetConn, 
					    sendBuf, 
					    strlen (sendBuf),
					    NULL)) == -1) {
	        rtx_error ("rtx_inet_write() failed in response"
			   " to message [%s] received "
			   "from %s:%d", recvBuf, 
			   rtx_inet_print_net_addr (&inetConn->remote), 
			   rtx_inet_print_net_portnum (&inetConn->remote));
	    }
	    ret = -1;
	    break;
    }
    free_message (catKeyTable, mp);
    if (ret == -1)
        return (rtx_error ("error handling request [%s] from %s:%d", recvBuf, 
			   rtx_inet_print_net_addr (&inetConn->remote), 
			   rtx_inet_print_net_portnum (&inetConn->remote)));
    return (0);
}

/**
 * cleanup when a connection is closed
 *
 * @return NULL
 */

void *
catalog_request_handler_cleanup (
				void * arg1,   /**< RtxInetTcpClient * */
				void * arg2    /**< CATALOG * */
				)
{
	RtxInetTcpClient * client;
	CATALOG * catalog;
	char * myHostName = NULL;
	STORE_LIST_ITEM *stp = NULL;


	client = (RtxInetTcpClient *) arg1;
	catalog = (CATALOG *) arg2;

	catalog_debug(("Executing cleanup (fd %d)",client->sock->sockfd));

    while ((stp = (STORE_LIST_ITEM *) rtx_list_iterate
	    (catalog->store)) != NULL) {
		if (stp->hostaddr.sockfd == client->sock->sockfd) {
			myHostName = stp->storeHostName;
			/* don't brake here to let rtx_list_iterate 
			 * finish and release its mutexes */
		}
    }
	
	if (myHostName == NULL) {
		/* already cleaned up, terminating */
		catalog_debug(("Store has terminated cleanly by itself"));
		return NULL;
	}
	catalog_debug(("Store '%s' has not terminated cleanly by itself",myHostName));

	catalog_deregister_store(myHostName,catalog);

	catalog_debug(("Store cleanup completed"));

	return NULL;
}

/**
 * thread started by the Inet server for each client store
 *
 * @return NULL
 */

void *
catalog_request_handler_thread (
				void * arg1,   /**< RtxInetTcpClient * */
				void * arg2    /**< CATALOG * */
				)
{
	RtxInetTcpClient * client;
	CATALOG * catalog;
	char * recvBuf, * sendBuf, ch;
	int done = 0, n = 1, i = 0;

	client = (RtxInetTcpClient *) arg1;
	catalog = (CATALOG *) arg2;
	rtx_error_init ("catalog[client_handler_thread]", 
			RTX_ERROR_STDERR | RTX_ERROR_MESSAGE, NULL);
	if (rtx_signal_block (SIGINT))
		rtx_error ("rtx_signal_block(SIGINT) failed");
	if (rtx_signal_block (SIGQUIT))
		rtx_error ("rtx_signal_block(SIGQUIT) failed");
	if (rtx_signal_block (SIGTSTP))
		rtx_error ("rtx_signal_block(SIGTSTP) failed");

	if ((recvBuf = calloc (1, CATALOG_BUF_SIZE)) == NULL) {
		rtx_error ("unable to allocate buffer");
		pthread_exit ((void *) 0);
	}
	if ((sendBuf = calloc (1, CATALOG_BUF_SIZE)) == NULL) {
		rtx_error ("unable to allocate buffer");
		free (recvBuf);
		pthread_exit ((void *) 0);
	}
	catalog_debug (("catalog-req-handler: starting (fd=%d)",client->sock->sockfd));
	while (! done) {
		i = 0; n = 1;
		while ((n > 0) && (i < (CATALOG_BUF_SIZE-1))) {
			if ((n = read (client->sock->sockfd, &ch, 1)) < 1) {
				done = 1;
				if ((n < 0) && (! catalog->done) && 
						(errno != ECONNRESET))
					rtx_error_errno ("read failed [%d]", errno);
				done = catalog->done;
				continue;
			}
			if (ch == '\n') {
				recvBuf[i++] = '\0';
				break;
			}
			recvBuf[i++] = ch;
		}
		if (n <= 0) {
			done = 1;
			continue;
		}
		/* Have a complete request for processing */
		catalog_debug (("catalog_request_handler_thread: "
					"from %s %d - %s", 
					rtx_inet_print_net_addr (&client->sock->remote),
					rtx_inet_print_net_portnum (&client->sock->remote),
					recvBuf));
		if (catalog_process_request (client->sock, recvBuf,
					sendBuf, catalog) == -1) {
			rtx_error_flush ("catalog_process_request() failed");
		}
	}
	catalog_debug (("catalog-req-handler: completed (n=%d)",n));
	if ((n == 0) || (errno == ECONNRESET))
		rtx_error_traceback_depth (0);
	free (recvBuf);
	free (sendBuf);
	return (0);
}

/**
 * initialize the multicast parameters for the catalog
 *
 * @return -1 on error, 0 if OK
 */

int
catalog_init_multi_addr (
			 CATALOG * catalog    /**< global state */
			 )
{
    RtxInetEndpoint addr;
    char * hostIpNum;
    int subnetNum, machineNum;

    if (rtx_inet_get_net_addr (catalog->myHostName, catalog->portNum,
			       &addr) == -1)
        return (rtx_error ("catalog_init_multi_addr:rtx_inet_get_net_addr"
			   "(%s:%d) failed", catalog->myHostName, 
			   catalog->portNum));
    if ((hostIpNum = rtx_inet_print_net_addr (&addr)) == NULL)
        return (rtx_error ("catalog_init_multi_addr: rtx_inet_print_net_addr"
			   "(%s:%d) failed", catalog->myHostName,
			   catalog->portNum));
    if (sscanf (hostIpNum, "%*d.%*d.%d.%d", &subnetNum, &machineNum) != 2)
        return (rtx_error ("catalog_init_multi_addr: sscanf() (%s:%d) "
			   "failed", catalog->myHostName, 
			   catalog->portNum));
    if (sprintf (catalog->multiAddr, "%s.%d.%d", catalog->multiNet, subnetNum,
		 machineNum) <= 0)
        return (rtx_error ("catalog_init_multi_addr: sprintf() (%s:%d) "
			   "failed", catalog->myHostName, 
			   catalog->portNum));
    return (0);
}

/**
 * initialize the catalog
 *
 * @return -1 on error, 0 if OK
 */

int
catalog_init (
	      CATALOG * catalog    /**< global state */
    )
{
    catalog->done = 0;
    catalog->pid = getpid();
    catalog->v = NULL;
    catalog->store = NULL;
    if ((catalog->mutex = rtx_mutex_init (NULL, RTX_MUTEX_DEFAULT,
					  0)) == NULL)
        return (rtx_error ("catalog_init: rtx_mutex_init() failed"));
    if ((catalog->store = rtx_list_init ()) == NULL)
        return (rtx_error ("catalog_init: rtx_list_init() failed"));
    if ((catalog->v = rtx_list_init ()) == NULL)
        return (rtx_error ("catalog_init: rtx_list_init() failed"));
    if (catalog_init_multi_addr (catalog) == -1)
        return (rtx_error ("catalog_init: catalog_init_multi_addr() "
			   "failed"));
    if ((catalog->server = rtx_inet_init (RTX_INET_TCP_SERVER,
					  NULL, catalog->portNum, NULL, 0,
					  catalog_request_handler_thread, 
					  catalog_request_handler_cleanup, catalog)) == NULL)
        return (rtx_error ("catalog_init: rtx_inet_init() "
			   "failed"));
    return (0);
}

/**
 * shutdown the catalog
 *
 * @return -1 on error, 0 if OK
 */

int
catalog_close (
        CATALOG * catalog
    )
{
    int errs = 0;

    catalog_debug (("Shutting down inet server"));
    if (rtx_inet_done (catalog->server) == -1) {
        rtx_error ("catalog_close: rtx_thread_join() failed");
	errs++;
    }
    if (rtx_mutex_destroy (catalog->mutex) == -1) {
        rtx_error ("catalog_close: rtx_mutex_destroy() failed");
	errs++;
    }
    catalog_debug (("Done"));
    if (errs)
        return (-1);
    return (0);
}

/**
 * main program
 *
 * @return program status (0 if OK)
 */

int
main (
      int argc,           /**< number of tokens on command line */
      char * argv[]       /**< array of command-line tokens */
    )
{
    int ret, errs = 0;
    pid_t pid;

    catalog->portNum = CATALOG_PORT_NUM;
    catalog->curMultiPort = CATALOG_INIT_MULTI_PORT;
    catalog->multiNet = CATALOG_MULTI_NET_ADDR;
    catalog->forkFlag = 0;
    if ((ret = RTX_GETOPT_CMD (catalogOpts, argc, argv, rcsid, 
			       catalogHelpStr)) == -1) {
        RTX_GETOPT_PRINT (catalogOpts, argv[0], rcsid, catalogHelpStr);
        exit (1);
    }
    catalog->verbose = rtx_getopt_get_verbose (0);

    /*      
     * Fork the process
     */     
    if (catalog->forkFlag) {
        pid = fork();
	if (pid < 0) {
	    rtx_error_errno_flush ("main: fork() failed\n"); 
	    exit(1);
	} else if (pid > 0) {
	    exit(0);
	}
	setsid();
	chdir("/");
	umask(0002);
    }

    rtx_main_init ("catalog", RTX_ERROR_STDERR | RTX_ERROR_MESSAGE);

    if (gethostname (catalog->myHostName, 64) == -1) {
	perror ("catalog");
	exit (1);
    }

    if (catalog_init (catalog)) {
        rtx_error_flush ("catalog_init() failed");
	exit (1);
    }

    catalog_debug (("catalog: pid = %d", catalog->pid));
    if (rtx_main_wait_shutdown (0) == -1) {
        rtx_error ("rtx_init_wait_shutdown() failed");
	errs++;
    }
    catalog->done = 1;
    catalog_debug (("catalog: exiting ..."));
    if (catalog_close (catalog) == -1) {
        rtx_error ("catalog_close() failed");
	errs++;
    }
    if (errs) {
        rtx_error_flush ("errors while shutting down");
	return (1);
    }
    return (0);
}

