
/***********************************************************************
 * catalog-handlers.c - ddx catalog message handling functions
 *
 * CSIRO MANUFACTURING SCIENCE & TECHNOLOGY
 * HIGH-SPEED SERIAL INTERFACE PROJECT
 * 
 * $Id: catalog-handlers.c 2200 2007-11-23 05:46:38Z roy029 $
 * 
 * Copyright (c) CSIRO Manufacturing Science & Technology.
 *
 ***********************************************************************
 */

#ifdef __GNUC__
#define DDX_STATIC_ATTRIBUTE __attribute__((unused))
#else
#define DDX_STATIC_ATTRIBUTE
#endif
static char *rcsid DDX_STATIC_ATTRIBUTE = "$Id: catalog-handlers.c 2200 2007-11-23 05:46:38Z roy029 $";

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <errno.h>
#include <signal.h>
#include <pthread.h>

#include <rtx/message.h>
#include <rtx/error.h>
#include <rtx/list.h>

#include "store-ipc.h"
#include "store-msg.h"
#include "catalog-messages.h"
#include "store-messages.h"
#include "store-mem.h"
#include "catalog-main.h"
#include "catalog-handlers.h"

int
catalog_send_error_response (
			     char * msg,
			     int keyId,
			     char * txbuf,
			     RtxInetConn * fromAddr,
			     CATALOG * catalog
    )
{
    int numChars;

    sprintf (txbuf, "%s %s = %s %s = %s%s\n",
	     catKeyTable[MSG_TOK_CATALOG_ERROR_REPLY].name,
	     catKeyTable[MSG_TOK_CATALOG_RESPONSE].name, msg,
	     catKeyTable[MSG_TOK_CATALOG_MESSAGE_ID].name, 
	     "catalogToken_", catKeyTable[keyId].name);
    if ((numChars = rtx_inet_write (fromAddr, txbuf, strlen (txbuf), 
				    NULL)) == -1) {
        return (rtx_error ("catalog_send_error_response: "
			   "rtx_inet_write() failed"));
    }
    return (0);
}
	
int
catalog_send_ok_response (
			  int keyId,
			  char * txbuf,
			  RtxInetConn * fromAddr,
			  CATALOG * catalog
			  )
{
    int numChars;

    sprintf (txbuf, "%s %s = %s\n",
	     catKeyTable[keyId].name,
	     catKeyTable[MSG_TOK_CATALOG_RESPONSE].name, "OK");
    if ((numChars = rtx_inet_write (fromAddr, txbuf, strlen (txbuf), 
				    NULL)) == -1) {
        return (rtx_error ("catalog_send_ok_response: "
			   "rtx_inet_write() failed"));
    }
    return (0);
}
	
int
catalog_handle_register_store_request (
				       MSG * mp,
				       char * rxbuf,
				       char * txbuf,
				       RtxInetConn * fromAddr,
				       CATALOG * catalog
				       )
{
    int numChars;
    STORE_LIST_ITEM storeInfo, * storeItemP;
    char * msgParm, * storeHostName;

    catalog_debug (("catalog_handle_register_store_request: handling"));
    if (catalog->verbose)
        rtx_list_print (catalog->store);
    if ((storeHostName = get_message_part (catKeyTable, mp, 
	    MSG_TOK_CATALOG_HOST_NAME)) == NULL) {
        catalog_send_error_response ("unable to extract hostname",
				     MSG_TOK_CATALOG_REGISTER_STORE_REPLY,
				     txbuf, fromAddr, catalog);
	return (rtx_error ("catalog_handle_register_store_request: unable to "
			   "get hostname"));
    }
    if ((msgParm = get_message_part (catKeyTable, mp, 
	    MSG_TOK_CATALOG_PORT_NUM)) == NULL) {
        catalog_send_error_response ("unable to extract port number",
				     MSG_TOK_CATALOG_REGISTER_STORE_REPLY,
				     txbuf, fromAddr, catalog);
	return (rtx_error ("catalog_handle_register_store_request: unable to "
			   "get port number"));
    }
    if (sscanf (msgParm, "%d", &(storeInfo.portNum)) != 1) {
        catalog_send_error_response ("invalid port number",
				     MSG_TOK_CATALOG_REGISTER_STORE_REPLY,
				     txbuf, fromAddr, catalog);
	return (rtx_error ("catalog_handle_register_store_request: invalid (%d) "
			   "port number", storeInfo.portNum));
    }
    if ((msgParm = get_message_part (catKeyTable, mp, 
	    MSG_TOK_CATALOG_ALIGNMENT)) == NULL) {
        catalog_send_error_response ("unable to extract alignment",
				     MSG_TOK_CATALOG_REGISTER_STORE_REPLY,
				     txbuf, fromAddr, catalog);
	return (rtx_error ("catalog_handle_register_store_request: unable to "
			   "get alignments"));
    }
    if (sscanf (msgParm, "%d %d %d %d %d %d", &(storeInfo.alignment[0]),
		&(storeInfo.alignment[1]), &(storeInfo.alignment[2]),
		&(storeInfo.alignment[3]), &(storeInfo.alignment[4]),
		&(storeInfo.alignment[5])) != 6) {
        catalog_send_error_response ("invalid alignments",
				     MSG_TOK_CATALOG_REGISTER_STORE_REPLY,
				     txbuf, fromAddr, catalog);
	return (rtx_error ("catalog_handle_register_store_request: invalid "
			   "alignments"));
    }
    if ((msgParm = get_message_part (catKeyTable, mp, 
	    MSG_TOK_CATALOG_DATA_SIZE)) == NULL) {
        catalog_send_error_response ("unable to extract data size",
				     MSG_TOK_CATALOG_REGISTER_STORE_REPLY,
				     txbuf, fromAddr, catalog);
	return (rtx_error ("catalog_handle_register_store_request: unable to "
			   "get data sizes"));
    }
    if (sscanf (msgParm, "%d %d %d %d %d %d", &(storeInfo.dataSize[0]),
		&(storeInfo.dataSize[1]), &(storeInfo.dataSize[2]),
		&(storeInfo.dataSize[3]), &(storeInfo.dataSize[4]),
		&(storeInfo.dataSize[5])) != 6) {
        catalog_send_error_response ("invalid dataSizes",
				     MSG_TOK_CATALOG_REGISTER_STORE_REPLY,
				     txbuf, fromAddr, catalog);
	return (rtx_error ("catalog_handle_register_store_request: invalid data "
			   "sizes"));
    }
    if ((msgParm = get_message_part (catKeyTable, mp, 
	    MSG_TOK_CATALOG_BYTE_ORDER)) == NULL) {
        catalog_send_error_response ("unable to extract byte order",
				     MSG_TOK_CATALOG_REGISTER_STORE_REPLY,
				     txbuf, fromAddr, catalog);
	return (rtx_error ("catalog_handle_register_store_request: unable to "
			   "get byte order"));
    }
    if (sscanf (msgParm, "%d %d %d %d", &(storeInfo.byteOrder[0]),
		&(storeInfo.byteOrder[1]), &(storeInfo.byteOrder[2]),
		&(storeInfo.byteOrder[3])) != 4) {
        catalog_send_error_response ("invalid byte orders",
				     MSG_TOK_CATALOG_REGISTER_STORE_REPLY,
				     txbuf, fromAddr, catalog);
	return (rtx_error ("catalog_handle_register_store_request: invalid byte "
			   "orders"));
    }
    if ((storeItemP = calloc (1, sizeof (STORE_LIST_ITEM))) == NULL) {
        catalog_send_error_response ("unable to allocate memory",
				     MSG_TOK_CATALOG_REGISTER_STORE_REPLY,
				     txbuf, fromAddr, catalog);
	return (rtx_error_errno ("catalog_handle_register_store_request: calloc"
				 "() failed"));
    }
    if ((storeInfo.storeHostName = strdup (storeHostName)) == NULL) {
        catalog_send_error_response ("unable to allocate memory",
				     MSG_TOK_CATALOG_REGISTER_STORE_REPLY,
				     txbuf, fromAddr, catalog);
	free (storeItemP);
	return (rtx_error_errno ("catalog_handle_register_store_request: strdup"
				 "() failed"));
    }
    * storeItemP = storeInfo;
    storeItemP->hostaddr = * fromAddr;
    /* now, make sure that the store has not already been registered */
    catalog_debug (("catalog_handle_register_store_request: checking"));
    if (rtx_list_lookup (catalog->store, 
			 storeItemP->storeHostName) != NULL) {
        /* store name has been registered before */
        catalog_send_error_response ("store already registered",
				     MSG_TOK_CATALOG_REGISTER_STORE_REPLY,
				     txbuf, fromAddr, catalog);
	free (storeItemP);
	return (rtx_error ("catalog_handle_register_store_request: store already "
			   "registered"));;
    }
    rtx_list_add (catalog->store, storeItemP->storeHostName,
		  storeItemP);
    if (catalog->verbose)
        rtx_list_print (catalog->store);
    catalog_debug (("catalog_handle_register_store_request: done"));
    sprintf (txbuf, "%s %s = %d\n",
	     catKeyTable[MSG_TOK_CATALOG_REGISTER_STORE_REPLY].name,
	     catKeyTable[MSG_TOK_CATALOG_ID].name, 0);
    if ((numChars = rtx_inet_write (fromAddr, txbuf, strlen (txbuf), 
				    NULL)) == -1) {
        return (rtx_error ("catalog_handler_register_var_request: send_buffer_tcp() "
			   "failed"));
    }
    return (0);
}

int
catalog_deregister_store (const char * storeHostName,
					 CATALOG * catalog)
{
	STORE_LIST_ITEM * storeItemP;
	VAR_LIST_ITEM * varItemP;
	int activeClients = 0;

	catalog_debug (("catalog_deregister_store: handling"));
	if (catalog->verbose)
		rtx_list_print (catalog->store);
	catalog_debug (("catalog_deregister_store: store = %s",
				storeHostName));
	while ((varItemP = (VAR_LIST_ITEM *) rtx_list_iterate (catalog->v))
			!= NULL) {
		if ((storeItemP = (STORE_LIST_ITEM *) rtx_list_lookup
					(varItemP->clients, storeHostName)) != NULL) {
			activeClients++;
			varItemP->activeStores--;
			rtx_list_del (varItemP->clients, storeHostName, 0);
		}
	}
	if (activeClients)
		rtx_message_warning ("catalog: clients active on %s", storeHostName);
	catalog_debug (("catalog_deregister_store: looking for store"));

	if ((storeItemP = (STORE_LIST_ITEM *) rtx_list_lookup
				(catalog->store, storeHostName)) == NULL) {
		return (-1);
	}
	catalog_debug (("catalog_deregister_store: found store"));

	rtx_list_del (catalog->store, storeHostName, 0);
	free(storeItemP->storeHostName);
	free(storeItemP);

	if (catalog->verbose)
		rtx_list_print (catalog->store);
	catalog_debug (("catalog_deregister_store: done"));
	return (0);
}

int
catalog_handle_deregister_store_request (
					 MSG * mp, 
					 char * rxbuf, 
					 char * txbuf,
					 RtxInetConn * fromAddr,
					 CATALOG * catalog
					 )
{
	int numChars = 0;
	char * storeHostName;

	catalog_debug (("catalog_handle_deregister_store_request: handling"));
	if (catalog->verbose)
		rtx_list_print (catalog->store);
	if ((storeHostName = get_message_part (catKeyTable, mp, 
					MSG_TOK_CATALOG_HOST_NAME)) == NULL) {
		catalog_send_error_response ("unable to extract hostname",
				MSG_TOK_CATALOG_DEREGISTER_STORE_REPLY,
				txbuf, fromAddr, catalog);
		return (-1);
	}
	if (catalog_deregister_store(storeHostName,catalog)<0) {
		catalog_send_error_response ("store name not registered",
				MSG_TOK_CATALOG_DEREGISTER_STORE_REPLY,
				txbuf, fromAddr, catalog);
		return (-1);
	}
	sprintf (txbuf, "%s\n",
			catKeyTable[MSG_TOK_CATALOG_DEREGISTER_STORE_REPLY].name);
	if ((numChars = rtx_inet_write (fromAddr, txbuf, strlen (txbuf),
					NULL)) == -1) {
		return (rtx_error ("catalog_handler_deregister_store_request: send_"
					"buffer() failed"));
	}
	return (0);
}

VAR_LIST_ITEM *
catalog_add_var (
		 char * varName,
		 char * varDecl,
		 char * hostName,
		 char * txbuf,
		 RtxInetConn * fromAddr,
		 CATALOG * catalog
		 )
{
	VAR_LIST_ITEM * varItemP;

	catalog_debug (("catalog_add_var: adding name '%s' decl '%s'",varName,varDecl));
	if (varDecl == NULL) {
		catalog_send_error_response ("var decl not provided",
				MSG_TOK_CATALOG_LOOKUP_VAR_REPLY,
				txbuf, fromAddr, catalog);
		return (rtx_error_errno_null ("catalog_add_var (%s): missing var decl", varName));
	}
	if ((varItemP = calloc (1, sizeof (VAR_LIST_ITEM))) == NULL) {
		catalog_send_error_response ("unable to allocate memory",
				MSG_TOK_CATALOG_LOOKUP_VAR_REPLY,
				txbuf, fromAddr, catalog);
		return (rtx_error_errno_null ("catalog_add_var (%s): calloc() failed", varName));
	}
	if ((varItemP->clients = rtx_list_init()) == NULL) {
		catalog_send_error_response ("unable to allocate memory",
				MSG_TOK_CATALOG_LOOKUP_VAR_REPLY,
				txbuf, fromAddr, catalog);
		free (varItemP);
		return (rtx_error_null ("catalog_add_var (%s): rtx_list_init() failed", varName));
	}
	if ((varItemP->name = strdup (varName)) == NULL) {
		catalog_send_error_response ("unable to allocate memory",
				MSG_TOK_CATALOG_LOOKUP_VAR_REPLY,
				txbuf, fromAddr, catalog);
		free (varItemP);
		return (rtx_error_null ("catalog_add_var (%s): strdup() failed", varName));
	}
	if ((varItemP->decl = strdup (varDecl)) == NULL) {
		catalog_send_error_response ("unable to allocate memory",
				MSG_TOK_CATALOG_LOOKUP_VAR_REPLY,
				txbuf, fromAddr, catalog);
		free (varItemP->name);
		free (varItemP);
		return (rtx_error_null ("catalog_add_var (%s): strdup() failed", varName));
	}
	varItemP->multiPort = catalog->curMultiPort++;
	varItemP->activeStores = 0;
	rtx_list_add (catalog->v, varItemP->name, (void *) varItemP);
	return (varItemP);
}

int
catalog_handle_lookup_var_request (
				   MSG * mp,
				   char * rxbuf,
				   char * txbuf,
				   RtxInetConn * fromAddr,
				   CATALOG * catalog
				   )
{
	int numChars;
	VAR_LIST_ITEM * varItemP;
	char * varName, * varDecl, *hostName;
	STORE_LIST_ITEM * storeItemP, * stp;
	STORE_LIST_ITEM * firstStore = NULL;

	catalog_debug (("catalog_handle_lookup_var_request: handling"));
	if ((varName = get_message_part (catKeyTable, mp, 
					MSG_TOK_CATALOG_VAR_NAME)) == NULL) {
		catalog_send_error_response ("unable to extract var name",
				MSG_TOK_CATALOG_LOOKUP_VAR_REPLY,
				txbuf, fromAddr, catalog);
		return (rtx_error ("catalog_handle_lookup_var_request: "
					"get_message_part() failed"));
	}
	varDecl = get_message_part (catKeyTable, mp, MSG_TOK_CATALOG_VAR_DECL);
	if ((hostName = get_message_part (catKeyTable, mp, 
					MSG_TOK_CATALOG_HOST_NAME)) == NULL) {
		catalog_send_error_response ("unable to extract host name",
				MSG_TOK_CATALOG_LOOKUP_VAR_REPLY,
				txbuf, fromAddr, catalog);
		return (rtx_error ("catalog_handle_lookup_var_request (%s): "
					"unable to get host name", varName));
	}
	if ((storeItemP = (STORE_LIST_ITEM *) rtx_list_lookup
				(catalog->store, hostName)) == NULL) {
		catalog_send_error_response ("store name not registered",
				MSG_TOK_CATALOG_LOOKUP_VAR_REPLY,
				txbuf, fromAddr, catalog);
		return (rtx_error ("catalog_handle_lookup_var_request (%s):"
					" store (%s) "
					"not registered", varName, hostName));
	}
	catalog_debug (("catalog_handle_lookup_var_request: looking for var"));
	if (rtx_mutex_lock (catalog->mutex) == -1)
		rtx_error_flush ("catalog_handle_lookup_var_request: rtx_mutex_lock");
	if ((varItemP = (VAR_LIST_ITEM *) rtx_list_lookup 
				(catalog->v, varName)) == NULL) {
		/* var is not in the catalog, put it in */
		if ((varItemP = catalog_add_var (varName, varDecl, hostName, txbuf, fromAddr, catalog)) == NULL) {
			/* error response already sent by catalog_add_var() */
			if (rtx_mutex_unlock (catalog->mutex) == -1)
				rtx_error_flush ("catalog_handle_lookup_var_request: rtx_mutex_unlock");
			rtx_error_flush("Failure detected so far\n");
			return (rtx_error ("catalog_handle_lookup_var_request (%s): "
						"catalog_add_var() failed", varName));
		}
		catalog_debug (("catalog_handle_lookup_var_request: var added"));
	} else {
		catalog_debug (("catalog_handle_lookup_var_request: var found"));
	}
	/* Now, find the store and increase the client count */
	if ((stp = (STORE_LIST_ITEM *) rtx_list_lookup 
				(varItemP->clients, hostName)) == NULL) {
		/* its a new store, find it in the store list and add a node */
		if (varItemP->activeStores == 1)
			firstStore = varItemP->clients->head->d;
		varItemP->activeStores++;
		rtx_list_add (varItemP->clients, hostName, 
				(void *) storeItemP);
		catalog_debug (("catalog_handle_lookup_var_request: added store %s to client list [%d]", hostName, varItemP->activeStores));
	}
	if (rtx_mutex_unlock (catalog->mutex) == -1)
		rtx_error_flush ("catalog_handle_lookup_var_request: rtx_mutex_unlock");
	if (catalog->verbose)
		rtx_list_print (varItemP->clients);
	if (varItemP->activeStores > 1) {
		sprintf (txbuf, "%s %s = %s %s = %s %s = %s %s = %d %s\n",
				catKeyTable[MSG_TOK_CATALOG_LOOKUP_VAR_REPLY].name,
				catKeyTable[MSG_TOK_CATALOG_VAR_NAME].name, varName,
				catKeyTable[MSG_TOK_CATALOG_VAR_DECL].name, varItemP->decl,
				catKeyTable[MSG_TOK_CATALOG_MULTICAST_ADDR].name,
				catalog->multiAddr,
				catKeyTable[MSG_TOK_CATALOG_MULTICAST_PORT].name,
				varItemP->multiPort,
				catKeyTable[MSG_TOK_CATALOG_MULTICAST_ON].name);
	} else {
		sprintf (txbuf, "%s %s = %s %s = %s %s = %s %s = %d\n",
				catKeyTable[MSG_TOK_CATALOG_LOOKUP_VAR_REPLY].name,
				catKeyTable[MSG_TOK_CATALOG_VAR_NAME].name, varName,
				catKeyTable[MSG_TOK_CATALOG_VAR_DECL].name, varItemP->decl,
				catKeyTable[MSG_TOK_CATALOG_MULTICAST_ADDR].name,
				catalog->multiAddr,
				catKeyTable[MSG_TOK_CATALOG_MULTICAST_PORT].name,
				varItemP->multiPort);
	}
	catalog_debug (("catalog_handle_lookup_var_request: sending response "
				"(%d chars) = %s", strlen (txbuf), txbuf));
	if ((numChars = rtx_inet_write (fromAddr, txbuf, strlen (txbuf),
					NULL)) == -1) {
		return (rtx_error ("catalog_handler_lookup_var_request (%s): rtx_inet_write() "
					"failed", varName));
	}
	catalog_debug (("catalog_handle_lookup_var_request: sent %d chars", numChars));
	/* if multicast was turned on with this request, inform the first store */
	if (firstStore != NULL) {
		catalog_debug (("catalog_handle_lookup_var_request: sending start_multicast "
					"request to store@%s:%d", 
					rtx_inet_print_net_addr (&(firstStore->hostaddr.remote)),
					rtx_inet_print_net_portnum (&(firstStore->hostaddr.remote))));
		sprintf (txbuf, "%s %s = %s\n",
				catKeyTable[MSG_TOK_CATALOG_START_MULTICAST_CMD_TO_STORE].name,
				catKeyTable[MSG_TOK_CATALOG_VAR_NAME].name, varName);
		if ((numChars = rtx_inet_write (&firstStore->hostaddr, txbuf, 
						strlen (txbuf), NULL))
				== -1) {
			return (rtx_error ("catalog_handler_lookup_var_request (%s): rtx_inet_write() "
						"failed", varName));
		}	
	}
	return (0);
}

int
catalog_handle_done_var_request (
				 MSG * mp,
				 char * rxbuf,
				 char * txbuf,
				 RtxInetConn * fromAddr,
				 CATALOG * catalog
				 )
{
    int numChars;
    VAR_LIST_ITEM * varItemP;
    char * varName, * hostName;
    STORE_LIST_ITEM * storeItemP;
    int stopMulticastMsg = 0;

    catalog_debug (("catalog_handle_done_var_request: handling"));
    if ((varName = get_message_part (catKeyTable, mp, 
				     MSG_TOK_CATALOG_VAR_NAME)) == NULL) {
        catalog_send_error_response ("unable to extract var name",
				     MSG_TOK_CATALOG_DONE_VAR_REPLY,
				     txbuf, fromAddr, catalog);
	return (rtx_error ("catalog_handle_done_var_request: "
			   "unable to get var name"));
    }
    if ((hostName = get_message_part (catKeyTable, mp, 
	    MSG_TOK_CATALOG_HOST_NAME)) == NULL) {
        catalog_send_error_response ("unable to extract host name",
				     MSG_TOK_CATALOG_DONE_VAR_REPLY,
				     txbuf, fromAddr, catalog);
	return (rtx_error ("catalog_handle_done_var_request (%s): "
			   "unable to get host name", varName));
    }
    catalog_debug (("catalog_handle_done_var_request (%s): looking for var", varName));
    if ((varItemP = (VAR_LIST_ITEM *) rtx_list_lookup 
	 (catalog->v, varName)) == NULL) {
        catalog_send_error_response ("var name not registered",
				     MSG_TOK_CATALOG_DONE_VAR_REPLY,
				     txbuf, fromAddr, catalog);
	return (rtx_error ("catalog_handle_done_var_request (%s): "
			   "var name not registered", varName));
    }
    catalog_debug (("catalog_handle_done_var_request (%s): found", varName));
    if (catalog->verbose)
        rtx_list_print (varItemP->clients);
    catalog_debug (("catalog_handle_done_var_request (%s): looking for store %s", varName, hostName));
    if (rtx_mutex_lock (catalog->mutex) == -1)
        rtx_error_flush ("catalog_handle_done_var_request: rtx_mutex_unlock");
    if ((storeItemP = (STORE_LIST_ITEM *) rtx_list_lookup
	 (varItemP->clients, hostName)) != NULL) {
        varItemP->activeStores--;
	if (varItemP->activeStores == 1)
	    stopMulticastMsg = 1;
	rtx_list_del (varItemP->clients, hostName, 0);
	catalog_debug (("catalog_handle_done_var_request (%s): found store %s [%d]", varName, hostName, varItemP->activeStores));
    }
    if (rtx_mutex_unlock (catalog->mutex) == -1)
        rtx_error_flush ("catalog_handle_done_var_request: rtx_mutex_unlock");
    if (stopMulticastMsg) {
        sprintf (txbuf, "%s %s\n",
		 catKeyTable[MSG_TOK_CATALOG_DONE_VAR_REPLY].name,
		 catKeyTable[MSG_TOK_CATALOG_MULTICAST_OFF].name);
    } else {
        sprintf (txbuf, "%s\n",
		 catKeyTable[MSG_TOK_CATALOG_DONE_VAR_REPLY].name);
    }
    catalog_debug (("catalog_handle_done_var_request (%s): sending reply", varName));
    if ((numChars = rtx_inet_write (fromAddr, txbuf, strlen (txbuf),
				    NULL)) == -1) {
        return (rtx_error ("catalog_handler_register_var_request (%s): "
			   "rtx_inet_write() failed", varName));
    }
    catalog_debug (("catalog_handle_done_var_request (%s): checking multicast", varName));
    /* if active stores is == 1, then tell it to stop multicasting */
    if (stopMulticastMsg) {
        storeItemP = (STORE_LIST_ITEM *) varItemP->clients->head->d;
        catalog_debug (("catalog_handle_done_var_request: sending stop_multicast"
			"request to store@%s:%d", 
			rtx_inet_print_net_addr (&(storeItemP->hostaddr.remote)),
			rtx_inet_print_net_portnum (&(storeItemP->hostaddr.remote))));
        sprintf (txbuf, "%s %s = %s\n",
		 catKeyTable[MSG_TOK_CATALOG_STOP_MULTICAST_CMD_TO_STORE].name,
		 catKeyTable[MSG_TOK_CATALOG_VAR_NAME].name, varName);
	if ((numChars = rtx_inet_write (&storeItemP->hostaddr, txbuf, 
					strlen (txbuf), NULL)) == -1) {
	    return (rtx_error ("catalog_handler_register_var_request "
			       "(%s): rtx_inet_write() failed",
			       varName));
	}
    }
    catalog_debug (("catalog_handle_done_var_request (%s): done", varName));
    return (0);
}

int
catalog_handle_get_store_list_request (
				       MSG * mp, 
				       char * rxbuf, 
				       char * txbuf,
				       RtxInetConn * fromAddr,
				       CATALOG * catalog
				       )
{
    int numChars;
    STORE_LIST_ITEM * stp;

    catalog_debug (("catalog_handle_get_store_list_request: handling"));
    
    if (catalog->store != NULL)
        sprintf (txbuf, "%s %s = ",
		 catKeyTable[MSG_TOK_CATALOG_GET_STORE_LIST_REPLY].name,
		 catKeyTable[MSG_TOK_CATALOG_STORE_LIST].name);
    else
        sprintf (txbuf, "%s",
		 catKeyTable[MSG_TOK_CATALOG_GET_STORE_LIST_REPLY].name);

    while ((stp = (STORE_LIST_ITEM *) rtx_list_iterate
	    (catalog->store)) != NULL) {
        strcat (txbuf, stp->storeHostName);
	strcat (txbuf, " ");
    }
    strcat (txbuf, "\n");
    if ((numChars = rtx_inet_write (fromAddr, txbuf, strlen (txbuf),
				    NULL)) == -1) {
        return (rtx_error ("catalog_handler_get_store_list_request: rtx_inet_write() "
			   "failed"));
    }
    return (0);
}

int
catalog_handle_get_item_list_request (
				      MSG * mp, 
				      char * rxbuf, 
				      char * txbuf,
				      RtxInetConn * fromAddr,
				      CATALOG * catalog
				      )
{
    int numChars;
    VAR_LIST_ITEM * vlp;

    catalog_debug (("catalog_handle_get_item_list_request: handling"));
    
    if (catalog->v != NULL)
        sprintf (txbuf, "%s %s = ",
		 catKeyTable[MSG_TOK_CATALOG_GET_ITEM_LIST_REPLY].name,
		 catKeyTable[MSG_TOK_CATALOG_STORE_LIST].name);
    else
        sprintf (txbuf, "%s",
		 catKeyTable[MSG_TOK_CATALOG_GET_ITEM_LIST_REPLY].name);

    while ((vlp = (VAR_LIST_ITEM *) rtx_list_iterate
	    (catalog->v)) != NULL) {
        strcat (txbuf, vlp->name);
	strcat (txbuf, " ");
    }
    strcat (txbuf, "\n");
    if ((numChars = rtx_inet_write (fromAddr, txbuf, strlen (txbuf),
				    NULL)) == -1) {
        return (rtx_error ("catalog_handler_get_item_list_request: rtx_inet_write() "
			   "failed"));
    }
    return (0);
}

int
catalog_handle_get_store_info_request (
				       MSG * mp, 
				       char * rxbuf, 
				       char * txbuf,
				       RtxInetConn * fromAddr,
				       CATALOG * catalog
				       )
{
    int numChars;
    STORE_LIST_ITEM * storeItemP;
    char * storeHostName;

    catalog_debug (("catalog_handle_get_store_info_request: handling"));
    if ((storeHostName = get_message_part (catKeyTable, mp, 
	    MSG_TOK_CATALOG_HOST_NAME)) == NULL) {
        catalog_send_error_response ("unable to extract hostname",
				     MSG_TOK_CATALOG_GET_STORE_INFO_REPLY,
				     txbuf, fromAddr, catalog);
	return (rtx_error ("catalog_handle_get_store_info_request: "
			   "unable to get host name"));
    }
    catalog_debug (("catalog_handle_get_store_info_request: looking for "
		    "store on host %s", storeHostName));
    if ((storeItemP = (STORE_LIST_ITEM *) rtx_list_lookup
	 (catalog->store, storeHostName)) == NULL) {
        catalog_send_error_response ("store name not registered",
				     MSG_TOK_CATALOG_GET_STORE_INFO_REPLY,
				     txbuf, fromAddr, catalog);
	return (rtx_error ("catalog_handle_get_store_info_request (%s): "
				   "unable to find store", storeHostName));
    }
    sprintf (txbuf, "%s %s = %s %s = %d %s = %d %d %d %d %d %d "
	     "%s = %d %d %d %d %d %d %s = %d %d %d %d %s = %s\n",
	     catKeyTable[MSG_TOK_CATALOG_GET_STORE_INFO_REPLY].name,
	     catKeyTable[MSG_TOK_CATALOG_HOST_NAME].name,
	     storeItemP->storeHostName,
	     catKeyTable[MSG_TOK_CATALOG_PORT_NUM].name,
	     storeItemP->portNum,
	     catKeyTable[MSG_TOK_CATALOG_ALIGNMENT].name,
	     storeItemP->alignment[0], storeItemP->alignment[1],
	     storeItemP->alignment[2], storeItemP->alignment[3],
	     storeItemP->alignment[4], storeItemP->alignment[5], 
	     catKeyTable[MSG_TOK_CATALOG_DATA_SIZE].name,
	     storeItemP->dataSize[0], storeItemP->dataSize[1],
	     storeItemP->dataSize[2], storeItemP->dataSize[3],
	     storeItemP->dataSize[4], storeItemP->dataSize[5], 
	     catKeyTable[MSG_TOK_CATALOG_BYTE_ORDER].name,
	     storeItemP->byteOrder[0], storeItemP->byteOrder[1],
	     storeItemP->byteOrder[2], storeItemP->byteOrder[3],
	     catKeyTable[MSG_TOK_CATALOG_MULTICAST_ADDR].name,
	     catalog->multiAddr);
    if ((numChars = rtx_inet_write (fromAddr, txbuf, strlen (txbuf),
				    NULL)) == -1) {
        return (rtx_error ("catalog_handler_get_store_info_request (%s):"
			   " rtx_inet_write() failed", storeHostName));
    }
    return (0);
}

