#ifndef DDX_MARLINS_TOOLS_H
#define DDX_MARLINS_TOOLS_H

#include <libraw1394/raw1394.h>

/**
  Loads a shading image into the AVT Marlin camera.
  See documentation for details.
  \arg handle : raw1394 handle
  \arg node : device id
  \arg filename : name of a PGM file containing the shading image
**/
extern int shading_load_image(raw1394handle_t handle, nodeid_t node, char * filename);

/**
  Shows the shading image as the camera image (untested)
  \arg handle : raw1394 handle
  \arg node : device id
  \arg on : boolean, activates/desactivates this functionality
**/
extern int shading_show_image(raw1394handle_t handle, nodeid_t node, int on);

/**
  Activate the shading process (untested)
  \arg handle : raw1394 handle
  \arg node : device id
  \arg on : boolean, activates/desactivates this functionality
**/
extern int shading_activate(raw1394handle_t handle, nodeid_t node,int on);

/**
  Asks the camera to build its own shading image. Image should be static and 
  not completely white.
  \arg handle : raw1394 handle
  \arg node : device id
  \arg nb_images : number of images used to build the image
  \arg wait_max : time out on the build process, in seconds
**/
extern int shading_build_image(raw1394handle_t handle, nodeid_t node, 
				unsigned int nb_images, unsigned int wait_max);

/**
  \struct LUT_Table
  Struture used to represent a LUT table
**/
typedef struct {
		unsigned char id;
		unsigned char lut[1024];
} LUT_Table;

/**
  Loads a LUT table into the AVT Marlin camera.
  See documentation for details.
  \arg handle : raw1394 handle
  \arg node : device id
  \arg filename : name of a file containing the lut data. Format is 
  given below. Data types conforms to the LUT_Table struct definition.
  \sa LUT_Table
  <pre>
  id
  lut(0)
  ...
  lut(1024)
  </pre>
**/
extern int LUT_load_from_file(raw1394handle_t handle, nodeid_t node, char * filename);

/**
  Loads a LUT table into the AVT Marlin camera.
  See documentation for details.
  \arg handle : raw1394 handle
  \arg node : device id
  \arg lut_table : pointer on a LUT_Table
  		representing the lut function (see doc)
**/
extern int LUT_load_table(raw1394handle_t handle, nodeid_t node, LUT_Table * table);

/**
  Activate the LUT process using LUT id
  \arg handle : raw1394 handle
  \arg node : device id
  \arg on : boolean, activates/desactivates this functionality
  \arg lutid : LUT table to use. Should have been loaded previously
  		This arg is not used if on == 0
**/
extern int LUT_activate(raw1394handle_t handle, nodeid_t node,int on, unsigned char lutid);

/**
  Query the number of internal LUT tables
  \arg handle : raw1394 handle
  \arg node : device id
**/
extern unsigned int LUT_query_num_of_luts(raw1394handle_t handle, nodeid_t node);

#endif // DDX_MARLINS_TOOLS_H
