
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <errno.h>

extern "C" {
#include <pip.h>
}

#include "ddxmarlintools.h"

#define BASE_ADDR 0xFFFF00000000ULL
#define SHDG_CTRL     0xF1000250ULL
#define SHDG_MEM_CTRL 0xF1000254ULL
#define LUT_CTRL      0xF1000240ULL
#define LUT_MEM_CTRL  0xF1000244ULL
#define LUT_INFO      0xF1000248ULL
#define GPDATA_INFO   0xF1000FFCULL
#define GPDATA_BUFFER 0xF1001000ULL

static inline int is_big_endian() {
	unsigned char uc[2] = {0xAB,0xCD};
	unsigned short * us = (unsigned short*)uc;
	return *us != 0xABCD;
}



#define ACKDISPLAY(ackcode,rcode) fprintf(stderr, "Ack code: 0x%0x, Response code: 0x%0x\n",(ackcode),(rcode));
#define MAXTRIES 20
#define DELAY 10000

//#define DEBUG

/** ****************************
  Reads a buffer from a 1394 device, 
  retries if needed and manage errors
  \arg handle : raw1394 handle
  \arg node : device id
  \arg addr : where to read
  \arg length : number of quadlets to read
  \arg buffer : data destination
 **/
static int cooked1394_read(raw1394handle_t handle, 
		nodeid_t node, nodeaddr_t addr,
		size_t length, quadlet_t *buffer) 
{
	int retval;
	unsigned int i;
#ifdef DEBUG
	printf("Read %d quads from addr %012llX to %p\n",length,addr,buffer);
#endif
	for(i=0; i<MAXTRIES; i++) {
		retval = raw1394_read(handle, 0xffc0 | node, addr, length, buffer);
		if( retval >= 0 ) {
			if (is_big_endian()) {
				for (i=0;i<length/4;i++) {
					buffer[i] = ((buffer[i] & 0x000000FF) << 24) 
						| ((buffer[i] & 0x0000FF00) << 8) 
						| ((buffer[i] & 0x00FF0000) >> 8) 
						| ((buffer[i] & 0xFF000000) >> 24) ;
#ifdef DEBUG
					if (i<6) printf("%08X ",buffer[i]);
#endif
				}
#ifdef DEBUG
				printf("\n");
#endif
			}
			return 1;	/* Everything went fine */
		}

		/* Error management */
		ACKDISPLAY( raw1394_get_ack(raw1394_get_errcode(handle)),
				raw1394_get_rcode(raw1394_get_errcode(handle)) );
		if( errno != EAGAIN ) break;
		usleep(DELAY);
	}
	perror("Could not read from raw1394:");
	return 0;
}

/** ****************************
  Writes a buffer into a 1394 device, 
  retries if needed and manage errors
  \arg handle : raw1394 handle
  \arg node : device id
  \arg addr : where to write
  \arg length : number of quadlets to write
  \arg buffer : data source
 **/
static int cooked1394_write(raw1394handle_t handle, nodeid_t node, nodeaddr_t addr,
		size_t length, quadlet_t *data) 
{
	int retval;
	unsigned int i;
	quadlet_t buffer[1024]; 
#ifdef DEBUG
	printf("Write %d quads to addr %012llX from %p\n",length,addr,data);
#endif
	if (is_big_endian()) {
		for (i=0;i<length/4;i++) {
			buffer[i] = ((data[i] & 0x000000FF) << 24) 
				| ((data[i] & 0x0000FF00) << 8) 
				| ((data[i] & 0x00FF0000) >> 8) 
				| ((data[i] & 0xFF000000) >> 24) ;
#ifdef DEBUG
			if (i<6) printf("%08X ",data[i]);
#endif
		}
#ifdef DEBUG
		printf("\n");
#endif
	}
	for(i=0; i<MAXTRIES; i++) {
		retval = raw1394_write(handle, 0xffc0 | node, addr, length, buffer);
		if( retval >= 0 ) {
			return 1;	/* Everything went fine */
		}

		/* Error management */
		ACKDISPLAY( raw1394_get_ack(raw1394_get_errcode(handle)),
				raw1394_get_rcode(raw1394_get_errcode(handle)) );
		if( errno != EAGAIN ) break;
		usleep(DELAY);
	}
	perror("Could not write to raw1394:");
	return 0;
}

#define MIN(a,b) (((a)<(b))?(a):(b))

								 /**
								   Loads a shading image into the AVT Marlin camera.
								   See documentation for details.
								   \arg handle : raw1394 handle
								   \arg node : device id
								   \arg filename : name of a PGM file containing the shading image
								  **/
								 int shading_load_image(raw1394handle_t handle, nodeid_t node, char * filename)
{
	quadlet_t shdg_ctrl[3];
	quadlet_t gpdata_info[1];
	unsigned int loaded;
	unsigned char * img_ptr;
	unsigned int max_image_size;
	unsigned int gpdata_buffer_size;
	Pimage * image;
	image = PIP_new();
	if (PIP_load(image,filename) != PIPOK) {
		fprintf(stderr,"Shading: Cannot load shading image\n");
		return 0;
	}

	if (!cooked1394_read(handle,node,BASE_ADDR + SHDG_CTRL,12,shdg_ctrl)) {
		fprintf(stderr,"Shading: Cannot read SHDG_CTRL register\n");
		return 0;
	}
	if (!(shdg_ctrl[0] & 0x80000000) || !(shdg_ctrl[1] & 0x80000000) 
			|| !(shdg_ctrl[2] & 0x80000000)) {
		fprintf(stderr,"Shading: Shading functionnality not available (%08X %08X %08X)\n",
				shdg_ctrl[0],shdg_ctrl[1],shdg_ctrl[2]);
		return 0;
	}

	max_image_size = shdg_ctrl[2] & 0x00FFFFFF;
	printf("Shading: Max image size : %db\n",max_image_size);
	if (image->np > (signed)max_image_size) {
		fprintf(stderr,"Shading: Loaded image to big : %db > %db\n",
				image->np,max_image_size);
		return 0;
	}

	if (!cooked1394_read(handle,node,BASE_ADDR + GPDATA_INFO,4,gpdata_info)) {
		fprintf(stderr,"Shading: Cannot read GPDATA_INFO register\n");
		return 0;
	}
	gpdata_buffer_size = gpdata_info[0] & 0x0000FFFF;
	printf("Shading: Buffer size : %db\n",gpdata_buffer_size);

	shdg_ctrl[1] |= 0x04000000; // EnableMemWR = 1, AddrOffset = 0	
	if (!cooked1394_write(handle,node,BASE_ADDR + SHDG_MEM_CTRL,4,shdg_ctrl+1)) {
		fprintf(stderr,"Shading: Cannot enable MemWR\n");
		return 0;
	}
	printf("Shading: Loading image:\n");
	loaded = 0;
	img_ptr = image->data;
	while (1) {
		if (!cooked1394_write(handle,node,BASE_ADDR + GPDATA_BUFFER,
					MIN(image->np-loaded,gpdata_buffer_size),
					(quadlet_t*)img_ptr)) {
			fprintf(stderr,"Shading: Cannot write image data\n");
			return 0;
		}
		loaded += MIN(image->np-loaded,gpdata_buffer_size);
		img_ptr = image->data + loaded;
		printf("Loaded %8d/%8d           \r",loaded,image->np);
		if ((signed)loaded == image->np)  break;
		// EnableMemWR = 1, AddrOffset = loaded
		shdg_ctrl[1] = (shdg_ctrl[1] & 0xFF000000) | 0x04000000 | loaded; 
		if (!cooked1394_write(handle,node,BASE_ADDR + SHDG_MEM_CTRL,4,shdg_ctrl+1)) {
			fprintf(stderr,"\nShading: Cannot enable MemWR and set new offset\n");
			return 0;
		}
	}
	printf("\n");
	// EnableMemWR = 0, AddrOffset = 0
	shdg_ctrl[0] = shdg_ctrl[0] & 0xF5000000;
	shdg_ctrl[1] = shdg_ctrl[1] & 0xFD000000;
	if (!cooked1394_write(handle,node,BASE_ADDR + SHDG_CTRL,8,shdg_ctrl)) {
		fprintf(stderr,"Shading: Cannot enable MemWR and set new offset\n");
		return 0;
	}

	printf("Loaded Shading Image\n");
	PIP_free(image);
	return 1;
}

/**
  Shows the shading image as the camera image (untested)
  \arg handle : raw1394 handle
  \arg node : device id
  \arg on : boolean, activates/desactivates this functionality
 **/
int shading_show_image(raw1394handle_t handle, nodeid_t node, int on)
{
	quadlet_t shdg_ctrl[1];
	if (!cooked1394_read(handle,node,BASE_ADDR + SHDG_CTRL,4,shdg_ctrl)) {
		fprintf(stderr,"Shading: Cannot read SHDG_CTRL register\n");
		return 0;
	}
	if (!(shdg_ctrl[0] & 0x80000000)) {
		fprintf(stderr,"Shading: Shading functionnality not available (%08X)\n",
				shdg_ctrl[0]);
		return 0;
	}
	printf("Before show : %08X\n",shdg_ctrl[0]);
	if (on) {
		shdg_ctrl[0] |= 0x0A000000;
	} else {
		shdg_ctrl[0] &= 0xF5FFFFFF;
	}
	if (!cooked1394_write(handle,node,BASE_ADDR + SHDG_CTRL,4,shdg_ctrl)) {
		fprintf(stderr,"Shading: Cannot write SHDG_CTRL register\n");
		return 0;
	}
	if (!cooked1394_read(handle,node,BASE_ADDR + SHDG_CTRL,4,shdg_ctrl)) {
		fprintf(stderr,"Shading: Cannot read SHDG_CTRL register\n");
		return 0;
	}
	usleep(50000);
	printf("After show : %08X\n",shdg_ctrl[0]);
	return 1;
}

/**
  Activate the shading process (untested)
  \arg handle : raw1394 handle
  \arg node : device id
  \arg on : boolean, activates/desactivates this functionality
 **/
int shading_activate(raw1394handle_t handle, nodeid_t node,int on)
{
	quadlet_t shdg_ctrl[1];
	if (!cooked1394_read(handle,node,BASE_ADDR + SHDG_CTRL,4,shdg_ctrl)) {
		fprintf(stderr,"Shading: Cannot read SHDG_CTRL register\n");
		return 0;
	}
	if (!(shdg_ctrl[0] & 0x80000000)) {
		fprintf(stderr,"Shading: Shading functionnality not available (%08X)\n",
				shdg_ctrl[0]);
		return 0;
	}
	if (on) {
		shdg_ctrl[0] |= 0x02000000;
	} else {
		shdg_ctrl[0] &= 0xFDFFFFFF;
	}
	if (!cooked1394_write(handle,node,BASE_ADDR + SHDG_CTRL,4,shdg_ctrl)) {
		fprintf(stderr,"Shading: Cannot write SHDG_CTRL register\n");
		return 0;
	}
	return 1;
}

/**
  Asks the camera to build its own shading image. Image should be static and 
  not completely white.
  \arg handle : raw1394 handle
  \arg node : device id
  \arg nb_images : number of images used to build the image
  \arg wait_max : time out on the build process, in seconds
 **/
int shading_build_image(raw1394handle_t handle, nodeid_t node, unsigned int nb_images, unsigned int wait_max)
{
	int result;
	unsigned int waited;
	quadlet_t shdg_ctrl[1];
	if (nb_images > 255) {
		printf("Shading: shading image can only be built from less than 256 grabs\n");
		nb_images = 255;
	}
	if (!cooked1394_read(handle,node,BASE_ADDR + SHDG_CTRL,4,shdg_ctrl)) {
		fprintf(stderr,"Shading: Cannot read SHDG_CTRL register\n");
		return 0;
	}
	if (!(shdg_ctrl[0] & 0x80000000)) {
		fprintf(stderr,"Shading: Shading functionnality not available (%08X)\n",
				shdg_ctrl[0]);
		return 0;
	}
	shdg_ctrl[0] = (shdg_ctrl[0] & 0xF5000000) | 0x04000000 | nb_images;
	if (!cooked1394_write(handle,node,BASE_ADDR + SHDG_CTRL,4,shdg_ctrl)) {
		fprintf(stderr,"Shading: Cannot write SHDG_CTRL register\n");
		return 0;
	}
	printf("Shading: Build request sent\n");
	waited = 0;
	while (waited < wait_max*2) {
		usleep(500000); // poll at 2Hz, according to documentation
		if (!cooked1394_read(handle,node,BASE_ADDR + SHDG_CTRL,4,shdg_ctrl)) {
			fprintf(stderr,"Shading: Cannot poll SHDG_CTRL register\n");
			return 0;
		}
		if (!(shdg_ctrl[0] & 0x05000000)) // busy = 0 && buildimage = 0
			break;
		waited += 1;
	}
	result = (waited < wait_max);
	if (result) {
		printf("Shading: shading image successfully built\n");
	} else {
		printf("Shading: time out !\n");
	}
	return result;
}


/**
  Loads a LUT table into the AVT Marlin camera and activate it.
  See documentation for details.
  \arg handle : raw1394 handle
  \arg node : device id
  \arg filename : name of a file containing the lut data. Format is 
  given below. Data types conforms to the LUT_Table struct definition.
  \sa LUT_Table
  <pre>
  id
  lut(0)
  ...
  lut(1024)
  </pre>
 **/
int LUT_load_from_file(raw1394handle_t handle, nodeid_t node, char * filename)
{
	int tmp,i;
	LUT_Table t;
	FILE * fp = fopen(filename,"r");
	if (fp == NULL) {
		fprintf(stderr,"LUT: Cannot open lut table '%s'\n",filename);
		return 0;
	}
	if (fscanf(fp," %d ", &tmp) != 1) {
		fprintf(stderr,"LUT: Cannot read lut table id in '%s'\n",filename);
		return 0;
	}
	if ((tmp < 0) && (tmp > 63)) {
		fprintf(stderr,"LUT: in '%s', lut id should be in [0,63]\n",filename);
		return 0;
	}
	t.id = tmp;
	for (i=0;i<1024;i++) {
		if (fscanf(fp," %d ", &tmp) != 1) {
			fprintf(stderr,"LUT: Cannot read lut table line %d in '%s'\n",i,filename);
			return 0;
		}
		if ((tmp < 0) && (tmp > 255)) {
			fprintf(stderr,"LUT: in '%s', lut line %d should be in [0,255]\n",filename,i);
			return 0;
		}
		t.lut[i] = tmp;
	}

	if (!LUT_load_table(handle,node,&t)) return 0;

	return LUT_activate(handle,node,1,t.id);
}

/**
  Loads a LUT table into the AVT Marlin camera.
  See documentation for details.
  \arg handle : raw1394 handle
  \arg node : device id
  \arg lut_table : pointer on a LUT_Table
  representing the lut function (see doc)
 **/
int LUT_load_table(raw1394handle_t handle, nodeid_t node, LUT_Table * lut)
{
	quadlet_t lut_ctrl[3];
	quadlet_t gpdata_info[1];
	unsigned int loaded;
	unsigned char * lut_ptr;
	unsigned int max_lut_size;
	unsigned int max_lut_nb;
	unsigned int gpdata_buffer_size;

	if (!cooked1394_read(handle,node,BASE_ADDR + LUT_CTRL,12,lut_ctrl)) {
		fprintf(stderr,"LUT: Cannot read LUT_CTRL register\n");
		return 0;
	}
	if (!(lut_ctrl[0] & 0x80000000) || !(lut_ctrl[1] & 0x80000000) 
			|| !(lut_ctrl[2] & 0x80000000)) {
		fprintf(stderr,"LUT: LUT functionnality not available (%08X %08X %08X)\n",
				lut_ctrl[0],lut_ctrl[1],lut_ctrl[2]);
		return 0;
	}

	max_lut_size = lut_ctrl[2] & 0x0000FFFF;
	if (max_lut_size != 1024) {
		fprintf(stderr,"LUT: Unexpected LUT size (%d). See documentation.\n",max_lut_size);
		return 0;
	}
	max_lut_nb = (lut_ctrl[2] & 0x00FF0000) >> 16;
	printf("LUT: Available: %d tables of %d bytes\n",max_lut_nb,max_lut_size);
	if (lut->id >= max_lut_nb) {
		fprintf(stderr,"LUT: lut id %d too big (> %d)\n",
				lut->id,max_lut_nb);
		return 0;
	}

	if (!cooked1394_read(handle,node,BASE_ADDR + GPDATA_INFO,4,gpdata_info)) {
		fprintf(stderr,"LUT: Cannot read GPDATA_INFO register\n");
		return 0;
	}
	gpdata_buffer_size = gpdata_info[0] & 0x0000FFFF;
	printf("LUT: Buffer size : %db\n",gpdata_buffer_size);

	// EnableMemWR = 1, AddrOffset = 0, AccessLutNb = lut->id	
	lut_ctrl[0] = (lut_ctrl[0] & 0xFD000000);
	lut_ctrl[1] = (lut_ctrl[1] & 0xFF000000) | 0x04000000 
		| (((unsigned long)(lut->id)) << 16);
	if (!cooked1394_write(handle,node,BASE_ADDR + LUT_CTRL,8,lut_ctrl)) {
		fprintf(stderr,"LUT: Cannot enable MemWR\n");
		return 0;
	}
	printf("LUT: Loading table:\n");
	loaded = 0;
	lut_ptr = lut->lut;
	while (1) {
		if (!cooked1394_write(handle,node,BASE_ADDR + GPDATA_BUFFER,
					MIN(max_lut_size-loaded,gpdata_buffer_size),
					(quadlet_t*)lut_ptr)) {
			fprintf(stderr,"LUT: Cannot write table\n");
			return 0;
		}
		loaded += MIN(max_lut_size-loaded,gpdata_buffer_size);
		lut_ptr = lut->lut + loaded;
		printf("Loaded %8d/%8d           \r",loaded,max_lut_size);
		if (loaded == max_lut_size)  break;
		// EnableMemWR = 1, AddrOffset = loaded
		lut_ctrl[1] = (lut_ctrl[1] & 0xFF000000) | 0x04000000 
			| (((unsigned long)(lut->id)) << 16) | loaded;
		if (!cooked1394_write(handle,node,BASE_ADDR + LUT_MEM_CTRL,4,lut_ctrl+1)) {
			fprintf(stderr,"\nLUT: Cannot enable MemWR and set new offset\n");
			return 0;
		}
	}
	printf("\n");
	// EnableMemWR = 0, AddrOffset = 0
	lut_ctrl[1] = lut_ctrl[1] & 0xFB000000;
	if (!cooked1394_write(handle,node,BASE_ADDR + LUT_MEM_CTRL,4,lut_ctrl+1)) {
		fprintf(stderr,"LUT: Cannot disable MemWR\n");
		return 0;
	}

	printf("Loaded LUT table %d\n",lut->id);
	return 1;
}

/**
  Activate the LUT process using LUT id
  \arg handle : raw1394 handle
  \arg node : device id
  \arg on : boolean, activates/desactivates this functionality
  \arg lutid : LUT table to use. Should have been loaded previously
  This arg is not used if on == 0
 **/
int LUT_activate(raw1394handle_t handle, nodeid_t node,int on, unsigned char lutid)
{
	quadlet_t lut_ctrl[1];
	if (!cooked1394_read(handle,node,BASE_ADDR + LUT_CTRL,4,lut_ctrl)) {
		fprintf(stderr,"LUT: Cannot read LUT_CTRL register\n");
		return 0;
	}
	if (!(lut_ctrl[0] & 0x80000000)) {
		fprintf(stderr,"LUT: LUT functionnality not available (%08X)\n",
				lut_ctrl[0]);
		return 0;
	}
	if (on) {
		lut_ctrl[0] = (lut_ctrl[0] & 0xFFFFFFC0) | 0x02000000
			| (lutid & 0x3F) ;
	} else {
		lut_ctrl[0] &= 0xFDFFFFFF;
	}
	if (!cooked1394_write(handle,node,BASE_ADDR + LUT_CTRL,4,lut_ctrl)) {
		fprintf(stderr,"LUT: Cannot write LUT_CTRL register\n");
		return 0;
	}
	return 1;
}


/**
  Query the number of internal LUT tables
  \arg handle : raw1394 handle
  \arg node : device id
 **/
unsigned int LUT_query_num_of_luts(raw1394handle_t handle, nodeid_t node)
{
	quadlet_t lut_info[1];
	if (!cooked1394_read(handle,node,BASE_ADDR + LUT_INFO,4,lut_info)) {
		fprintf(stderr,"LUT: Cannot read LUT_INFO register\n");
		return 0;
	}
	if (!(lut_info[0] & 0x80000000)) {
		fprintf(stderr,"LUT: LUT functionnality not available (%08X)\n",
				lut_info[0]);
		return 0;
	}
	return (lut_info[0] & 0x00FF0000) >> 16;
}
