#include <assert.h>
#include <stdlib.h>
#include <stdio.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

extern "C" {
#include <jpeglib.h>
}
#include <string.h>
#include <setjmp.h>

#include "JpegCodec.h"

JpegCodec::JpegCodec(int qual)
{
	width = height = 0;
	outputColorSpace = cmAuto;
	inputColorSpace = cmAuto;
	size = 0;
	buffer = NULL;
	conv_buffer = NULL;
	timestamp = 0;
	setQuality(qual);
}

JpegCodec::~JpegCodec()
{
	//printf("~JpegCodec\n");
	free(conv_buffer);
	conv_buffer = NULL;
	free(buffer);
	buffer = NULL;
	
}

void JpegCodec::setQuality(int qual)
{
	if (qual <= 0) qual = 75;
	if (qual > 100) qual = 100;
	quality = qual;
}

void JpegCodec::setTimeStamp()
{
	struct timeval tv;
	gettimeofday(&tv,NULL);
	timestamp = tv.tv_sec + 1e-6*tv.tv_usec;
}

void JpegCodec::setTimeStamp(double ts)
{
	timestamp = ts;
}

struct my_error_mgr {
	struct jpeg_error_mgr pub;	/* "public" fields */

	jmp_buf setjmp_buffer;	/* for return to caller */
};

typedef struct my_error_mgr * my_error_ptr;

/*
 * Here's the routine that will replace the standard error_exit method:
 */

void my_error_exit (j_common_ptr cinfo)
{
	/* cinfo->err really points to a my_error_mgr struct, so coerce pointer */
	my_error_ptr myerr = (my_error_ptr) cinfo->err;

	/* Always display the message. */
	/* We could postpone this until after returning, if we chose. */
	(*cinfo->err->output_message) (cinfo);

	/* Return control to the setjmp point */
	longjmp(myerr->setjmp_buffer, 1);
}


bool JpegCodec::decodeHeader(const char * filename)
{
	struct stat st;
	if (stat(filename,&st)) {
		fprintf(stderr,"JpegCodec::decodeHeader: failed to stat '%s'\n",filename);
		return false;
	}

	char * src = (char*)malloc(st.st_size);
	FILE * fp = fopen(filename,"rb");
	if (fp == NULL) {
		fprintf(stderr,"JpegCodec::decodeHeader: failed to open '%s'\n",filename);
		return false;
	}
	
	fread(src,st.st_size,1,fp);
	fclose(fp);

	bool res = decodeHeader((const unsigned char*)src,st.st_size);
	free(src);
	return res;
}


bool JpegCodec::decode(const char * filename, DDXVideo * dest, unsigned int field)
{
	struct stat st;
	if (stat(filename,&st)) {
		fprintf(stderr,"JpegCodec::decode: failed to stat '%s'\n",filename);
		return false;
	}

	char * src = (char*)malloc(st.st_size);
	FILE * fp = fopen(filename,"rb");
	if (fp == NULL) {
		fprintf(stderr,"JpegCodec::decode: failed to open '%s'\n",filename);
		return false;
	}
	
	fread(src,st.st_size,1,fp);
	fclose(fp);

	bool res = decode((const unsigned char*)src,st.st_size,dest,field);
	free(src);
	return res;
}


/*
 * Initialize source --- called by jpeg_read_header
 * before any data is actually read.
 */

static void init_source (j_decompress_ptr cinfo)
{
}



static boolean fill_input_buffer (j_decompress_ptr cinfo)
{
  return FALSE;
}


static void skip_input_data (j_decompress_ptr cinfo, long num_bytes)
{
  cinfo->src->next_input_byte += (size_t) num_bytes;
  cinfo->src->bytes_in_buffer -= (size_t) num_bytes;
}


static void term_source (j_decompress_ptr cinfo)
{
  /* no work necessary here */
}


/*
 * Prepare for input from a stdio stream.
 * The caller must have already opened the stream, and is responsible
 * for closing it after finishing decompression.
 */

static void jpeg_mem_src (j_decompress_ptr cinfo, 
		const unsigned char * src, unsigned int srcsize)
{
  /* The source object and input buffer are made permanent so that a series
   * of JPEG images can be read from the same file by calling jpeg_stdio_src
   * only before the first one.  (If we discarded the buffer at the end of
   * one image, we'd likely lose the start of the next one.)
   * This makes it unsafe to use this manager and a different source
   * manager serially with the same JPEG object.  Caveat programmer.
   */
  if (cinfo->src == NULL) {	/* first time for this JPEG object? */
    cinfo->src = (struct jpeg_source_mgr *)
      (*cinfo->mem->alloc_small) ((j_common_ptr) cinfo, JPOOL_PERMANENT,
				  sizeof(jpeg_source_mgr));
	cinfo->src->next_input_byte = src;
	cinfo->src->bytes_in_buffer = srcsize;
  }

  cinfo->src->init_source = init_source;
  cinfo->src->fill_input_buffer = fill_input_buffer;
  cinfo->src->skip_input_data = skip_input_data;
  cinfo->src->resync_to_restart = jpeg_resync_to_restart; /* use default method */
  cinfo->src->term_source = term_source;
}



bool JpegCodec::decodeHeader(const unsigned char * src, unsigned int srcsize)
{
	//printf("Reading\n");
	/* This struct contains the JPEG decompression parameters and pointers to
	 * working space (which is allocated as needed by the JPEG library).
	 */
	struct jpeg_decompress_struct cinfo;
	/* We use our private extension JPEG error handler.
	 * Note that this struct must live as long as the main JPEG parameter
	 * struct, to avoid dangling-pointer problems.
	 */
	struct my_error_mgr jerr;
	/* More stuff */

	/* Step 1: allocate and initialize JPEG decompression object */

	/* We set up the normal JPEG error routines, then override error_exit. */
	cinfo.err = jpeg_std_error(&jerr.pub);
	jerr.pub.error_exit = my_error_exit;
	/* Establish the setjmp return context for my_error_exit to use. */
	if (setjmp(jerr.setjmp_buffer)) {
		/* If we get here, the JPEG code has signaled an error.
		 * We need to clean up the JPEG object, close the input file, and return.
		 */
		jpeg_destroy_decompress(&cinfo);
		return 0;
	}
	/* Now we can initialize the JPEG decompression object. */
	jpeg_create_decompress(&cinfo);

	/* Step 2: specify data source (eg, a file) */

	jpeg_mem_src(&cinfo, src, srcsize);

	/* Step 3: read file parameters with jpeg_read_header() */

	int res;
	res = jpeg_read_header(&cinfo, TRUE);
	//printf("Header : %d\n",res);
	/* We can ignore the return value from jpeg_read_header since
	 *   (a) suspension is not possible with the stdio data source, and
	 *   (b) we passed TRUE to reject a tables-only JPEG file as an error.
	 * See libjpeg.doc for more info.
	 */


	width = cinfo.image_width;
	height = cinfo.image_height;
	size = 0;
	switch (outputColorSpace) {
		case cmAuto :
			switch (cinfo.jpeg_color_space) {
				case JCS_GRAYSCALE:
					outputColorSpace = cmGray;
					cinfo.out_color_space = JCS_GRAYSCALE;
					size = width*height*sizeof(char);
					break;
				case JCS_YCbCr:
					outputColorSpace = cmYUV;
					cinfo.out_color_space = JCS_YCbCr;
					size = width*height*sizeof(char)*3;
					break;
				default :
					outputColorSpace = cmRGB;
					cinfo.out_color_space = JCS_RGB;
					size = width*height*sizeof(char)*3;
					break;
			}
			break;
		case cmRGB:
			outputColorSpace = cmRGB;
			cinfo.out_color_space = JCS_RGB;
			size = width*height*sizeof(char)*3;
			break;
		case cmYUV:
			outputColorSpace = cmYUV;
			cinfo.out_color_space = JCS_YCbCr;
			size = width*height*sizeof(char)*3;
			break;
		case cmGray:
		default :
			outputColorSpace = cmGray;
			cinfo.out_color_space = JCS_GRAYSCALE;
			size = width*height*sizeof(char);
			break;
			
	}
	buffer = (unsigned char*)(realloc(buffer,size));
	/* And we're done! */
	return true;
}


bool JpegCodec::decode(const unsigned char * src, unsigned int srcsize, 
		DDXVideo * dest, unsigned int field)
{
	//printf("Reading\n");
	/* This struct contains the JPEG decompression parameters and pointers to
	 * working space (which is allocated as needed by the JPEG library).
	 */
	struct jpeg_decompress_struct cinfo;
	/* We use our private extension JPEG error handler.
	 * Note that this struct must live as long as the main JPEG parameter
	 * struct, to avoid dangling-pointer problems.
	 */
	struct my_error_mgr jerr;
	/* More stuff */
	int row_stride;		/* physical row width in output buffer */
	JSAMPARRAY lbuffer;		/* Output row buffer */

	/* Step 1: allocate and initialize JPEG decompression object */

	/* We set up the normal JPEG error routines, then override error_exit. */
	cinfo.err = jpeg_std_error(&jerr.pub);
	jerr.pub.error_exit = my_error_exit;
	/* Establish the setjmp return context for my_error_exit to use. */
	if (setjmp(jerr.setjmp_buffer)) {
		/* If we get here, the JPEG code has signaled an error.
		 * We need to clean up the JPEG object, close the input file, and return.
		 */
		jpeg_destroy_decompress(&cinfo);
		return 0;
	}
	/* Now we can initialize the JPEG decompression object. */
	jpeg_create_decompress(&cinfo);

	/* Step 2: specify data source (eg, a file) */

	jpeg_mem_src(&cinfo, src, srcsize);

	/* Step 3: read file parameters with jpeg_read_header() */

	int res = jpeg_read_header(&cinfo, TRUE);
	//printf("Header : %d\n",res);
	/* We can ignore the return value from jpeg_read_header since
	 *   (a) suspension is not possible with the stdio data source, and
	 *   (b) we passed TRUE to reject a tables-only JPEG file as an error.
	 * See libjpeg.doc for more info.
	 */


	width = cinfo.image_width;
	height = cinfo.image_height;
	size = 0;
	switch (outputColorSpace) {
		case cmAuto :
			switch (cinfo.jpeg_color_space) {
				case JCS_GRAYSCALE:
					outputColorSpace = cmGray;
					cinfo.out_color_space = JCS_GRAYSCALE;
					size = width*height*sizeof(char);
					break;
				case JCS_YCbCr:
					outputColorSpace = cmYUV;
					cinfo.out_color_space = JCS_YCbCr;
					size = width*height*sizeof(char)*3;
					break;
				default :
					outputColorSpace = cmRGB;
					cinfo.out_color_space = JCS_RGB;
					size = width*height*sizeof(char)*3;
					break;
			}
			break;
		case cmRGB:
			outputColorSpace = cmRGB;
			cinfo.out_color_space = JCS_RGB;
			size = width*height*sizeof(char)*3;
			break;
		case cmYUV:
			outputColorSpace = cmYUV;
			cinfo.out_color_space = JCS_YCbCr;
			size = width*height*sizeof(char)*3;
			break;
		case cmGray:
		default :
			outputColorSpace = cmGray;
			cinfo.out_color_space = JCS_GRAYSCALE;
			size = width*height*sizeof(char);
			break;
			
	}
	buffer = (unsigned char*)(realloc(buffer,size));
	assert(buffer != NULL);
#if 0
	printf("Data stream: %dx%d, %s : %d bytes\n",width,height,
			rgb?"RGB":"GRAY", size);
#endif

	/* Step 4: set parameters for decompression */

	/* In this example, we don't need to change any of the defaults set by
	 * jpeg_read_header(), so we do nothing here.
	 */

	/* Step 5: Start decompressor */

	(void) jpeg_start_decompress(&cinfo);
	/* We can ignore the return value since suspension is not possible
	 * with the stdio data source.
	 */

	/* We may need to do some setup of our own at this point before reading
	 * the data.  After jpeg_start_decompress() we have the correct scaled
	 * output image dimensions available, as well as the output colormap
	 * if we asked for color quantization.
	 * In this example, we need to make an output work buffer of the right size.
	 */ 
	/* JSAMPLEs per row in output buffer */
	row_stride = cinfo.output_width * cinfo.output_components;
	/* Make a one-row-high sample array that will go away when done with image */
	lbuffer = (*cinfo.mem->alloc_sarray)
		((j_common_ptr) &cinfo, JPOOL_IMAGE, row_stride, 1);

	/* Step 6: while (scan lines remain to be read) */
	/*           jpeg_read_scanlines(...); */

	/* Here we use the library's state variable cinfo.output_scanline as the
	 * loop counter, so that we don't have to keep track ourselves.
	 */
	while (cinfo.output_scanline < cinfo.output_height) {
		/* jpeg_read_scanlines expects an array of pointers to scanlines.
		 * Here the array is only one element long, but you could ask for
		 * more than one scanline at a time if that's more convenient.
		 */
		res = jpeg_read_scanlines(&cinfo, lbuffer, 1);
		//printf("scanline : %d %d\n",res,cinfo.output_scanline);
		//assert((cinfo.output_scanline+1)*cinfo.output_width <= width*height);
		memcpy(buffer + (cinfo.output_scanline-1)*row_stride, 
				lbuffer[0], row_stride);
	}

	/* Step 7: Finish decompression */

	(void) jpeg_finish_decompress(&cinfo);
	/* We can ignore the return value since suspension is not possible
	 * with the stdio data source.
	 */

	/* Step 8: Release JPEG decompression object */

	/* This is an important step since it will release a good deal of memory. */
	jpeg_destroy_decompress(&cinfo);

	/* At this point you may want to check to see whether any corrupt-data
	 * warnings occurred (test whether jerr.pub.num_warnings is nonzero).
	 */

	if (!dest) return true;
	switch (outputColorSpace) {
		case cmGray: 
			dest->readMono(buffer,field);
			break;
		case cmRGB:
			dest->readRGB(buffer,field);
			break;
		case cmYUV:
			dest->readYCbCr(buffer,field);
			break;
		default:
			/* should never arrive here */
			assert(false);
	}

	/* And we're done! */
	return true;
}

/* Expanded data destination object for mem output */

#define OUTPUT_BUF_SIZE  4096	/* choose an efficiently fwrite'able size */
typedef struct {
  struct jpeg_destination_mgr pub; /* public fields */

  size_t buffersize, bufferpos;
  JpegCodec *codec;		/* start of buffer */
  JOCTET * wbuffer;

} my_destination_mgr;

typedef my_destination_mgr * my_dest_ptr;

/*
 * Initialize destination --- called by jpeg_start_compress
 * before any data is actually written.
 */

METHODDEF(void)
init_destination (j_compress_ptr cinfo)
{
  my_dest_ptr dest = (my_dest_ptr) cinfo->dest;

  /* Allocate the output buffer --- it will be released when done with image */
  dest->wbuffer = (JOCTET *)
      (*cinfo->mem->alloc_small) ((j_common_ptr) cinfo, JPOOL_IMAGE,
				  OUTPUT_BUF_SIZE * sizeof(JOCTET));

  dest->pub.next_output_byte = dest->wbuffer;
  dest->pub.free_in_buffer = OUTPUT_BUF_SIZE;
  dest->bufferpos = 0;
}


/*
 * Empty the output buffer --- called whenever buffer fills up.
 *
 * In typical applications, this should write the entire output buffer
 * (ignoring the current state of next_output_byte & free_in_buffer),
 * reset the pointer & count to the start of the buffer, and return TRUE
 * indicating that the buffer has been dumped.
 *
 * In applications that need to be able to suspend compression due to output
 * overrun, a FALSE return indicates that the buffer cannot be emptied now.
 * In this situation, the compressor will return to its caller (possibly with
 * an indication that it has not accepted all the supplied scanlines).  The
 * application should resume compression after it has made more room in the
 * output buffer.  Note that there are substantial restrictions on the use of
 * suspension --- see the documentation.
 *
 * When suspending, the compressor will back up to a convenient restart point
 * (typically the start of the current MCU). next_output_byte & free_in_buffer
 * indicate where the restart point will be if the current call returns FALSE.
 * Data beyond this point will be regenerated after resumption, so do not
 * write it out when emptying the buffer externally.
 */

#if 0
static void printhex(unsigned char * b, unsigned int len) 
{
	unsigned int i;
	for (i=0;i<len;i++) {
		printf("%02X ",(unsigned int)b[i]);
		if ((i%16)==0) printf("\n");
	}
	printf("\n");
}
#endif
	

METHODDEF(boolean)
empty_output_buffer (j_compress_ptr cinfo)
{
  my_dest_ptr dest = (my_dest_ptr) cinfo->dest;
  dest->codec->resizebuffer(dest->codec->bsize() + 10*OUTPUT_BUF_SIZE);

  //printhex(dest->wbuffer,OUTPUT_BUF_SIZE);

  memcpy(dest->codec->getBuffer() + dest->bufferpos, 
		  dest->wbuffer,OUTPUT_BUF_SIZE);

  dest->bufferpos += OUTPUT_BUF_SIZE;
  dest->pub.next_output_byte = dest->wbuffer;
  dest->pub.free_in_buffer = OUTPUT_BUF_SIZE;

  return TRUE;
}


/*
 * Terminate destination --- called by jpeg_finish_compress
 * after all data has been written.  Usually needs to flush buffer.
 *
 * NB: *not* called by jpeg_abort or jpeg_destroy; surrounding
 * application must deal with any cleanup that should happen even
 * for error exit.
 */

METHODDEF(void)
term_destination (j_compress_ptr cinfo)
{
  my_dest_ptr dest = (my_dest_ptr) cinfo->dest;
  size_t datacount = OUTPUT_BUF_SIZE - dest->pub.free_in_buffer;
  //printhex(dest->wbuffer,datacount);

  /* Write any data remaining in the buffer */
  if (datacount > 0) {
	  dest->codec->resizebuffer(dest->codec->bsize() + datacount);
	  memcpy(dest->codec->getBuffer() + dest->bufferpos, 
			  dest->wbuffer,datacount);

	  dest->bufferpos += datacount;
  }
}


/*
 * Prepare for output to a mem stream.
 * The caller must have already opened the stream, and is responsible
 * for closing it after finishing compression.
 */

GLOBAL(void)
jpeg_mem_dest (j_compress_ptr cinfo, JpegCodec *that)
{
  my_dest_ptr dest;

  /* The destination object is made permanent so that multiple JPEG images
   * can be written to the same file without re-executing jpeg_stdio_dest.
   * This makes it dangerous to use this manager and a different destination
   * manager serially with the same JPEG object, because their private object
   * sizes may be different.  Caveat programmer.
   */
  if (cinfo->dest == NULL) {	/* first time for this JPEG object? */
    cinfo->dest = (struct jpeg_destination_mgr *)
      (*cinfo->mem->alloc_small) ((j_common_ptr) cinfo, JPOOL_PERMANENT,
				  sizeof(my_destination_mgr));
  }

  dest = (my_dest_ptr) cinfo->dest;
  dest->pub.init_destination = init_destination;
  dest->pub.empty_output_buffer = empty_output_buffer;
  dest->pub.term_destination = term_destination;
  dest->wbuffer = NULL;
  dest->codec = that;
  dest->bufferpos = 0;
}

bool JpegCodec::encode(DDXVideo * video, int field)
{
	/* This struct contains the JPEG compression parameters and pointers to
	 * working space (which is allocated as needed by the JPEG library).
	 * It is possible to have several such structures, representing multiple
	 * compression/decompression processes, in existence at once.  We refer
	 * to any one struct (and its associated working data) as a "JPEG object".
	 */
	struct jpeg_compress_struct cinfo;
	/* This struct represents a JPEG error handler.  It is declared separately
	 * because applications often want to supply a specialized error handler
	 * (see the second half of this file for an example).  But here we just
	 * take the easy way out and use the standard error handler, which will
	 * print a message on stderr and call exit() if compression fails.
	 * Note that this struct must live as long as the main JPEG parameter
	 * struct, to avoid dangling-pointer problems.
	 */
	struct my_error_mgr jerr;
	/* More stuff */
	JSAMPROW row_pointer[1];	/* pointer to JSAMPLE row[s] */
	int row_stride;		/* physical row width in image buffer */

	/* Step 1: allocate and initialize JPEG compression object */

	/* We have to set up the error handler first, in case the initialization
	 * step fails.  (Unlikely, but it could happen if you are out of memory.)
	 * This routine fills in the contents of struct jerr, and returns jerr's
	 * address which we place into the link field in cinfo.
	 */
	cinfo.err = jpeg_std_error(&jerr.pub);
	jerr.pub.error_exit = my_error_exit;
	/* Establish the setjmp return context for my_error_exit to use. */
	if (setjmp(jerr.setjmp_buffer)) {
		/* If we get here, the JPEG code has signaled an error.
		 * We need to clean up the JPEG object, close the input file, and return.
		 */
		jpeg_destroy_compress(&cinfo);
		return 0;
	}
	/* Now we can initialize the JPEG compression object. */
	jpeg_create_compress(&cinfo);

	/* Step 2: specify data destination (eg, a file) */
	/* Note: steps 2 and 3 can be done in either order. */

	jpeg_mem_dest(&cinfo, this);

	/* Step 3: set parameters for compression */

	/* First we supply a description of the input image.
	 * Four fields of the cinfo struct must be filled in:
	 */
	cinfo.image_width = video->width(); 	/* image width and height, in pixels */
	cinfo.image_height = video->height();
	if (field < 0) {
		cinfo.image_height *= video->fields();
#if 0
		printf("Recording all %d fields -> height %d\n",
				video->fields(), cinfo.image_height);
#endif
	}
	unsigned int csize = cinfo.image_width*cinfo.image_height;
	unsigned char * src = NULL;
	switch (video->type()) {
		case DDXVideo::MONO8 : 
			cinfo.input_components = 1;
			cinfo.in_color_space = JCS_GRAYSCALE;
			row_stride = cinfo.image_width * 1;	/* JSAMPLEs per row in image_buffer */
			if (field < 0) {
				src = video->greyPlane(0);
			} else {
				src = video->greyPlane(field);
			}
			break;
		case DDXVideo::RGB8 :
			cinfo.input_components = 3;
			cinfo.in_color_space = JCS_YCbCr;
			row_stride = cinfo.image_width * 3;	/* JSAMPLEs per row in image_buffer */
			csize *= 3;
			conv_buffer = (unsigned char*)realloc(conv_buffer,csize);
			if (field < 0) {
				unsigned char * it = conv_buffer;
				for (unsigned int i=0;i<video->fields();i++) {
					video->convertToYCbCr(it,i);
					it += video->width()*video->height()*3;
				}
			} else {
				video->convertToYCbCr(conv_buffer,field);
			}
			src = conv_buffer;
			break;
		case DDXVideo::YUV420 :
			cinfo.input_components = 3;
			cinfo.in_color_space = JCS_YCbCr;
			row_stride = cinfo.image_width * 3;	/* JSAMPLEs per row in image_buffer */
			csize *= 3;
			conv_buffer = (unsigned char*)realloc(conv_buffer,csize);
			if (field < 0) {
				unsigned char * it = conv_buffer;
				for (unsigned int i=0;i<video->fields();i++) {
					video->convertToYCbCr(it,i);
					it += video->width()*video->height()*3;
				}
			} else {
				video->convertToYCbCr(conv_buffer,field);
			}
			src = conv_buffer;
			break;
		default :
			fprintf(stderr,"Invalid DDXVideo color type\n");
			jpeg_destroy_compress(&cinfo);
			return false;
	}
			
			
			
	/* Now use the library's routine to set default compression parameters.
	 * (You must set at least cinfo.in_color_space before calling this,
	 * since the defaults depend on the source color space.)
	 */
	jpeg_set_defaults(&cinfo);
	/* Now you can set any non-default parameters you wish to.
	 * Here we just illustrate the use of quality (quantization table) scaling:
	 */
	jpeg_set_quality(&cinfo, quality, TRUE /* limit to baseline-JPEG values */);

	switch (outputColorSpace) {
		case cmGray:
			cinfo.num_components = 1;
			cinfo.jpeg_color_space = JCS_GRAYSCALE;
			break;
		case cmRGB:
			cinfo.num_components = 3;
			cinfo.jpeg_color_space = JCS_RGB;
			break;
		case cmYUV:
			cinfo.num_components = 3;
			cinfo.jpeg_color_space = JCS_YCbCr;
			break;
		case cmAuto:
			break;
		default :
			fprintf(stderr,"Invalid output color type\n");
			return false;
	}

	/* Step 4: Start compressor */

	/* TRUE ensures that we will write a complete interchange-JPEG file.
	 * Pass TRUE unless you are very sure of what you're doing.
	 */
	jpeg_start_compress(&cinfo, TRUE);

	/* Step 5: while (scan lines remain to be written) */
	/*           jpeg_write_scanlines(...); */

	/* Here we use the library's state variable cinfo.next_scanline as the
	 * loop counter, so that we don't have to keep track ourselves.
	 * To keep things simple, we pass one scanline per call; you can pass
	 * more if you wish, though.
	 */

	while (cinfo.next_scanline < cinfo.image_height) {
		/* jpeg_write_scanlines expects an array of pointers to scanlines.
		 * Here the array is only one element long, but you could pass
		 * more than one scanline at a time if that's more convenient.
		 */
		row_pointer[0] = src + (cinfo.image_height - cinfo.next_scanline - 1)*row_stride;
		(void) jpeg_write_scanlines(&cinfo, row_pointer, 1);
	}


	/* Step 6: Finish compression */

	jpeg_finish_compress(&cinfo);
	resizebuffer(((my_dest_ptr)(cinfo.dest))->bufferpos);

	/* Step 7: release JPEG compression object */

	/* This is an important step since it will release a good deal of memory. */
	jpeg_destroy_compress(&cinfo);

	return true;

}

bool JpegCodec::encode(DDXVideo * src, const char * filename, unsigned int field)
{
	bool res = encode(src,field);
	if (!res) return false;
	FILE *fp = fopen(filename,"wb");
	if (fp == NULL) {
		fprintf(stderr,"JpegCodec::encode: could not create '%s'\n",filename);
		return false;
	}
	fwrite(buffer,size,1,fp);
	fclose(fp);
	return true;
}


void JpegCodec::setOutputColorSpace(ColorSpace cspace)
{
	outputColorSpace = cspace;
}

void JpegCodec::setInputColorSpace(ColorSpace cspace)
{
	inputColorSpace = cspace;
}

bool JpegCodec::resizebuffer(unsigned int newsize)
{
	size = newsize;
	buffer = (unsigned char*)realloc(buffer,size);
	return (buffer != NULL);
}

