#include <stdlib.h>
#include <stdio.h>
#include <assert.h>
#include <string.h>
#include <string>
#include <rtx/error.h>
#include <rtx/message.h>

#define DDX_VIDEO_IMPLEMENTATION
#include "DDXVideo.h"


// Uncomment that to avoid inverting U and V in the std_video
// Standard structure...
#define PRESERVE_COMPATIBILTY
#ifdef PRESERVE_COMPATIBILTY
//#warning Preserving compatibility with std_video: inverting U and V
#endif

using namespace std;

DDXVideo::DDXVideo()
{
	video = NULL;
	std_video = NULL;
}

DDXVideo::~DDXVideo()
{
}


static const char * codingString(DDXVideo::CodingType type)
{
	switch (type) {
		case DDXVideo::MONO8: 
			return "MONO8";
		case DDXVideo::RGB8:
			return "RGB8";
		case DDXVideo::YUV420:
			return "YUV420";
		default :
			break;
	}
	return "UNDEF_TYPE";
}

static unsigned int codingSize(unsigned int width, unsigned int height,
		DDXVideo::CodingType type, unsigned int fields)
{
	switch (type) {
		case DDXVideo::MONO8: 
			return width*height*fields;
		case DDXVideo::RGB8:
			return width*height*3*fields;
		case DDXVideo::YUV420:
			return (width*height + 2*width*height/4)*fields;
		default :
			break;
	}
	return 0;
}

bool DDXVideo::registerAdvancedVariable(DDXStore & store, const std::string & name, 
		CodingType coding, unsigned int width, unsigned int height, 
		unsigned int nimages, bool _direct)
{
	if ((video != NULL) && (std_video != NULL)) 
		return rtx_error_cpp("DDXVideo::registerAdvancedVariable variable already registered") ;

	std_video = NULL;

	unsigned int num_bytes = codingSize(width,height,coding,nimages);

	// Generation of type name and description
	char pattern_name[] = "std_video_%dx%d_%sx%d";
	char pattern_descr[] = 
		"struct {int width;int height;int frame;int fields;"
		"	int type;int num_bytes;char bytes[%d];} %s;";
	char type_descr[1024];
	char type_name[1024];
	sprintf(type_name,pattern_name,
			width,height,codingString(coding),nimages);
	sprintf(type_descr,pattern_descr,num_bytes,type_name);

	if (!store.registerType (string(type_descr))) {
		return rtx_error_cpp("DDXVideo::registerAdvancedVariable: registerType failed");
	}
	if (!store.registerVariable (*this, name,
				string(type_name), _direct)) {
		return rtx_error_cpp("DDXVideo::registerAdvancedVariable: registerVariable failed");
	}

	printf("DDXVideo: allocated %d bytes\n",size());
	video = (BaseVideo*)rawPointer();
	if (video == NULL) return rtx_error_cpp("DDXVideo::registerAdvancedVariable: rawPointer is NULL");

	video->type = coding;
	video->fields = nimages;
	video->num_bytes= num_bytes;

	setSize(width,height);
	setFrame(0);

	return write();
}

bool DDXVideo::parseStdVariable()
{
	std::string def = definition();
	//printf("Definition: %s\n",def.c_str());
	unsigned int ypos = def.find(" y");
	if (ypos == def.length()) {return rtx_error_cpp("DDXVideo::parseStdVariable: can't find y buffer");}
	unsigned int upos = def.find(" u");
	if (upos == def.length()) {return rtx_error_cpp("DDXVideo::parseStdVariable: can't find u buffer");}
	unsigned int vpos = def.find(" v");
	if (vpos == def.length()) {return rtx_error_cpp("DDXVideo::parseStdVariable: can't find v buffer");}
	//printf("Pos: %d %d %d\n",ypos,upos,vpos);
	stdYsize = stdUsize = stdVsize = 0;
	sscanf(def.c_str()+ypos+2,"[%d]",&stdYsize);
	if (stdYsize == 0) return rtx_error_cpp("DDXVideo::parseStdVariable: can't parse Y size");
	sscanf(def.c_str()+upos+2,"[%d]",&stdUsize);
	if (stdUsize == 0) return rtx_error_cpp("DDXVideo::parseStdVariable: can't parse U size");
	sscanf(def.c_str()+vpos+2,"[%d]",&stdVsize);
	//printf("Size: %d %d %d\n",stdYsize,stdUsize,stdVsize);
	if (stdVsize == 0) return rtx_error_cpp("DDXVideo::parseStdVariable: can't parse V size");
	if (stdVsize < stdYsize/4)
		rtx_message("DDXVideo::parseStdVariable: Warning: inconsistent V size\n");
	if (stdUsize < stdYsize/4)
		rtx_message("DDXVideo::parseStdVariable: Warning: inconsistent U size\n");
	//printf("Standard variable: Y %d U %d V %d\n", stdYsize,stdUsize,stdVsize);
	return true;
}

bool DDXVideo::registerStdVariable(DDXStore & store, const std::string & name, 
		unsigned int width, unsigned int height, 
		unsigned int nimages, bool _direct)
{
	if ((video != NULL) && (std_video != NULL)) 
		return rtx_error_cpp("DDXVideo::registerStdVariable variable already registered") ;

	video = NULL;

	if (DDX_STORE_REGISTER_TYPE(store.getId(),DDX_VIDEO) != 0) {
		return false;
	}
	if (!store.registerVariable (*this, name,
				string("DDX_VIDEO"), _direct)) {
		return rtx_error_cpp("DDXVideo::registerStdVariable: registerType failed");
	}
	if (!parseStdVariable()) {
		rtx_message("DDXVideo::registerStdVariable: Warning: failed to parse std variable\n");
	}

	printf("DDXVideo: allocated %d bytes\n",size());
	std_video = (StdVideo*)rawPointer();
	if (std_video == NULL) return rtx_error_cpp("DDXVideo::registerStdVariable: rawPointer is NULL");
	std_video->fields = nimages;

	setSize(width,height);
	setFrame(0);

	return write();
}


bool DDXVideo::lookupVariable(DDXStore & store, const std::string & name,
		bool _direct)
{
	if ((video != NULL) && (std_video != NULL)) 
		return rtx_error_cpp("DDXVideo::lookupVariable variable already registered") ;

	video = NULL;
	std_video = NULL;

	bool res = store.lookupVariable(*this,name,_direct);
	if (!res) return rtx_error_cpp("DDXVideo::lookupVariable: lookupVariable failed");
	if (definition().find("type") < definition().length()) {
		// printf("Advanced variable\n");
		video = (BaseVideo*)rawPointer();
	} else {
		// printf("Standard variable\n");
		if (!parseStdVariable()) {
			rtx_message("DDXVideo::lookupVariable: Warning: failed to parse std variable\n");
		}
		std_video = (StdVideo*)rawPointer();
	}
	return res;
}

///////////////////////////////////////////////////////////
//
// Class specific operations (non const)

unsigned char * DDXVideo::rgbPlane(unsigned int field) {
	if (video == NULL) 
		return (unsigned char*)rtx_error_null("DDXVideo::rgbPlane: video object not initialised");
	if (video->type != RGB8) 
		return (unsigned char*)rtx_error_null("DDXVideo::rgbPlane: image is not in RGB8 format");
	if (field >= video->fields) {
		return (unsigned char*)rtx_error_null("DDXVideo::rgbPlane: requested field does not exist");
	}
	unsigned int size = video->width*video->height;
	return video->bytes + field*3*size; /* + 0*(width*height) */
}

unsigned char * DDXVideo::YPlane(unsigned int field) 
{
	if ((video == NULL) && (std_video == NULL))
		return (unsigned char*)rtx_error_null("DDXVideo::YPlane: video object not initialised");
	if (video == NULL) {
		if (field >= (unsigned int)std_video->fields) {
			return (unsigned char*)rtx_error_null("DDXVideo::rgbPlane: requested field does not exist");
		}
		unsigned int n = std_video->width*std_video->height;
		return (unsigned char*)(std_video->bytes+n*field);
	} else {
		if (field >= video->fields) {
			return (unsigned char*)rtx_error_null("DDXVideo::rgbPlane: requested field does not exist");
		}
		if (video->type != YUV420) return (unsigned char*)rtx_error_null("DDXVideo::YPlane: image is not in YUV format");
		unsigned int n = video->width*video->height;
		return video->bytes + (field * 3 * n)/2; /* + 0*(width*height) */
	}
}

unsigned char * DDXVideo::UPlane(unsigned int field) 
{
	if ((video == NULL) && (std_video == NULL))
		return (unsigned char*)rtx_error_null("DDXVideo::UPlane: video object not initialised");
	if (video == NULL) {
		if (field >= (unsigned int)std_video->fields) {
			return (unsigned char*)rtx_error_null("DDXVideo::rgbPlane: requested field does not exist");
		}
		unsigned int n = std_video->width*std_video->height/4;
#ifdef PRESERVE_COMPATIBILTY
		return (unsigned char*)(std_video->bytes+
				stdYsize+stdUsize+n*field);
#else
		return (unsigned char*)(std_video->bytes+
				stdYsize+n*field);
#endif
	} else {
		if (field >= video->fields) {
			return (unsigned char*)rtx_error_null("DDXVideo::rgbPlane: requested field does not exist");
		}
		if (video->type != YUV420) return (unsigned char*)rtx_error_null("DDXVideo::UPlane: image is not in YUV format");
		unsigned int n = video->width*video->height;
		return video->bytes + (field * 3 * n)/2 + n;
	}
}

unsigned char * DDXVideo::VPlane(unsigned int field) 
{
	if ((video == NULL) && (std_video == NULL))
		return (unsigned char*)rtx_error_null("DDXVideo::VPlane: video object not initialised");
	if (video == NULL) {
		if (field >= (unsigned int)std_video->fields) {
			return (unsigned char*)rtx_error_null("DDXVideo::rgbPlane: requested field does not exist");
		}
		unsigned int n = std_video->width*std_video->height/4;
#ifdef PRESERVE_COMPATIBILTY
		return (unsigned char*)(std_video->bytes+
				stdYsize+n*field);
#else
		return (unsigned char*)(std_video->bytes+
				stdYsize+stdUsize+n*field);
#endif
	} else {
		if (field >= video->fields) {
			return (unsigned char*)rtx_error_null("DDXVideo::rgbPlane: requested field does not exist");
		}
		if (video->type != YUV420) return (unsigned char*)rtx_error_null("DDXVideo::VPlane: image is not in YUV format");
		unsigned int n = video->width*video->height;
		return video->bytes + (field * 3 * n)/2 + n + n/4;
	}
}

unsigned char * DDXVideo::greyPlane(unsigned int field) {
	if (video == NULL) 
		return (unsigned char*)rtx_error_null("DDXVideo::greyPlane: video object not initialised");
	if (video->type != MONO8) return (unsigned char*)rtx_error_null("DDXVideo::greyPlane: image is not in MONO8 format");
	if (field >= video->fields) {
		return (unsigned char*)rtx_error_null("DDXVideo::rgbPlane: requested field does not exist");
	}
	unsigned int size = video->width*video->height;
	return video->bytes+field*size; 
}


///////////////////////////////////////////////////////////
//
// Class specific operations (const)

const unsigned char * DDXVideo::rgbPlane(unsigned int field) const {
	if (video == NULL) 
		return (const unsigned char*)rtx_error_null("DDXVideo::rgbPlane: video object not initialised");
	if (video->type != RGB8) 
		return (const unsigned char*)rtx_error_null("DDXVideo::rgbPlane: image is not in RGB8 format");
	if (field >= video->fields) {
		return (unsigned char*)rtx_error_null("DDXVideo::rgbPlane: requested field does not exist");
	}
	unsigned int size = video->width*video->height;
	return video->bytes + field*3*size; /* + 0*(width*height) */
}

const unsigned char * DDXVideo::YPlane(unsigned int field) const
{
	if ((video == NULL) && (std_video == NULL))
		return (const unsigned char*)rtx_error_null("DDXVideo::YPlane: video object not initialised");
	if (video == NULL) {
		if (field >= (unsigned int)std_video->fields) {
			return (unsigned char*)rtx_error_null("DDXVideo::rgbPlane: requested field does not exist");
		}
		unsigned int n = std_video->width*std_video->height;
		return (unsigned char*)(std_video->bytes+n*field);
	} else {
		if (field >= video->fields) {
			return (unsigned char*)rtx_error_null("DDXVideo::rgbPlane: requested field does not exist");
		}
		if (video->type != YUV420) return (const unsigned char*)rtx_error_null("DDXVideo::YPlane: image is not in YUV format");
		unsigned int n = video->width*video->height;
		return video->bytes + (field * 3 * n)/2; /* + 0*(width*height) */
	}
}

const unsigned char * DDXVideo::UPlane(unsigned int field) const
{
	if ((video == NULL) && (std_video == NULL))
		return (const unsigned char*)rtx_error_null("DDXVideo::UPlane: video object not initialised");
	if (video == NULL) {
		if (field >= (unsigned int)std_video->fields) {
			return (unsigned char*)rtx_error_null("DDXVideo::rgbPlane: requested field does not exist");
		}
		unsigned int n = std_video->width*std_video->height/4;
#ifdef PRESERVE_COMPATIBILTY
		return (unsigned char*)(std_video->bytes+
				stdYsize+stdUsize+n*field);
#else
		return (unsigned char*)(std_video->bytes+
				stdYsize+n*field);
#endif
	} else {
		if (field >= video->fields) {
			return (unsigned char*)rtx_error_null("DDXVideo::rgbPlane: requested field does not exist");
		}
		if (video->type != YUV420) return (const unsigned char*)rtx_error_null("DDXVideo::UPlane: image is not in YUV format");
		unsigned int n = video->width*video->height;
		return video->bytes + (field * 3 * n)/2 + n;
	}
}

const unsigned char * DDXVideo::VPlane(unsigned int field) const
{
	if ((video == NULL) && (std_video == NULL))
		return (const unsigned char*)rtx_error_null("DDXVideo::VPlane: video object not initialised");
	if (video == NULL) {
		if (field >= (unsigned int)std_video->fields) {
			return (unsigned char*)rtx_error_null("DDXVideo::rgbPlane: requested field does not exist");
		}
		unsigned int n = std_video->width*std_video->height/4;
#ifdef PRESERVE_COMPATIBILTY
		return (unsigned char*)(std_video->bytes+
				stdYsize+n*field);
#else
		return (unsigned char*)(std_video->bytes+
				stdYsize+stdUsize+n*field);
#endif
	} else {
		if (field >= video->fields) {
			return (unsigned char*)rtx_error_null("DDXVideo::rgbPlane: requested field does not exist");
		}
		if (video->type != YUV420) return (const unsigned char*)rtx_error_null("DDXVideo::VPlane: image is not in YUV format");
		unsigned int n = video->width*video->height;
		return video->bytes + (field * 3 * n)/2 + n + n/4;
	}
}

const unsigned char * DDXVideo::greyPlane(unsigned int field) const {
	if (video == NULL) 
		return (const unsigned char*)rtx_error_null("DDXVideo::greyPlane: video object not initialised");
	if (video->type != MONO8) return (const unsigned char*)rtx_error_null("DDXVideo::greyPlane: image is not in MONO8 format");
	if (field >= video->fields) {
		return (unsigned char*)rtx_error_null("DDXVideo::rgbPlane: requested field does not exist");
	}
	unsigned int size = video->width*video->height;
	return video->bytes+field*size; 
}


unsigned int DDXVideo::fields() const
{
	if ((video == NULL) && (std_video == NULL))
		return rtx_error("DDXVideo::fields: video object not initialised");
	if (video == NULL) {
		return std_video->fields;
	} else {
		return video->fields;
	}
}

unsigned int DDXVideo::width() const
{
	if ((video == NULL) && (std_video == NULL))
		return rtx_error("DDXVideo::width: video object not initialised");
	if (video == NULL) {
		return std_video->width;
	} else {
		return video->width;
	}
}

unsigned int DDXVideo::height() const
{
	if ((video == NULL) && (std_video == NULL))
		return rtx_error("DDXVideo::height: video object not initialised");
	if (video == NULL) {
		return std_video->height;
	} else {
		return video->height;
	}
}

DDXVideo::CodingType DDXVideo::type() const
{
	if (video == NULL) return YUV420;
	switch (video->type) {
		case MONO8 : return MONO8;
		case RGB8 : return RGB8;
		case YUV420 : return YUV420;
		default : break;
	}
	return YUV420;
}

unsigned int DDXVideo::frame() const
{
	if ((video == NULL) && (std_video == NULL))
		return rtx_error("DDXVideo::frame: video object not initialised");
	if (video == NULL) {
		return std_video->frame;
	} else {
		return video->frame;
	}
}

bool DDXVideo::setSize(unsigned int width,unsigned int height)
{
	if ((video == NULL) && (std_video == NULL)) {
		rtx_error("DDXVideo::setSize: video object not initialised");
		return false;
	}
	if (video == NULL) {
		//assert((fields()*width*height)  <= sizeof(stdYsize));
		std_video->width = width;
		std_video->height = height;
	} else {
		assert(width*height*fields() <= video->num_bytes);
		video->width = width;
		video->height = height;
	}
	return true;
}

void DDXVideo::setFrame(unsigned int f)
{
	if ((video == NULL) && (std_video == NULL)) {
		rtx_error("DDXVideo::setFrame: video object not initialised");
		return ;
	}
	if (video == NULL) 
		std_video->frame = f;
	else 
		video->frame = f;
}

/**
 * Convert YUV to RGB 
 * \warning These value must be ints
 */

#define YUV2RGB(y,u,v,r,g,b)       u -= 128; v -= 128; \
										r = y + ((v*1436) >> 10);        r = r < 0 ? 0 : r; r = r > 255 ? 255 : r; \
g = y - ((u*352 + v*731) >> 10); g = g < 0 ? 0 : g; g = g > 255 ? 255 : g; \
b = y + ((u*1814) >> 10);        b = b < 0 ? 0 : b; b = b > 255 ? 255 : b

/**
 * Convert RGB to YUV
 * \warning These value must be ints
 */

#define RGB2YUV(r,g,b,y,u,v) \
	y = (306*r + 601*g + 117*b)  >> 10;          y = y < 0 ? 0 : y; y = y > 255 ? 255 : y; \
	u = ((-172*r - 340*g + 512*b) >> 10)  + 128; u = u < 0 ? 0 : u; u = u > 255 ? 255 : u; \
	v = ((512*r - 429*g - 83*b) >> 10) + 128;    v = v < 0 ? 0 : v; v = v > 255 ? 255 : v

#define RGB2Y(r,g,b,y) \
	y = (306*r + 601*g + 117*b)  >> 10;          y = y < 0 ? 0 : y; y = y > 255 ? 255 : y 


void DDXVideo::readMono(const unsigned char * buffer,unsigned int field)
{
	if ((video == NULL) && (std_video == NULL)) {
		rtx_error("DDXVideo::readMono: video object not initialised");
		return ;
	}
	if (field >= fields()) {
		rtx_error("DDXVideo::readMono: requested field does not exits");
		return ;
	}
	unsigned int i;
	unsigned int w = width();
	unsigned int h = height();
	unsigned int p = w*h;
	switch (type()) {
		case MONO8 :
			memcpy(greyPlane(field),buffer,video->num_bytes);
			break;
		case YUV420 :
			memcpy(YPlane(field),buffer,p);
			memset(UPlane(field),128,p/4);
			memset(VPlane(field),128,p/4);
			break;
		case RGB8 :
			{
				unsigned char * rgb = rgbPlane(field);
				for (i=0;i<p;i++) {
					rgb[0] = rgb[1] =rgb[2] =  *buffer++;
					rgb += 3;
				}
			}
			break;
	}
}

void DDXVideo::readMonoWithStride(const unsigned char * buffer,
		unsigned int stride, unsigned int field)
{
	if ((video == NULL) && (std_video == NULL)) {
		rtx_error("DDXVideo::readMono: video object not initialised");
		return ;
	}
	if (field >= fields()) {
		rtx_error("DDXVideo::readMonoWithStride: requested field does not exits");
		return ;
	}
	unsigned int i,j;
	unsigned int w = width();
	unsigned int h = height();
	unsigned int p = w*h;
	switch (type()) {
		case MONO8 :
			for (i=0;i<h;i++) {
				const unsigned char * line = buffer + i * stride;
				memcpy(greyPlane(field)+i*w,line,w);
			}
			break;
		case YUV420 :
			for (i=0;i<h;i++) {
				const unsigned char * line = buffer + i * stride;
				memcpy(YPlane(field)+i*w,line,w);
			}
			memset(UPlane(field),128,p/4);
			memset(VPlane(field),128,p/4);
			break;
		case RGB8 :
			{
				for (j=0;j<h;j++) {
					const unsigned char * line = buffer + j*stride;
					unsigned char * rgb = rgbPlane(field) + j*w*3;
					for (i=0;i<w;i++) {
						rgb[0] = rgb[1] =rgb[2] =  *line++;
						rgb += 3;
					}
				}
			}
			break;
	}
}

void DDXVideo::readMono16(const unsigned char * buffer,unsigned int field)
{
	if ((video == NULL) && (std_video == NULL)) {
		rtx_error("DDXVideo::readMono16: video object not initialised");
		return ;
	}
	if (field >= fields()) {
		rtx_error("DDXVideo::readMono16: requested field does not exits");
		return ;
	}
	unsigned int i;
	unsigned int w = width();
	unsigned int h = height();
	unsigned int p = w*h;
	switch (type()) {
		case MONO8 :
			{
				unsigned char * gp = greyPlane(field);
				for (i=0;i<p;i++) { *gp++ = *buffer++; buffer++;}
			}
			break;
		case YUV420 :
			{
				unsigned char * yp = YPlane(field);
				for (i=0;i<p;i++) { *yp++ = *buffer++; buffer++;}
				memset(UPlane(field),128,p/4);
				memset(VPlane(field),128,p/4);
			}
			break;
		case RGB8 :
			{
				unsigned char * rgb = rgbPlane(field);
				for (i=0;i<p;i++) {
					rgb[0] = rgb[1] =rgb[2] =  *buffer++;
					rgb += 3; buffer += 1;
				}
			}
			break;
	}
}




void DDXVideo::readRGB(const unsigned char * buffer,unsigned int field)
{
	if ((video == NULL) && (std_video == NULL)) {
		rtx_error("DDXVideo::readRGB: video object not initialised");
		return ;
	}
	if (field >= fields()) {
		rtx_error("DDXVideo::readRGB: requested field does not exits");
		return ;
	}
	unsigned int i,j;
	unsigned int w = width();
	unsigned int h = height();
	unsigned int p = w*h;

	switch (type()) {
		case MONO8 :
			{
				unsigned char * grey = greyPlane(field);
				for (i=0;i<p;i++) {
					unsigned int r,g,b;
					int y;
					r = *buffer++;
					g = *buffer++;
					b = *buffer++;
					y = (306*r + 601*g + 117*b)  >> 10;   
					y = y < 0 ? 0 : y; y = y > 255 ? 255 : y; 
					*grey++ = y;
				}
			}
			break;
		case YUV420 :
			{
				unsigned char *yp,*up,*vp;
				yp = YPlane(field);
				up = UPlane(field);
				vp = VPlane(field);
				for (j=0;j<h;j++) {
					for (i=0;i<w;i++) {
						int r,g,b;
						int y,u,v;
						r = *buffer++;
						g = *buffer++;
						b = *buffer++;
						RGB2YUV(r,g,b,y,u,v);
						*yp++ = y;
						if (((i%2)==0) && ((j%2)==0)) {
							*up++ = u;
							*vp++ = v;
						}
					}
				}
			}
			break;
		case RGB8 :
			memcpy(rgbPlane(field),buffer,3*p);
			break;
	}
}

void DDXVideo::readYUV411(const unsigned char * buffer,unsigned int field)
{
	if ((video == NULL) && (std_video == NULL)) {
		rtx_error("DDXVideo::readYUV411: video object not initialised");
		return ;
	}
	if (field >= fields()) {
		rtx_error("DDXVideo::readYUV411: requested field does not exits");
		return ;
	}
	unsigned int i,j;
	unsigned int w = width();
	unsigned int h = height();

	switch (type()) {
		case MONO8 :
			{
				unsigned char *y = greyPlane(field);

				for(j=0; j<h; j++ ) {
					for(i=0; i<w*3/2; i+=6) { 
						buffer++; *y++ = *buffer++; *y++ = *buffer++; 
						buffer++; *y++ = *buffer++; *y++ = *buffer++; 
					}
				}
			}
			break;
		case YUV420 :
			{
				unsigned char *y,*u,*v;
				y = YPlane(field);
				u = UPlane(field);
				v = VPlane(field);
				unsigned char c;

				for(j=0; j<h; j+=2 ) {
					for(i=0; i<w*3/2; i+=6) { 
						c = *buffer++; *u++ = c; *u++ = c; 
						*y++ = *buffer++; *y++ = *buffer++; 
						c = *buffer++; *v++ = c; *v++ = c; 
						*y++ = *buffer++; *y++ = *buffer++; 
					}
					for(i=0; i<w*3/2; i+=6) { 
						buffer++; *y++ = *buffer++; *y++ = *buffer++; 
						buffer++; *y++ = *buffer++; *y++ = *buffer++; 
					}
				}
			}
			break;
		case RGB8 :
			{
				rtx_error_flush("DDXVideo::readYUV411: read to RGB8 not implemented");
			}
			break;
	}
}

void DDXVideo::readYUV422(const unsigned char * buffer,unsigned int field)
{
	if ((video == NULL) && (std_video == NULL)) {
		rtx_error("DDXVideo::readYUV422: video object not initialised");
		return ;
	}
	if (field >= fields()) {
		rtx_error("DDXVideo::readYUV422: requested field does not exits");
		return ;
	}
	unsigned int i,j;
	unsigned int w = width();
	unsigned int h = height();

	switch (type()) {
		case MONO8 :
			{
				unsigned char *y;
				y = greyPlane(field);
				unsigned int l;
				for(j=0; j<h; j++) {
					l = j*w*2;
					for(i=0; i<w*2;) { 
						*y++ = buffer[l+i++]; i++;
						*y++ = buffer[l+i++]; i++; 
					}
				}
			}
			break;
		case YUV420 :
			{
				unsigned char *y,*u,*v;
				y = YPlane(field);
				u = UPlane(field);
				v = VPlane(field);
				unsigned int l;
				for(j=0; j<h; j++) {
					l = j*w*2;
					for(i=0; i<w*2;) { 
						*y++ = buffer[l+i];i++;
						*u++ = buffer[l+i];i++; 
						*y++ = buffer[l+i];i++;
						*v++ = buffer[l+i];i++;
					}
					j++;    // drop even lines
					for(i=0; i<w*2;) { 
						*y++ = buffer[l+i]; 
						i+=2; 
						*y++ = buffer[l+i]; 
						i+=2; 
					}
				}
			}
			break;
		case RGB8 :
			{
				rtx_error_flush("DDXVideo::readYUV422: read to RGB8 not implemented");
			}
			break;
	}
}

void DDXVideo::readYCbCr(const unsigned char * buffer,unsigned int field)
{
	if ((video == NULL) && (std_video == NULL)) {
		rtx_error("DDXVideo::readYCbCr: video object not initialised");
		return ;
	}
	if (field >= fields()) {
		rtx_error("DDXVideo::readYCbCr: requested field does not exits");
		return ;
	}
	unsigned int i,j;
	unsigned int w = width();
	unsigned int h = height();

	switch (type()) {
		case MONO8 :
			{
				unsigned int np = w*h;
				unsigned char *y;
				const unsigned char *p;
				y = greyPlane(field);
				p = buffer;
				for (j = 0; j < np; j += 1, p+=3) {*y++ = *p;}
			}
			break;
		case YUV420 :
			{
				unsigned char *y,*u,*v;
				y = YPlane(field);
				u = UPlane(field);
				v = VPlane(field);
				const unsigned char * p = buffer;
				for (j = 0; j < h; j += 2) {
					for (i = 0; i < w; i += 2, p += 6) {
						*y++ = p[0];*y++ = p[3];
						*u++ = p[1]; *v++ = p[2];
					}
					for (i = 0; i < w; i += 2, p += 6) {
						*y++ = p[0];*y++ = p[3];
					}
				}
			}
			break;
		case RGB8 :
			{
				rtx_error_flush("DDXVideo::readYCbCr: read to RGB8 not implemented");
			}
			break;
	}
}

void DDXVideo::readYUV444(const unsigned char * buffer,unsigned int field)
{
	if ((video == NULL) && (std_video == NULL)) {
		rtx_error("DDXVideo::readYUV444: video object not initialised");
		return ;
	}
	if (field >= fields()) {
		rtx_error("DDXVideo::readYUV444: requested field does not exits");
		return ;
	}
	unsigned int i,j;
	unsigned int w = width();
	unsigned int h = height();

	switch (type()) {
		case MONO8 :
			{
				unsigned char *y;
				y = greyPlane(field);
				for (j = 0; j < h; j += 1) {
					for (i = 0; i < w; i += 1) {
						const unsigned char * p = buffer + (j*w+i)*3;
						*y++ = p[1];
					}
				}
			}
			break;
		case YUV420 :
			{
				unsigned char *y,*u,*v;
				y = YPlane(field);
				u = UPlane(field);
				v = VPlane(field);
				const unsigned char * p = buffer;
				for (j = 0; j < h; j += 2) {
					for (i = 0; i < w; i += 2, p += 6) {
						*y++ = p[1];*y++ = p[4];
						*u++ = p[0]; *v++ = p[2];
					}
					for (i = 0; i < w; i += 2, p += 6) {
						*y++ = p[1];*y++ = p[4];
					}
				}
			}
			break;
		case RGB8 :
			{
				rtx_error_flush("DDXVideo::readYUV444: read to RGB8 not implemented");
			}
			break;
	}
}

void DDXVideo::readBayerRGGB(const unsigned char * buffer,unsigned int field)
{
	if ((video == NULL) && (std_video == NULL)) {
		rtx_error("DDXVideo::readBayerRGGB: video object not initialised");
		return ;
	}
	if (field >= fields()) {
		rtx_error("DDXVideo::readBayerRGGB: requested field does not exits");
		return ;
	}
	unsigned int i,j;
	unsigned int w = width();
	unsigned int h = height();

	switch (type()) {
		case MONO8 :
			{
				unsigned char *yp = greyPlane(field);
				int r,g,b,y,u,v,l;

				for(i=0; i<h; i+=2)  {
					for(j=0; j<w; j+=2) {
						l = i*w +j;
						r = buffer[l];
						g = (buffer[l+1]+buffer[l+w])/2;
						b = buffer[l+w+1];
						RGB2YUV(r,g,b,y,u,v);
						yp[l]     = y;
						yp[l+1]   = y;
						yp[l+w]   = y; 
						yp[l+w+1] = y;
					}
				}
			}
			break;
		case YUV420 :
			{
				unsigned char *yp,*up,*vp;
				yp = YPlane(field);
				up = UPlane(field);
				vp = VPlane(field);
				int r,g,b,y,u,v,l;

				for(i=0; i<h; i+=2)  {
					for(j=0; j<w; j+=2) {
						l = i*w +j;
						r = buffer[l];
						g = (buffer[l+1]+buffer[l+w])/2;
						b = buffer[l+w+1];
						RGB2YUV(r,g,b,y,u,v);
						yp[l]     = y;
						yp[l+1]   = y;
						yp[l+w]   = y; 
						yp[l+w+1] = y;
						*up++ = u; 
						*vp++ = v;
					}
				}
			}
			break;
		case RGB8 :
			{
				// TODO : make a real interpolation!
				unsigned char *rgb = rgbPlane(field);
				int r,g,b,l;

				for(i=0; i<h; i+=2)  {
					for(j=0; j<w; j+=2) {
						l = i*w +j;
						r = buffer[l];
						g = (buffer[l+1]+buffer[l+w])/2;
						b = buffer[l+w+1];
						rgb[l+0] = rgb[l+3+0] = 
							rgb[l+w*3+0] = rgb[l+w*3+3+0] = r;
						rgb[l+1] = rgb[l+3+1] = 
							rgb[l+w*3+1] = rgb[l+w*3+3+1] = g;
						rgb[l+2] = rgb[l+3+2] = 
							rgb[l+w*3+2] = rgb[l+w*3+3+2] = b;
					}
				}
			}
			break;
	}
}


void DDXVideo::readBayerBGGR(const unsigned char * buffer,unsigned int field)
{
	if ((video == NULL) && (std_video == NULL)) {
		rtx_error("DDXVideo::readBayerBGGR: video object not initialised");
		return ;
	}
	if (field >= fields()) {
		rtx_error("DDXVideo::readBayerBGGR: requested field does not exits");
		return ;
	}
	unsigned int i,j;
	unsigned int w = width();
	unsigned int h = height();

	switch (type()) {
		case MONO8 :
			{
				unsigned char *yp = greyPlane(field);
				int r,g,b,y,u,v,l;

				for(i=0; i<h; i+=2)  {
					for(j=0; j<w; j+=2) {
						l = i*w +j;
						b = buffer[l];
						g = (buffer[l+1]+buffer[l+w])/2;
						r = buffer[l+w+1];
						RGB2YUV(r,g,b,y,u,v);
						yp[l]     = y;
						yp[l+1]   = y;
						yp[l+w]   = y; 
						yp[l+w+1] = y;
					}
				}
			}
			break;
		case YUV420 :
			{
				unsigned char *yp,*up,*vp;
				yp = YPlane(field);
				up = UPlane(field);
				vp = VPlane(field);
				int r, g, b, y,u,v,l;

				for(i=0; i<h; i+=2)  {
					for(j=0; j<w; j+=2) {
						l = i*w +j;
						b = buffer[l];
						g = (buffer[l+1]+buffer[l+w])/2;
						r = buffer[l+w+1];
						RGB2YUV(r,g,b,y,u,v);
						yp[l]     = y;
						yp[l+1]   = y;
						yp[l+w]   = y; 
						yp[l+w+1] = y;
						*up++ = u; 
						*vp++ = v;

					}
				}
			}
			break;
		case RGB8 :
			{
				// TODO : make a real interpolation!
				unsigned char *rgb = rgbPlane(field);
				int r,g,b,l;

				for(i=0; i<h; i+=2)  {
					for(j=0; j<w; j+=2) {
						l = i*w +j;
						b = buffer[l];
						g = (buffer[l+1]+buffer[l+w])/2;
						r = buffer[l+w+1];
						rgb[l+0] = rgb[l+3+0] = 
							rgb[l+w*3+0] = rgb[l+w*3+3+0] = r;
						rgb[l+1] = rgb[l+3+1] = 
							rgb[l+w*3+1] = rgb[l+w*3+3+1] = g;
						rgb[l+2] = rgb[l+3+2] = 
							rgb[l+w*3+2] = rgb[l+w*3+3+2] = b;
					}
				}
			}
			break;
	}
}



bool DDXVideo::convertToMono(unsigned char * buffer,unsigned int field) const
{
	if ((video == NULL) && (std_video == NULL)) {
		return rtx_error_cpp("DDXVideo::convertToMono: video object not initialised");
	}
	if (buffer == NULL) 
		return rtx_error_cpp("DDXVideo::convertToMono: output buffer is NULL");
	if (field >= fields()) {
		rtx_error_cpp("DDXVideo::convertToMono: requested field does not exits");
	}
	unsigned int i,j;
	unsigned int w = width();
	unsigned int h = height();
	unsigned char * it = buffer;
	int r,g,b,y;

	switch (type()) {
		case MONO8 :
			{
				const unsigned char * gp = greyPlane(field);
				memcpy(buffer,gp,w*h);
			}
			break;
		case YUV420 :
			{
				const unsigned char *yp;
				yp = YPlane(field);
				memcpy(buffer,yp,w*h);
			}
			break;
		case RGB8 :
			{
				const unsigned char *rgb = rgbPlane(field);

				for(j=0; j<h; j++)  {
					for(i=0; i<w; i++) {
						r = rgb[0]; g = rgb[1]; b = rgb[2];
#ifdef PRESERVE_COMPATIBILTY
						RGB2Y(r,g,b,y);
#else
						RGB2Y(r,g,b,y);
#endif
						*it++ = y; 
						rgb += 3;
					}
				}
			}
			break;
	}
	
	return true;
}

bool DDXVideo::convertToYCbCr(unsigned char * buffer,unsigned int field) const
{
	if ((video == NULL) && (std_video == NULL)) {
		return rtx_error_cpp("DDXVideo::convertToYCbCr: video object not initialised");
	}
	if (buffer == NULL) 
		return rtx_error_cpp("DDXVideo::convertToYCbCr: output buffer is NULL");
	if (field >= fields()) {
		rtx_error_cpp("DDXVideo::convertToYCbCr: requested field does not exits");
	}
	unsigned int i,j;
	unsigned int w = width();
	unsigned int h = height();
	unsigned char * it = buffer;
	int r,g,b,y,u,v,l;

	switch (type()) {
		case MONO8 :
			{
				const unsigned char * gp = greyPlane(field);
				for (j=0;j<h;j++) {
					for (i=0;i<w;i++) {
						unsigned char c = gp[(h-j-1)*w + i];
						*it++ = c;
						*it++ = 128;
						*it++ = 128;
					}
				}
			}
			break;
		case YUV420 :
			{
				const unsigned char *yp,*up,*vp;
				yp = YPlane(field);
				up = UPlane(field);
				vp = VPlane(field);

				for(j=0; j<h; j++)  {
					for(i=0; i<w; i++) {
						l = (h-j-1)*w+i;
						y = yp[l]; 
						l = (h/2-j/2-1)*w/2+i/2;
						u = up[l]; v = vp[l];
#ifdef PRESERVE_COMPATIBILTY
						*it++ = y; *it++ = u; *it++ = v;
#else
						*it++ = y; *it++ = v; *it++ = u;
#endif
					}
				}

			}
			break;
		case RGB8 :
			{
				const unsigned char *rgb = rgbPlane(field);

				for(j=0; j<h; j++)  {
					for(i=0; i<w; i++) {
						r = rgb[0]; g = rgb[1]; b = rgb[2];
#ifdef PRESERVE_COMPATIBILTY
						RGB2YUV(r,g,b,y,u,v);
#else
						RGB2YUV(r,g,b,y,v,u);
#endif
						*it++ = y; *it++ = u; *it++ = v;
						rgb += 3;
					}
				}
			}
			break;
	}
	
	return true;
}


bool DDXVideo::convertToRGB(unsigned char * buffer,unsigned int field) const
{
	if ((video == NULL) && (std_video == NULL)) {
		return rtx_error_cpp("DDXVideo::convertToRGB: video object not initialised");
	}
	if (buffer == NULL) 
		return rtx_error_cpp("DDXVideo::convertToRGB: output buffer is NULL");
	if (field >= fields()) {
		rtx_error_cpp("DDXVideo::convertToRGB: requested field does not exits");
	}
	unsigned int i,j;
	unsigned int w = width();
	unsigned int h = height();
	unsigned char * it = buffer;
	int r,g,b,y,u,v,l;

	switch (type()) {
		case MONO8 :
			{
				const unsigned char * gp = greyPlane(field);
				for (j=0;j<h;j++) {
					for (i=0;i<w;i++) {
						unsigned char c = gp[(h-j-1)*w + i];
						*it++ = c;
						*it++ = c;
						*it++ = c;
					}
				}
			}
			break;
		case YUV420 :
			{
				const unsigned char *yp,*up,*vp;
				yp = YPlane(field);
				up = UPlane(field);
				vp = VPlane(field);

				for(j=0; j<h; j++)  {
					for(i=0; i<w; i++) {
						l = (h-j-1)*w+i;
						y = yp[l]; 
						l = (h/2-j/2-1)*w/2+i/2;
						u = up[l]; v = vp[l];
#ifdef PRESERVE_COMPATIBILTY
						YUV2RGB(y,u,v,r,g,b);
#else
						YUV2RGB(y,u,v,r,g,b);
#endif
						*it++ = r; *it++ = g; *it++ = b;
					}
				}
			}
			break;
		case RGB8 :
			{
				const unsigned char *rgb = rgbPlane(field);
				memcpy(buffer,rgb,w*h);
			}
			break;
	}
	
	return true;
}

bool DDXVideo::saveAsTga(const char * fname,unsigned int field) const
{
	if ((video == NULL) && (std_video == NULL)) {
		return rtx_error_cpp("DDXVideo::saveAsTga: video object not initialised");
	}
	if (fname == NULL) 
		return rtx_error_cpp("DDXVideo::saveAsTga: filename is NULL");
	if (field >= fields()) {
		rtx_error_cpp("DDXVideo::saveAsTga: requested field does not exits");
	}
	FILE * fp = fopen(fname,"w");
	if (fp == NULL) 
		return rtx_error_cpp(string("DDXVideo::saveAsTga: failed to open output file ")+fname);
	unsigned int i,j;
	unsigned int w = width();
	unsigned int h = height();
	unsigned int size = w*h*3;
	unsigned char *buffer,*it;
	buffer = (unsigned char *)malloc(size);
	assert(buffer!=NULL);

	char header[18] = { 0, 0, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0,
		(char) w, (char) (w>> 8),
		(char) h, (char) (h>> 8),
		24, 0 
	};
	fwrite(header,1,18,fp);

	// can't use convertToRGB since TGA starts from bottom left of the image.

	it = buffer;
	switch (type()) {
		case MONO8 :
			{
				const unsigned char * gp = greyPlane(field);
				for (j=0;j<h;j++) {
					for (i=0;i<w;i++) {
						unsigned char c = gp[(h-j-1)*w + i];
						*it++ = c;
						*it++ = c;
						*it++ = c;
					}
				}
			}
			break;
		case YUV420 :
			{
				const unsigned char *yp,*up,*vp;
				yp = YPlane(field);
				up = UPlane(field);
				vp = VPlane(field);
				int r,g,b,y,u,v,l;

				for(j=0; j<h; j++)  {
					for(i=0; i<w; i++) {
						l = (h-j-1)*w+i;
						y = yp[l]; 
						l = (h/2-j/2-1)*w/2+i/2;
						u = up[l]; v = vp[l];
#ifdef PRESERVE_COMPATIBILTY
						YUV2RGB(y,v,u,r,g,b);
#else
						YUV2RGB(y,u,v,r,g,b);
#endif
						*it++ = r; *it++ = g; *it++ = b;
					}
				}
			}
			break;
		case RGB8 :
			{
				const unsigned char *rgb = rgbPlane(field);
				for(j=0; j<h; j++)  {
					for(i=0; i<w; i++) {
						unsigned int l = ((h-j-1)*w+i)*3;
						*it++ = rgb[l+0];
						*it++ = rgb[l+1];
						*it++ = rgb[l+2];
					}
				}
			}
			break;
	}
	
	fwrite(buffer,1,size,fp);
	fclose(fp);
	return true;
}


