#ifndef DDX_VARIABLE_H
#define DDX_VARIABLE_H

#include <string>
#include <map>
#include <vector>
#include <rtx/error.h>

#include <ddx.h>

#define DDX_STORE_STRUCT(name,def...) \
	struct name { def }; \
	private: \
	static const char * store_struct_def_##name() {return "struct {" #def "} " #name ";";}
#define DDX_STORE_DEF(name) store_struct_def_##name()

/**
 * \class DDXVariable: access to a variable registered in the 
 * ddx store.
 * **/
class DDXVariable {
	protected :
		bool verbose;
		DDX_STORE_ITEM * itemId;
		bool direct;
		bool timedout; // triggered when a sync read timedout
		double timestamp;
		void * buffer;

		int queueTailCount;

	public:
		/** output an error with the semantic used in the DDX functions
		 * true on success, false on failure **/
		bool rtx_error_cpp(const std::string & s) const;

		DDXVariable(bool verb=true);

		/* this is dangerous, only works if var is not initialised */
		DDXVariable(const DDXVariable & var);

		/** Creates a DDXVariable from an existing store item, 
		 * takes ownership of the item, and release it in the 
		 * destructor **/
		DDXVariable(DDX_STORE_ITEM * item, bool drct=false,bool verb=true);

		/** Terminates a variable: release the internal memory buffer, and 
		 * call ddx_store_item_done **/
		void terminate();

		/** Just calls terminate **/
		virtual ~DDXVariable();

		/** Force the ddx store item. Again, the class takes ownership of the 
		 * item, and will call ddx_store_done_item in the destructor **/
		void setDdxItem(DDX_STORE_ITEM * item);

		/** Get the item, mainly for interaction with older code written in C
		 * **/
		DDX_STORE_ITEM * getId() {return itemId;}

		/** Get the variable name. Returns an empty string if the variable has 
		 * not been initialised yet **/
		std::string getName() const;

		/** Get the number of writes done in the local store, on this variable
		 * **/
		int  getCount() const {
			if (!itemId) return rtx_error("DDXVariable::getCount: variable must be initialised");
			return itemId->headerP->count; 
		}
        int getCountInQueue() const {
			if (!itemId) return rtx_error("DDXVariable::getCountInQueue: variable must be initialised");
            return itemId->headerP->count - queueTailCount;
        }

		/**
		 * Builds and return a parsed version of the variable definition.
		 * The parsed var must be freed by rtx_parse_free_var by the caller.
		 * It is a memory leak not to do so.
		 * **/
		RtxParseVar* getParsedVar();

		/**
		 * Change the direct status of the variable
		 * **/
		void setDirect(bool drct=true); 
		/**
		 * Get the direct status of the variable
		 * **/
		bool isDirect() const {return direct;}

		/**
		 * Check if this variable has been linked to a store item 
		 * **/
		bool isRegistered() const {return itemId != NULL;}
		/** 
		 * Get verbosity status
		 * **/
		bool getVerbose() const {return verbose;}
		/** 
		 * Set verbosity status
		 * **/
		void setVerbose(bool v) {verbose = v;}

		/** 
		 * Set store queue length, that is the number of item remembered
		 * in the store.
		 * **/
		bool setQueueLength( int length );
		/** 
		 * Get store queue length
		 * **/
		int  getQueueLength() const {
			if (!itemId)return rtx_error("DDXVariable::getQueueLength: variable must be initialised");
			return itemId->headerP->queueLength; 
		}
		/** 
		 * Forget about items currently in the queue
		 * **/
		bool flushqueue();

	public :
		/**
		 * Builds an returns a C-like definition of the variable 
		 * Pretty printing adds indentation and line breaks
		 * **/
		std::string definition(bool pretty=false) const;
		/**
		 * Builds an returns a C-like definition of the variable type
		 * Pretty printing adds indentation and line breaks
		 * **/
		std::string type_definition(bool pretty=false) const;
		/**
		 * Get timestamp for the last read of the variable
		 * **/
		double timeStamp() const {return timestamp;}
		/**
		 * Get the size (in bytes) of the variable in memory. If the variable
		 * has a queue in the store, it changes the space occupied in the
		 * store, but not the return value of this function.
		 * **/
		unsigned int size() const;

	public :
		/**
		 * Tells if last read failed because of timeout
		 * **/
		bool hasTimedOut() const {return timedout;}
		/**
		 * Reads the variable state from the store, into the internal buffer
		 * timearg and skipCount semantic is the same as for ddx_store_read
		 * Returns false if the read failed or if a timeout occured
		 * **/
		bool read(double timearg,int skipCount);
		/**
		 * Reads the variable state from the store, to the specified buffer
		 * timearg and skipCount semantic is the same as for ddx_store_read
		 * Returns false if the read failed or if a timeout occured
		 * **/
		bool readto(void * dest,double timearg,int skipCount);
		/**
		 * Write the variable internal buffer to the store.
		 * The write is immediate and does not force a timestamp
		 * **/
		bool write();
		/**
		 * Write the variable internal buffer to the store.
		 * The write is immediate and force a timestamp ts
		 * **/
		bool write(RtxTime & ts);
		/**
		 * Write the src buffer to the store.
		 * The write is immediate and does not force a timestamp
		 * **/
		bool writefrom(const void * src);
		/**
		 * Write the src buffer to the store.
		 * The write is immediate and force a timestamp ts
		 * **/
		bool writefrom(const void * src,/*const*/ RtxTime & ts); // should be const, but is not due to ddx implementation
	
		/**
		 * Reads the queued variable state from the store, into the internal buffer
		 * The read is immediate and not blocking. 
		 * **/
		bool readqueue(double timeout);	// read next item from the queue

		/**
		 * Retrieve a raw pointer to the internal buffer.
		 * The internal buffer maybe the shared memory object
		 * if the variable is direct.
		 * **/
		const void * rawPointer() const {return buffer;}
		/**
		 * Retrieve a raw pointer to the internal buffer.
		 * The internal buffer maybe the shared memory object
		 * if the variable is direct.
		 * **/
		void * rawPointer() {return buffer;}

		/**
		 * Reads the variable state from the store, to the specified object
		 * The template version permits to check object size, and avoid buffer
		 * overflow. It has to be prefixed by t_ to avoid ambiguities.
		 * timearg and skipCount semantic is the same as for ddx_store_read
		 * Returns false if the read failed or if a timeout occured
		 * **/
		template <class C>
		bool t_readto(C & dest,double timearg,int skipCount) {
			if (sizeof(C) < size()) {
				rtx_error("t_readto: inconsistent size");
				return false;
			}
			return readto((void*)&dest,timearg,skipCount);
		}
		
		/**
		 * Write the src object to the store.
		 * The write is immediate and force a timestamp ts
		 * The template version permits to check object size, and avoid buffer
		 * overflow. It has to be prefixed by t_ to avoid ambiguities.
		 * timearg and skipCount semantic is the same as for ddx_store_read
		 * Returns false if the read failed or if a timeout occured
		 * **/
		template <class C>
		bool t_writefrom(const C & src,/*const*/ RtxTime & ts) {
			if (sizeof(C) > size()) {
				rtx_error("t_writefrom: inconsistent size");
				return false;
			}
			return writefrom((const void*)&src,ts);
		}
	
		/**
		 * Write the src object to the store.
		 * The write is immediate and does not force a timestamp
		 * The template version permits to check object size, and avoid buffer
		 * overflow. It has to be prefixed by t_ to avoid ambiguities.
		 * **/
		template <class C>
		bool t_writefrom(const C & src) {
			if (sizeof(C) > size()) {
				rtx_error("t_writefrom: inconsistent size");
				return false;
			}
			return writefrom((const void*)&src);
		}

		/**
		 * Get a const type-checked version of the internal buffer
		 * The template version permits to check object size, and avoid buffer
		 * overflow. 
		 * **/
		template <class C>
		const C & value() const {
			if (!isRegistered()) throw std::string("DDXVariable::value(): unregistered variable");
			if (sizeof(C) != size()) throw std::string("DDXVariable::value(): Variable size mismatch");
			return *((C*)buffer);
		}

		/**
		 * Get a type-checked version of the internal buffer
		 * The template version permits to check object size, and avoid buffer
		 * overflow. 
		 * **/
		template <class C>
		C & value() {
			if (!isRegistered()) throw std::string("DDXVariable::value(): unregistered variable");
			if (sizeof(C) != size()) throw std::string("DDXVariable::value(): Variable size mismatch");
			return *((C*)buffer);
		}


		/** 
		 * Serialisation tools: This set of function is created to take the
		 * content of a DDXVariable and put it in a buffer that should be
		 * readable on an other architecture.
		 * **/

		/** 
		 * Workspace for unpacking data chunk. This should be handled as 
		 * an abstract/private type. It is made public so that an application
		 * can store the parsed variable if receiving always the same data
		 * **/
		struct UnPackingWorkspace {
			RtxParseVar * local;
			RtxParseVar * remote;

			UnPackingWorkspace() : local(NULL), remote(NULL) {}
			void reset() {
				if (local) {rtx_parse_free_var(local); local = NULL;}
				if (remote) {rtx_parse_free_var(remote); remote = NULL;}
			}
			~UnPackingWorkspace() { reset(); }
		};
		
		/** 
		 * Returns the buffer size required to pack the content of this
		 * variable.
		 * **/
		unsigned int packsize() const ;
		/**
		 * Pack the content of the internal buffer into 'buffer'. Buffer is
		 * assumed to be big enough, i.e. to contains at least packsize bytes
		 * **/
		bool pack(unsigned char * buffer) const ;
		/**
		 * Pack the content of the internal buffer into 'buffer' and impose a
		 * timestamp. Buffer is assumed to be big enough, i.e. to contains at
		 * least packsize bytes
		 * **/
		bool pack(const RtxTime & ts, unsigned char * buffer) const ;
		/**
		 * Pack the content of the src buffer into 'buffer'. Buffer is
		 * assumed to be big enough, i.e. to contains at least packsize bytes
		 * src is assumed to be a pointer on an object consistent with the 
		 * variable associated with this object.
		 * **/
		bool pack(const void * src, unsigned char * buffer) const ;
		/**
		 * Pack the content of the src buffer into 'buffer' and impose a
		 * timestamp. Buffer is assumed to be big enough, i.e. to contains at
		 * least packsize bytes src is assumed to be a pointer on an object
		 * consistent with the variable associated with this object.
		 * **/
		bool pack(const void * src, const RtxTime & ts, unsigned char * buffer) const ;

		/**
		 * Unpack the content of the 'buffer' into the internal buffer. Buffer is
		 * assumed to be big enough, i.e. to contains at least packsize bytes
		 * The timestamp ts is updated with the timestamp contained in buffer.
		 * UnPackingWorkspace ws is used to store the parsed representation of
		 * the variable. Behavious is undefined if ws is tampered with between
		 * two calls of unpack.
		 * **/
		bool unpack(RtxTime & ts, UnPackingWorkspace & ws, unsigned char * buffer) ;
		/**
		 * Unpack the content of the 'buffer' into the internal buffer. Buffer is
		 * assumed to be big enough, i.e. to contains at least packsize bytes
		 * UnPackingWorkspace ws is used to store the parsed representation of
		 * the variable. Behavious is undefined if ws is tampered with between
		 * two calls of unpack.
		 * **/
		bool unpack(UnPackingWorkspace & ws, unsigned char * buffer);
		/**
		 * Unpack the content of the 'buffer' into the 'dest' buffer. Buffer is
		 * assumed to be big enough, i.e. to contains at least packsize bytes
		 * UnPackingWorkspace ws is used to store the parsed representation of
		 * the variable. Behavious is undefined if ws is tampered with between
		 * two calls of unpack.
		 * **/
		bool unpack(void * dest, RtxTime & ts, UnPackingWorkspace & ws, unsigned char * buffer) const ;
		/**
		 * Unpack the content of the 'buffer' into the 'dest' buffer. Buffer is
		 * assumed to be big enough, i.e. to contains at least packsize bytes
		 * The timestamp ts is updated with the timestamp contained in buffer.
		 * UnPackingWorkspace ws is used to store the parsed representation of
		 * the variable. Behavious is undefined if ws is tampered with between
		 * two calls of unpack.
		 * **/
		bool unpack(void * dest, UnPackingWorkspace & ws, unsigned char * buffer) const ;

		/** Template version for size checking **/

		/**
		 * Pack the content of the src object into 'buffer' and impose a
		 * timestamp. Buffer is assumed to be big enough, i.e. to contains at
		 * least packsize bytes src is assumed to be an object consistent with
		 * the variable associated with this object. The template version only
		 * check that the size if consistent. 
		 * **/
		template <class C>
		bool t_pack(const C & src, const RtxTime & ts, unsigned char * buffer) const  {
			if (sizeof(C) > packsize()) return false;
			return pack(&src,ts,buffer);
		}
		
		/**
		 * Pack the content of the src object into 'buffer'. Buffer is
		 * assumed to be big enough, i.e. to contains at least packsize bytes
		 * src is assumed to be an object consistent with the 
		 * variable associated with this object. The template version only
		 * check that the size if consistent. 
		 * **/
		template <class C>
		bool t_pack(const C & src, unsigned char * buffer) const  {
			if (sizeof(C) > packsize()) return false;
			return pack(&src,buffer);
		}

		/**
		 * Unpack the content of the 'buffer' into the 'dest' object. Buffer is
		 * assumed to be big enough, i.e. to contains at least packsize bytes
		 * The timestamp ts is updated with the timestamp contained in buffer.
		 * UnPackingWorkspace ws is used to store the parsed representation of
		 * the variable. Behavious is undefined if ws is tampered with between
		 * two calls of unpack.
		 * **/
		template <class C>
		bool t_unpack(C & dest,RtxTime & ts, UnPackingWorkspace & ws, unsigned char * buffer) const  {
			if (sizeof(C) < packsize()) return false;
			return unpack(&dest,ts,ws,buffer);
		}

		/**
		 * Unpack the content of the 'buffer' into the 'dest' object. Buffer is
		 * assumed to be big enough, i.e. to contains at least packsize bytes
		 * UnPackingWorkspace ws is used to store the parsed representation of
		 * the variable. Behavious is undefined if ws is tampered with between
		 * two calls of unpack.
		 * **/
		template <class C>
		bool t_unpack(C & dest,UnPackingWorkspace & ws, unsigned char * buffer) const  {
			if (sizeof(C) < packsize()) return false;
			return unpack(&dest,ws,buffer);
		}

	public:
		/**
		 * \class DDXValue: class representing the value of an object in the
		 * store. This class is not safe to store since it relies on a pointer
		 * to the DDXVariable own pointer. For this reason, it has no public 
		 * constructor, can only be created by a DDXVariable, and used
		 * immediately.
		 *  **/
		class DDXValue 
		{
			public:
				/** Friend is not even strong enough to tell how linked these
				 * two classes are **/
				friend class DDXVariable;
			protected:
				/** parsedMap is a static store of RtxParseVar used for 
				 * garbage collecting. Because all functions below returns
				 * DDXValue objects that share the same RtxParseVar, we have to
				 * keep a record of the number of references to know when to
				 * delete it.
				 * **/
				static std::map<RtxParseVar*,unsigned int> parsedMap;

				/** Buffer associated with the current DDXValue. This is what
				 * will be read if one of the "asBlah" function below is called
				 * **/
				void * buffer;

				/** 
				 * Parsed representation of the value, not stored in parsedMap
				 * **/
				RtxParseVar *parsed;
				/** 
				 * Parsed representation of the toplevel DDXVariable value, this is what 
				 * is stored in parsedMap, and eventually freed in the
				 * DDXVariable destructor
				 * **/
				RtxParseVar *parent;
				/**
				 * Array of dimension of the current value. The "asBlah"
				 * function below can only be called if this array is empty,
				 * that is if the value is unidimensional
				 * **/
				std::vector<unsigned int> dim;

				/**
				 * Constructor for a top level DDXValue object. Item is not
				 * stored in the DDXValue object, but used to generate a parsed 
				 * representation of the variable 
				 * */
				DDXValue(DDX_STORE_ITEM *item, void * buffer);
				/**
				 * Constructor for a DDXValue object which is an item in a
				 * multi-dimensional DDXValue object V. Range checking is
				 * performed on index
				 * */
				DDXValue(const DDXValue & V, unsigned int index) throw (std::exception);
				/**
				 * Constructor for a DDXValue object which is field in a
				 * structured DDXValue object V. 
				 * */
				DDXValue(const DDXValue & V, const std::string & field) throw (std::exception);

				/**
				 * Copy constructor. Only here to increment reference count
				 * **/
				DDXValue(const DDXValue & V);
			public:
				/**
				 * The destructor is public. Release the memory used by the
				 * RtxParseVar if the reference count comes to zero
				 * **/
				~DDXValue();

				/**
				 * Returns the value associated with this object as a double if
				 * appropriate. Throw an exception otherwise
				 * */
				double asDouble() const throw (std::exception);
				/**
				 * Returns the value associated with this object as a float if
				 * appropriate. Throw an exception otherwise
				 * */
				float asFloat() const throw (std::exception);
				/**
				 * Returns the value associated with this object as a long if
				 * appropriate. Throw an exception otherwise
				 * */
				long asLong() const throw (std::exception);
				/**
				 * Returns the value associated with this object as a int if
				 * appropriate. Throw an exception otherwise
				 * */
				int asInt() const throw (std::exception);
				/**
				 * Returns the value associated with this object as a short if
				 * appropriate. Throw an exception otherwise
				 * */
				short asShort() const throw (std::exception);
				/**
				 * Returns the value associated with this object as a char if
				 * appropriate. Throw an exception otherwise
				 * */
				char asChar() const throw (std::exception);

				/**
				 * Affects a double to the value if appropriate. d is casted to
				 * the appropriate type before affectation. Throw an exception
				 * if the affectation is not possible. Returns d.
				 * **/
				double operator=(double d) throw (std::exception);
				/**
				 * Affects a double to the value if appropriate. d is casted to
				 * the appropriate type before affectation. Throw an exception
				 * if the affectation is not possible. Returns d.
				 * **/
				float operator=(float d) throw (std::exception);
				/**
				 * Affects a double to the value if appropriate. d is casted to
				 * the appropriate type before affectation. Throw an exception
				 * if the affectation is not possible. Returns d.
				 * **/
				long operator=(long d) throw (std::exception);
				/**
				 * Affects a double to the value if appropriate. d is casted to
				 * the appropriate type before affectation. Throw an exception
				 * if the affectation is not possible. Returns d.
				 * **/
				int operator=(int d) throw (std::exception);
				/**
				 * Affects a double to the value if appropriate. d is casted to
				 * the appropriate type before affectation. Throw an exception
				 * if the affectation is not possible. Returns d.
				 * **/
				short operator=(short d) throw (std::exception);
				/**
				 * Affects a double to the value if appropriate. d is casted to
				 * the appropriate type before affectation. Throw an exception
				 * if the affectation is not possible. Returns d.
				 * **/
				char operator=(char d) throw (std::exception);

			public:
				/**
				 * Returns a DDXValue object which is an item in this
				 * multi-dimensional DDXValue object. Range checking is
				 * performed on index
				 * */
				DDXValue operator[](unsigned int i) throw (std::exception); 
				/**
				 * Returns a DDXValue object which is a field in this
				 * structured DDXValue object. 
				 * */
				DDXValue field(const std::string & s) throw (std::exception); 

				/**
				 * Returns a DDXValue which represents the string s.
				 * Returns itself of s is empty. Example of s includes:
				 * ".pls[1].range[2]", ".fieldx", "[32]"
				 * Note that a field in a structure is accessed by "." whereas 
				 * a item in an array is accessed by starting s by "[". Syntax
				 * errors or inconsistencies throw an exception
				 * */
				DDXValue parse(const std::string & s = "") throw (std::exception);
		};

		/**
		 * Use DDXValue::parse to access the item in this DDXVariable
		 * represented by s.
		 * **/
		DDXValue get(const std::string & s = "") {
			return DDXValue(itemId,buffer).parse(s);
		}
			
};




#endif // DDX_VARIABLE_H
