#include <string>
#include <assert.h>
#include <stdlib.h>
#include <string.h>
#include "DDXVariable.h"

extern "C" {
#include <rtx/error.h>
}

using namespace std;

/** output an error with the semantic used in the DDX functions
 * true on success, false on failure **/
bool DDXVariable::rtx_error_cpp(const std::string & s) const{
	if (verbose) rtx_error_flush(const_cast<char *>(s.c_str()));
	return false;
}


DDXVariable::DDXVariable(bool verb)
{
	//printf("#Creating (1) %p\n",this);
	verbose = verb;
	itemId = NULL;
	direct = false;
	buffer = NULL;
	timestamp = 0.0;
	queueTailCount = 0;
}

/** Creates a DDXVariable from an existing store item, 
 * takes ownership of the item, and release it in the 
 * destructor **/
DDXVariable::DDXVariable(DDX_STORE_ITEM * item, bool drct, bool verb)
{
	//printf("#Creating (2) %p\n",this);
	verbose = verb;
	itemId = item;
	buffer = NULL;
	setDirect(drct);
	timestamp = 0.0;
	queueTailCount = 0;
}

/* this is dangerous, only works if var is not initialised */
DDXVariable::DDXVariable(const DDXVariable & var)
{
	if (itemId || buffer) {
		bool DDXVariable_Copy_Allowed = false;
		assert(DDXVariable_Copy_Allowed);
	}

	buffer = NULL;
	itemId = NULL;
	verbose = var.verbose;
	direct = var.direct;
	timestamp = var.timestamp;
	timedout = var.timedout;
	queueTailCount = var.queueTailCount;
}

/** Terminates a variable: release the internal memory buffer, and 
 * call ddx_store_item_done **/
void DDXVariable::terminate()
{
	//printf("#Terminating %p item = %p\n",this,itemId);
	if (!direct) {
		free(buffer);
	}
	buffer = NULL;
	if (itemId != NULL)
		ddx_store_done_item(itemId);
	itemId = NULL;
	direct = false;
}

DDXVariable::~DDXVariable()
{
	terminate();
}

/** Force the ddx store item. Again, the class takes ownership of the 
 * item, and will call ddx_store_done_item in the destructor **/
void DDXVariable::setDdxItem(DDX_STORE_ITEM * item)
{
	terminate();
	itemId = item;
	setDirect(direct);
	timestamp = 0.0;
}

/** Get the variable name. Returns an empty string if the variable has 
 * not been initialised yet **/
std::string DDXVariable::getName() const 
{
	if (!isRegistered()) return string();
	return string(itemId->varName);
}
	

/**
 * Change the direct status of the variable
 * */
void DDXVariable::setDirect(bool drct)
{
  // check if variable is a queue
  if (drct && (this->getQueueLength() > 0) ) {
    rtx_error_cpp("DDXVariable::setDirect variable is a queue. Cannot set direct");
    drct = false;
  }
	if (!direct) free(buffer);
	buffer = NULL;
	direct = drct;
	if (itemId == NULL) return;

	if (direct) {
		buffer = ddx_store_var_pointer(itemId);
	} else {
		buffer = malloc(itemId->varSize);
	}	
}

/** 
 * Set store queue length, that is the number of item remembered
 * in the store.
 * **/
bool DDXVariable::setQueueLength(int length)
{
  if (this->itemId == NULL) return false;
  
  return  ddx_store_item_set_queue(this->itemId, length) == 0;
}


/**
 * Builds an returns a C-like definition of the variable 
 * Pretty printing adds indentation and line breaks
 * **/
std::string DDXVariable::definition(bool pretty) const 
{
	if (itemId == NULL) return string();
	
	char * def=ddx_store_item_get_definition(itemId, pretty?1:0);
	string sdef(def);
	free(def);
	
	return sdef;
}

/**
 * Builds an returns a C-like definition of the variable type
 * Pretty printing adds indentation and line breaks
 * **/
std::string DDXVariable::type_definition(bool pretty) const 
{
	if (itemId == NULL) return string();
	
	char * def=ddx_store_item_get_type_definition(itemId, pretty?1:0);
	string sdef(def);
	free(def);
	
	return sdef;
}

/**
 * Get the size (in bytes) of the variable in memory. If the variable
 * has a queue in the store, it changes the space occupied in the
 * store, but not the return value of this function.
 * **/
unsigned int DDXVariable::size() const 
{
	if (itemId == NULL) return 0;
	return itemId->varSize;
}

/**
 * Reads the variable state from the store, into the internal buffer
 * timearg and skipCount semantic is the same as for ddx_store_read
 * Returns false if the read failed or if a timeout occured
 * **/
bool DDXVariable::read(double timearg, int skipCount)
{
	int r;
	RtxTime ts;
	if (itemId == NULL) return rtx_error_cpp("DDXVariable::read variable not registered");
	if (direct)
		r = ddx_store_read_direct(itemId,&ts,timearg,skipCount);
	else 
		r = ddx_store_read(itemId,buffer,&ts,timearg,skipCount);
	timedout = (r>0);
	timestamp = rtx_time_to_double(&ts);
	if (r<0) return rtx_error_cpp("DDXVariable::read: ddx_store_read failed");
	return r==0;
}

/**
 * Reads the variable state from the store, to the specified buffer
 * timearg and skipCount semantic is the same as for ddx_store_read
 * Returns false if the read failed or if a timeout occured
 * **/
bool DDXVariable::readto(void * dest, double timearg, int skipCount)
{
	int r;
	RtxTime ts;
	if (itemId == NULL) return rtx_error_cpp("DDXVariable::readto variable not registered");
	if (direct) {
		memcpy(dest,buffer,size());
		r = ddx_store_read_direct(itemId,&ts,timearg,skipCount);
	} else {
		r = ddx_store_read(itemId,dest,&ts,timearg,skipCount);
	}
	if (r<0) return rtx_error_cpp("DDXVariable::readto: ddx_store_read failed");
	timedout = (r>0);
	timestamp = rtx_time_to_double(&ts);
	return r==0;
}

/** 
 * Forget about items currently in the queue
 * **/
bool DDXVariable::flushqueue()
{
  if (itemId == NULL) return rtx_error_cpp("DDXVariable::flushqueue variable not registered");
  
  if (getQueueLength() <= 0) return rtx_error_cpp("DDXVariable::flushqueue variable not a queue");
  
  // flush the queue by setting the tail count to the head count
  this->queueTailCount = itemId->headerP->count;
  return true;
}

/**
 * Reads the queued variable state from the store, into the internal buffer
 * The read is immediate and not blocking. 
 * **/
bool DDXVariable::readqueue(double timearg)
{
	int r;
	RtxTime ts;
	if (itemId == NULL) return rtx_error_cpp("DDXVariable::readqueue variable not registered");
	if (direct)
	  return rtx_error_cpp("DDXVariable::readqueue variable cannot be direct");
	else 
	  r = ddx_store_queue_read(itemId,buffer,&ts,timearg, &this->queueTailCount);
	timedout = (r==-2);
	timestamp = rtx_time_to_double(&ts);
	if (r==-1) return rtx_error_cpp("DDXVariable::read: ddx_store_read failed");
	return r>=0;
}


/**
 * Write the variable internal buffer to the store.
 * The write is immediate and force a timestamp ts
 * **/
bool DDXVariable::write(RtxTime & ts)
{
	int r;
	if (itemId == NULL) return rtx_error_cpp("DDXVariable::write variable not registered");
	if (direct)
		r = ddx_store_write_direct(itemId,&ts);
	else
		r = ddx_store_write(itemId,buffer,&ts);
	if (r<0) return rtx_error_cpp("DDXVariable::write: ddx_store_write failed");

	return r==0;
}

/**
 * Write the variable internal buffer to the store.
 * The write is immediate and does not force a timestamp
 * **/
bool DDXVariable::write()
{
	int r;
	if (itemId == NULL) return rtx_error_cpp("DDXVariable::write variable not registered");
	if (direct)
		r = ddx_store_write_direct(itemId,NULL);
	else
		r = ddx_store_write(itemId,buffer,NULL);
	if (r<0) return rtx_error_cpp("DDXVariable::write: ddx_store_write failed");

	return r==0;
}

/**
 * Write the src buffer to the store.
 * The write is immediate and force a timestamp ts
 * **/
bool DDXVariable::writefrom(const void * src,RtxTime & ts)
{
	int r;
	if (itemId == NULL) return rtx_error_cpp("DDXVariable::writefrom variable not registered");
	memcpy(buffer,src,itemId->varSize);
	if (direct)
		r = ddx_store_write_direct(itemId, &ts);
	else
		r = ddx_store_write(itemId,buffer, &ts);
	if (r<0) return rtx_error_cpp("DDXVariable::writefrom: ddx_store_write failed");

	return r==0;
}

/**
 * Write the src buffer to the store.
 * The write is immediate and does not force a timestamp
 * **/
bool DDXVariable::writefrom(const void * src)
{
	int r;
	if (itemId == NULL) return rtx_error_cpp("DDXVariable::writefrom variable not registered");
	memcpy(buffer,src,itemId->varSize);
	if (direct)
		r = ddx_store_write_direct(itemId, NULL);
	else
		r = ddx_store_write(itemId,buffer, NULL);
	if (r<0) return rtx_error_cpp("DDXVariable::writefrom: ddx_store_write failed");

	return r==0;
}


/** 
 * Returns the buffer size required to pack the content of this
 * variable.
 * **/
unsigned int DDXVariable::packsize() const
{
	return ddx_store_item_get_packsize(itemId);
}

/**
 * Pack the content of the internal buffer into 'buffer'. Buffer is
 * assumed to be big enough, i.e. to contains at least packsize bytes
 * **/
bool DDXVariable::pack(unsigned char * dbuffer) const
{
	int r;
#if 0
	printf("Packing %d bytes\n",size());
	for (r=0;r<size();r++) printf("%02X ",((unsigned char*)buffer)[r]);
	printf("\n");
#endif
	r = ddx_store_item_pack(itemId, rawPointer(), NULL, dbuffer);
	if (r<0) return rtx_error_cpp("DDXVariable::pack: ddx_store_item_pack failed");
	return r == 0;
}

/**
 * Pack the content of the internal buffer into 'buffer' and impose a
 * timestamp. Buffer is assumed to be big enough, i.e. to contains at
 * least packsize bytes
 * **/
bool DDXVariable::pack(const RtxTime & ts, unsigned char * dbuffer) const 
{
	int r;
	r = ddx_store_item_pack(itemId, rawPointer(), &ts, dbuffer);
	if (r<0) return rtx_error_cpp("DDXVariable::pack: ddx_store_item_pack failed");
	return r == 0;
}

/**
 * Pack the content of the src buffer into 'buffer'. Buffer is
 * assumed to be big enough, i.e. to contains at least packsize bytes
 * src is assumed to be a pointer on an object consistent with the 
 * variable associated with this object.
 * **/
bool DDXVariable::pack(const void * src, unsigned char * dbuffer) const 
{
	int r;
	r = ddx_store_item_pack(itemId, src, NULL, dbuffer);
	if (r<0) return rtx_error_cpp("DDXVariable::pack: ddx_store_item_pack failed");
	return r == 0;
}

/**
 * Pack the content of the src buffer into 'buffer' and impose a
 * timestamp. Buffer is assumed to be big enough, i.e. to contains at
 * least packsize bytes src is assumed to be a pointer on an object
 * consistent with the variable associated with this object.
 * **/
bool DDXVariable::pack(const void * src, const RtxTime & ts, unsigned char * dbuffer) const 
{
	int r;
	r = ddx_store_item_pack(itemId, src, &ts, dbuffer);
	if (r<0) return rtx_error_cpp("DDXVariable::pack: ddx_store_item_pack failed");
	return r == 0;
}

/**
 * Unpack the content of the 'buffer' into the 'dest' buffer. Buffer is
 * assumed to be big enough, i.e. to contains at least packsize bytes
 * UnPackingWorkspace ws is used to store the parsed representation of
 * the variable. Behavious is undefined if ws is tampered with between
 * two calls of unpack.
 * **/
bool DDXVariable::unpack(void * dest, RtxTime & ts, UnPackingWorkspace & ws, unsigned char * sbuffer) const 
{
	int r;
	r = ddx_store_item_unpack(itemId, dest, &ts, &ws.local, &ws.remote,  sbuffer);
	if (r<0) return rtx_error_cpp("DDXVariable::unpack: ddx_store_item_unpack failed");
	return r == 0;
}

/**
 * Unpack the content of the 'buffer' into the 'dest' buffer. Buffer is
 * assumed to be big enough, i.e. to contains at least packsize bytes
 * The timestamp ts is updated with the timestamp contained in buffer.
 * UnPackingWorkspace ws is used to store the parsed representation of
 * the variable. Behavious is undefined if ws is tampered with between
 * two calls of unpack.
 * **/
bool DDXVariable::unpack(void * dest, UnPackingWorkspace & ws, unsigned char * sbuffer) const 
{
	int r;
	r = ddx_store_item_unpack(itemId, dest, NULL, &ws.local, &ws.remote,  sbuffer);
	if (r<0) return rtx_error_cpp("DDXVariable::unpack: ddx_store_item_unpack failed");
	return r == 0;
}

/**
 * Unpack the content of the 'buffer' into the internal buffer. Buffer is
 * assumed to be big enough, i.e. to contains at least packsize bytes
 * The timestamp ts is updated with the timestamp contained in buffer.
 * UnPackingWorkspace ws is used to store the parsed representation of
 * the variable. Behavious is undefined if ws is tampered with between
 * two calls of unpack.
 * **/
bool DDXVariable::unpack(RtxTime & ts, UnPackingWorkspace & ws, unsigned char * sbuffer) 
{
	int r;
	r = ddx_store_item_unpack(itemId, rawPointer(), &ts, &ws.local, &ws.remote,  sbuffer);
	if (r<0) return rtx_error_cpp("DDXVariable::unpack: ddx_store_item_unpack failed");
	return r == 0;
}

/**
 * Unpack the content of the 'buffer' into the internal buffer. Buffer is
 * assumed to be big enough, i.e. to contains at least packsize bytes
 * UnPackingWorkspace ws is used to store the parsed representation of
 * the variable. Behavious is undefined if ws is tampered with between
 * two calls of unpack.
 * **/
bool DDXVariable::unpack(UnPackingWorkspace & ws, unsigned char * sbuffer) 
{
	int r;
	r = ddx_store_item_unpack(itemId, rawPointer(), NULL, &ws.local, &ws.remote,  sbuffer);
	if (r<0) return rtx_error_cpp("DDXVariable::unpack: ddx_store_item_unpack failed");
	return r == 0;
}

/**
 * Builds and return a parsed version of the variable definition.
 * The parsed var must be freed by rtx_parse_free_var by the caller.
 * It is a memory leak not to do so.
 * **/
RtxParseVar * DDXVariable::getParsedVar()
{
	if (!isRegistered()) {
		if (verbose) rtx_error("DDXVariable::getParsedVar: variable no registered");
		return NULL;
	}
	RtxParseVar *v = ddx_store_item_get_parsed_var(itemId);
	if (v==NULL) rtx_error("DDXVariable::getParsedVar: ddx_store_item_get_parsed_var failed");
	
	return v;
}


//////////////////////////////////////////////////////
//
// DDXValue function
//

/**
 * \class DDXValueException: small string based exception class 
 * thrown out by all the DDXValue functions
 * **/
class DDXValueException : public std::exception
{
	protected:
		std::string myself;
	public:
		DDXValueException(const std::string & s) : myself(s) {}
		virtual ~DDXValueException() throw () {}

		virtual const char * what() const throw () {
			return myself.c_str();
		}
};


/** parsedMap is a static store of RtxParseVar used for 
 * garbage collecting. Because all functions below returns
 * DDXValue objects that share the same RtxParseVar, we have to
 * keep a record of the number of references to know when to
 * delete it.
 * **/
std::map<RtxParseVar*,unsigned int> DDXVariable::DDXValue::parsedMap;
/** mutex protecting the parsedMap static variable, may be overkill **/
static pthread_mutex_t parsedMapMtx = PTHREAD_MUTEX_INITIALIZER;


/**
 * Constructor for a top level DDXValue object. Item is not
 * stored in the DDXValue object, but used to generate a parsed 
 * representation of the variable 
 * */
DDXVariable::DDXValue::DDXValue(DDX_STORE_ITEM *item, void * buf)
{
	unsigned int i;
	buffer = buf;
	parent = parsed = ddx_store_item_get_parsed_var(item);
	pthread_mutex_lock(&parsedMapMtx);
	parsedMap[parent] = 1;
	pthread_mutex_unlock(&parsedMapMtx);
	for (i=0;i<(unsigned int)parsed->dim;i++) {
		dim.push_back(parsed->arrayDim[i]);
	}
}

/**
 * Constructor for a DDXValue object which is an item in a
 * multi-dimensional DDXValue object V. Range checking is
 * performed on index
 * */
DDXVariable::DDXValue::DDXValue(const DDXValue & V, unsigned int index) throw (std::exception)
{
	unsigned int i,cumsize;
	parent = V.parent;
	parsed = V.parsed;
	if (V.dim.size() == 0) {
		throw DDXValueException("DDXValue::item can't used indexed access with unidimensional var");
	}
	if (index >= V.dim[0]) {
		// printf("Index %d out of range [0,%d[\n",index,parsed->arrayDim[0]);
		throw DDXValueException("DDXValue::item index out of range");
	}
	cumsize = parsed->elemSize;
	for (i=1;i<V.dim.size();i++) {
		cumsize *= V.dim[i];
		dim.push_back(V.dim[i]);
	}
	buffer = (void*)(((char*)V.buffer) + index * cumsize);
	pthread_mutex_lock(&parsedMapMtx);
	parsedMap[parent] += 1;
	pthread_mutex_unlock(&parsedMapMtx);
}

/**
 * Constructor for a DDXValue object which is field in a
 * structured DDXValue object V. 
 * */
DDXVariable::DDXValue::DDXValue(const DDXValue & V, const std::string & field) throw (std::exception)
{
	RtxParseVar *it = NULL; 
	unsigned int i;
	bool found = false;
	parent = V.parent;
	if (V.dim.size() != 0) {
		throw DDXValueException("DDXValue::field can't access field of a multi-dimensional variable");
	}
	if (V.parsed->type != rtx_struct_t) {
		throw DDXValueException("DDXValue::field can't access field of a non struct variable");
	}
	it = V.parsed->subVar;
	if (!it) {
		throw DDXValueException("DDXValue::field inconsistent parsed var");
	}
	while (!found && it) {
		if (strcmp(it->name,field.c_str())==0) {
			found = true;
			parsed = it;
		}
		it = it->next;
	}
	if (!found) {
		// printf("Field %s not found in %s\n",field.c_str(),V.parsed->name);
		throw DDXValueException("DDXValue::item inexistent field");
	}
	for (i=0;i<(unsigned int)parsed->dim;i++) {
		dim.push_back(parsed->arrayDim[i]);
	}
	buffer = (void*)((char*)V.buffer + parsed->offset);
	pthread_mutex_lock(&parsedMapMtx);
	parsedMap[parent] += 1;
	pthread_mutex_unlock(&parsedMapMtx);
}

/**
 * Copy constructor. Only here to increment reference count
 * **/
DDXVariable::DDXValue::DDXValue(const DDXValue & V)
{
	buffer = V.buffer;
	parsed = V.parsed;
	parent = V.parent;
	dim = V.dim;
	pthread_mutex_lock(&parsedMapMtx);
	parsedMap[parent] += 1;
	pthread_mutex_unlock(&parsedMapMtx);
}

/**
 * The destructor is public. Release the memory used by the
 * RtxParseVar if the reference count comes to zero
 * **/
DDXVariable::DDXValue::~DDXValue()
{
	pthread_mutex_lock(&parsedMapMtx);
	std::map<RtxParseVar*,unsigned int>::iterator it = parsedMap.find(parent);
	it->second -= 1;
	if (it->second == 0) {
		rtx_parse_free_var(parent);
		parsedMap.erase(it);
	}
	pthread_mutex_unlock(&parsedMapMtx);
}

/**
 * Generic templated function to retrieve a value from the buffer using
 * the parsed variable types. 
 * **/
template <class C>
static C parsedAs(RtxParseVar* parsed, void * buffer) throw (std::exception)
{
	if (parsed->type == rtx_struct_t) {
		throw DDXValueException("DDXValue: can't access the value of a struct variable, use field");
	}
	switch(parsed->type) {
		case rtx_double_t: 
			return (C)(*(double*)buffer);
		case rtx_float_t:  
			return (C)(*(float*)buffer);
		case rtx_long_t:   
			return (C)(*(long*)buffer);
		case rtx_int_t:    
			return (C)(*(int*)buffer);
		case rtx_short_t:  
			return (C)(*(short*)buffer);
		case rtx_char_t:   
			return (C)(*(char*)buffer);
		default:
			throw DDXValueException("DDXValue: unknown variable type");
	}
	
	// not reachable
	return 0;
}

/**
 * Returns the value associated with this object as a double if
 * appropriate. Throw an exception otherwise
 * */
double DDXVariable::DDXValue::asDouble() const throw (std::exception)
{
	if (dim.size() != 0) {
		throw DDXValueException("DDXValue: can't access the value of a multi-dimensional variable, use []");
	}
	return parsedAs<double>(parsed,buffer);
}

/**
 * Returns the value associated with this object as a float if
 * appropriate. Throw an exception otherwise
 * */
float DDXVariable::DDXValue::asFloat() const throw (std::exception)
{
	if (dim.size() != 0) {
		throw DDXValueException("DDXValue: can't access the value of a multi-dimensional variable, use []");
	}
	return parsedAs<float>(parsed,buffer);
}

/**
 * Returns the value associated with this object as a long if
 * appropriate. Throw an exception otherwise
 * */
long DDXVariable::DDXValue::asLong() const throw (std::exception)
{
	if (dim.size() != 0) {
		throw DDXValueException("DDXValue: can't access the value of a multi-dimensional variable, use []");
	}
	return parsedAs<long>(parsed,buffer);
}

/**
 * Returns the value associated with this object as a int if
 * appropriate. Throw an exception otherwise
 * */
int DDXVariable::DDXValue::asInt() const throw (std::exception)
{
	if (dim.size() != 0) {
		throw DDXValueException("DDXValue: can't access the value of a multi-dimensional variable, use []");
	}
	return parsedAs<int>(parsed,buffer);
}

/**
 * Returns the value associated with this object as a short if
 * appropriate. Throw an exception otherwise
 * */
short DDXVariable::DDXValue::asShort() const throw (std::exception)
{
	if (dim.size() != 0) {
		throw DDXValueException("DDXValue: can't access the value of a multi-dimensional variable, use []");
	}
	return parsedAs<short>(parsed,buffer);
}

/**
 * Returns the value associated with this object as a char if
 * appropriate. Throw an exception otherwise
 * */
char DDXVariable::DDXValue::asChar() const throw (std::exception)
{
	if (dim.size() != 0) {
		throw DDXValueException("DDXValue: can't access the value of a multi-dimensional variable, use []");
	}
	return parsedAs<char>(parsed,buffer);
}

/**
 * Generic templated function to affect a (casted) value to the buffer using
 * the parsed variable types. 
 * **/
template <class C>
static C affectAs(RtxParseVar* parsed, void * buffer, C val) throw (std::exception)
{
	if (parsed->type == rtx_struct_t) {
		throw DDXValueException("DDXValue: can't affect the value of a struct variable, use field");
	}
	switch(parsed->type) {
		case rtx_double_t: 
			(*(double*)buffer) = (double)val;
			break;
		case rtx_float_t:  
			(*(float*)buffer) = (float)val;
			break;
		case rtx_long_t:   
			(*(long*)buffer) = (long)val;
			break;
		case rtx_int_t:    
			(*(int*)buffer) = (int)val;
			break;
		case rtx_short_t:  
			(*(short*)buffer) = (short)val;
			break;
		case rtx_char_t:   
			(*(char*)buffer) = (char)val;
			break;
		default:
			throw DDXValueException("DDXValue: unknown variable type");
	}
	
	return val;
}

/**
 * Affects a double to the value if appropriate. d is casted to
 * the appropriate type before affectation. Throw an exception
 * if the affectation is not possible. Returns d.
 * **/
double DDXVariable::DDXValue::operator=(double d) throw (std::exception)
{
	if (dim.size() != 0) {
		throw DDXValueException("DDXValue: can't access the value of a multi-dimensional variable, use []");
	}
	return affectAs<double>(parsed,buffer,d);
}

/**
 * Affects a float to the value if appropriate. d is casted to
 * the appropriate type before affectation. Throw an exception
 * if the affectation is not possible. Returns d.
 * **/
float DDXVariable::DDXValue::operator=(float d) throw (std::exception)
{
	if (dim.size() != 0) {
		throw DDXValueException("DDXValue: can't access the value of a multi-dimensional variable, use []");
	}
	return affectAs<float>(parsed,buffer,d);
}

/**
 * Affects a long to the value if appropriate. d is casted to
 * the appropriate type before affectation. Throw an exception
 * if the affectation is not possible. Returns d.
 * **/
long DDXVariable::DDXValue::operator=(long d) throw (std::exception)
{
	if (dim.size() != 0) {
		throw DDXValueException("DDXValue: can't access the value of a multi-dimensional variable, use []");
	}
	return affectAs<long>(parsed,buffer,d);
}

/**
 * Affects a int to the value if appropriate. d is casted to
 * the appropriate type before affectation. Throw an exception
 * if the affectation is not possible. Returns d.
 * **/
int DDXVariable::DDXValue::operator=(int d) throw (std::exception)
{
	if (dim.size() != 0) {
		throw DDXValueException("DDXValue: can't access the value of a multi-dimensional variable, use []");
	}
	return affectAs<int>(parsed,buffer,d);
}

/**
 * Affects a short to the value if appropriate. d is casted to
 * the appropriate type before affectation. Throw an exception
 * if the affectation is not possible. Returns d.
 * **/
short DDXVariable::DDXValue::operator=(short d) throw (std::exception)
{
	if (dim.size() != 0) {
		throw DDXValueException("DDXValue: can't access the value of a multi-dimensional variable, use []");
	}
	return affectAs<short>(parsed,buffer,d);
}

/**
 * Affects a char to the value if appropriate. d is casted to
 * the appropriate type before affectation. Throw an exception
 * if the affectation is not possible. Returns d.
 * **/
char DDXVariable::DDXValue::operator=(char d) throw (std::exception)
{
	if (dim.size() != 0) {
		throw DDXValueException("DDXValue: can't access the value of a multi-dimensional variable, use []");
	}
	return affectAs<char>(parsed,buffer,d);
}

/**
 * Returns a DDXValue object which is an item in this
 * multi-dimensional DDXValue object. Range checking is
 * performed on index
 * */
DDXVariable::DDXValue DDXVariable::DDXValue::operator[](unsigned int i) throw (std::exception)
{
	return DDXValue(*this,i);
}

/**
 * Returns a DDXValue object which is a field in this
 * structured DDXValue object. 
 * */
DDXVariable::DDXValue DDXVariable::DDXValue::field(const std::string & s) throw (std::exception)
{
	return DDXValue(*this,s);
}

/**
 * Returns a DDXValue which represents the string s.
 * Returns itself of s is empty. Example of s includes:
 * ".pls[1].range[2]", ".fieldx", "[32]"
 * Note that a field in a structure is accessed by "." whereas 
 * a item in an array is accessed by starting s by "[". Syntax
 * errors or inconsistencies throw an exception
 * */
DDXVariable::DDXValue DDXVariable::DDXValue::parse(const std::string & s) throw (std::exception)
{
	// printf("Parsing '%s' in '%s'\n",s.c_str(),parsed->name);
	if (s.empty()) {
		return DDXValue(*this);
	}
	if (s[0] == '.') {
		string field = s.substr(1,s.length()-1);
		unsigned int pos = field.find_first_of(".[");
		// printf("Find first of '.[' in '%s': %d -> '%s'\n",field.c_str(),pos,field.substr(0,pos).c_str());
		if (pos == string::npos) {
			return DDXValue(*this,field).parse("");
		} else {
			return DDXValue(*this,field.substr(0,pos)).parse(field.substr(pos,field.length()-pos));
		}
	}
	if (s[0] == '[') {
		unsigned int pos = s.find_first_of("]");
		unsigned int index = 0;
		if (pos == string::npos) {
			throw DDXValueException("DDXValue:parse inconsistent brackets");
		}
		if (sscanf(s.substr(0,pos).c_str(),"[ %d",&index) != 1) {
			throw DDXValueException("DDXValue:parse inconsistent index");
		}
		// skip the closing bracket
		pos += 1;
		return  DDXValue(*this,index).parse(s.substr(pos,s.length()-pos));
	}
	throw DDXValueException("DDXValue:parse failed to parse variable string");
}


