#include "DDXTCPClient.h"
/**
 * \brief Constructor initializes class to default settings
 */
DDXTCPClient::DDXTCPClient( void ) {
  this->tcpHandle = NULL;
  this->remoteHost.clear();
  this->remotePort = -1;

}
/**
 * \brief Constructor initializes class and connects to the given host.
 * \param const std::string host - Host IP
 * \param const int port - Host port
 */
DDXTCPClient::DDXTCPClient( const std::string host, const int port ) {

  if( ! setRemoteHost( host ) ) {
    return;
  }
  if( ! setRemotePort( port ) ) {
    return;
  }
  initConnection();
}
/**
 * \brief Destructor closes connection.
 */
DDXTCPClient::~DDXTCPClient( void ) {
    closeConnection();
}
/**
 * \brief If a connection is present this is closed
 * \return bool - true if successful
 */
bool DDXTCPClient::closeConnection( void ) {
  if( this->tcpHandle != NULL ) {
    if( rtx_inet_done( this->tcpHandle ) == 0 ) {
      this->tcpHandle = NULL;
      return true;
    } else {
      return false;
    }
  }
  return true;
}
/**
 * \brief initializes the connection if a remote host and port is set.
 * \return bool - true if successful
 */
bool DDXTCPClient::initConnection( void ) {
  if( this->remoteHost.empty() ) {
    return false;
  }
  if( this->remotePort < 0 ) {
    return false;
  }
  // Init TCP connection
  this->tcpHandle = rtx_inet_init( RTX_INET_TCP_CLIENT, NULL, 0, 
				   this->remoteHost.c_str(),
				   this->remotePort, 
				   NULL, NULL, NULL );
  if( tcpHandle == NULL ) {
    return false;
  }

  return true;
}
/**
 * \brief Checks if a connection is present.
 * \return bool - true if connected
 */
bool DDXTCPClient::isConnected( void ) const {
   if( this->remoteHost.empty() ) {
    return false;
  }
  if( this->remotePort < 0 ) {
    return false;
  }
  if( this->tcpHandle == NULL ) {
    return false;
  }
  return true;
}
/**
 * \brief Receives the next message in queue from host.
 * \param std::string * msg - Pointer to a string to contain the message.
 * \param const int msgLength - A default msg length of 1024 is used. This can be overwritten here.
 * \return bool - true if successful
 */
bool DDXTCPClient::receive( std::string * msg, const int msgLength ) {
  if( ! isConnected() ) {
    return false;
  }
  char * received = new char[msgLength];
  int res;
  if( (res = rtx_inet_readline( this->tcpHandle->sock, received, msgLength, NULL )) == -1 ) {
    return false;
  }
  msg->assign( received, msgLength );
  delete received;
  return true;
}
/**
 * \brief Send a msg to host
 * \param const std::string msg - Message to be sent
 * \return bool - true if successful
 */
bool DDXTCPClient::send( const std::string msg ) {
  if( ! isConnected() ) {
    return false;
  }
  if( rtx_inet_write( tcpHandle->sock, msg.c_str(), msg.size(), NULL ) == -1 ) {
    return false;
  }
  return true;
}
/**
 * \brief Sets the IP of remote host
 * \param const std::string host - Name or IP of host.
 * \return bool - true if successful
 */
bool DDXTCPClient::setRemoteHost( const std::string host ) {
  this->remoteHost = host;
  return true;
}
/**
 * \brief Sets the port of remote host
 * \param const int port - Host port
 * \return bool - true if successful
 */
bool DDXTCPClient::setRemotePort( const int port ) {
  this->remotePort = port;
  return true;
}
