#include <stdlib.h>
#include <assert.h>
#include <algorithm>
#include <rtx/error.h>
#include "DDXStore.h"

using namespace std;

DDXStore::DDXStore(bool verb)
{
	verbose = verb;
	storeId = NULL;
}

DDXStore::~DDXStore()
{
	close();
}

bool DDXStore::open(const char * storeName,
		unsigned int storePort,unsigned int timeout)
{
	ddx_client_init(0);
	storeId = ddx_store_open(storeName,storePort,timeout);
	if (storeId==NULL) {
		if (verbose) rtx_error_flush("DDXStore::open failed");
		return false;
	}
	return true;
}

bool DDXStore::open(const string & storeName,
		unsigned int storePort,unsigned int timeout)
{
	ddx_client_init(0);
	storeId = ddx_store_open(storeName.c_str(),storePort,timeout);
	if (storeId==NULL) {
		if (verbose) rtx_error_flush("DDXStore::open failed");
		return false;
	}
	return true;
}

bool DDXStore::close()
{
	if (storeId != NULL) {
		ddx_store_close(storeId);
		ddx_client_done();
	}
	storeId = NULL;
	return true;
}

bool DDXStore::listConstants(vector<string> & constants) const
{
	constants.clear();
	DDX_STRING_LIST * list = ddx_store_get_constant_list(storeId);
	DDX_STRING_LIST * it = list;
	while (it != NULL) {
		constants.push_back(string(it->name));
		it = it->next;
	}
	ddx_stringlist_free(list);
	return true;
}

bool DDXStore::listTypes(vector<string> & types) const
{
	types.clear();
	DDX_STRING_LIST * list = ddx_store_get_type_list(storeId);
	DDX_STRING_LIST * it = list;
	while (it != NULL) {
		types.push_back(string(it->name));
		it = it->next;
	}
	ddx_stringlist_free(list);
	return true;
}

bool DDXStore::listVariables(vector<string> & variables) const
{
	variables.clear();
	DDX_STRING_LIST * list = ddx_store_get_variable_list(storeId);
	DDX_STRING_LIST * it = list;
	while (it != NULL) {
		variables.push_back(string(it->name));
		it = it->next;
	}
	ddx_stringlist_free(list);
	return true;
}

bool DDXStore::registerConstant(const string & constName, int value)
{
	int res = ddx_store_register_const(storeId,constName.c_str(),value);
	if (res) {
		if (verbose) rtx_error_flush("DDXStore::registerConstant failed");
		return false;
	}
	return true;
}

bool DDXStore::lookupConstant(const string & constName, int & value) const
{
	int v,r;
	r = ddx_store_lookup_const(storeId,constName.c_str(),&v);
	if (r == 0) {
		value = v;
		return true;
	}
	if (verbose) rtx_error_flush("DDXStore::lookupConstant failed");
	return false;
}

bool DDXStore::registerType(const string & typeDefinition)
{
	int res = ddx_store_register_type(storeId,typeDefinition.c_str());
	if (res) {
		if (verbose) rtx_error_flush("DDXStore::registerType failed");
		return false;
	}
	return true;
}

bool DDXStore::lookupType(const string & typeName,
		string & typeDefinition) const
{
	char * def;
	def = ddx_store_lookup_type(storeId,typeName.c_str());
	if (def != NULL) {
		typeDefinition = def;
		free(def);
		return true;
	}
	if (verbose) rtx_error_flush("DDXStore::lookupType failed");
	return false;
}

bool DDXStore::registerVariable(const string & variableName,
		const string & typeName)
{
	DDX_STORE_ITEM * item = ddx_store_lookup_item(storeId,
			variableName.c_str(),typeName.c_str(),0);
	if (item == NULL) {
		if (verbose) rtx_error_flush("DDXStore::registerVariable failed");
		return false;
	}
	ddx_store_done_item(item);
	return true;
}

bool DDXStore::registerVariable(DDXVariable & var,
		const string & variableName,
		const string & typeName,
		bool direct)
{
	DDX_STORE_ITEM * item = ddx_store_lookup_item(storeId,
			variableName.c_str(),typeName.c_str(),0);
	if (item == NULL) {
		if (verbose) rtx_error_flush("DDXStore::registerVariable failed");
		return false;
	}
	var.setDdxItem(item);
	var.setDirect(direct);
	return true;
}

bool DDXStore::lookupVariable(DDXVariable & var,
		const string & variableName,
		bool direct) const 
{
	// if (!exists(variableName)) return false;
	DDX_STORE_ITEM * item = ddx_store_lookup_item(storeId,
			variableName.c_str(),NULL,0);
	if (item == NULL) {
		if (verbose) rtx_error_flush("DDXStore::lookupVariable failed");
		return false;
	}
	var.setDdxItem(item);
	var.setDirect(direct);
	return true;
}


bool DDXStore::exists(const string & variableName) const
{
	vector<string> vars;
	listVariables(vars);
	return (find(vars.begin(),vars.end(),variableName) != vars.end());
}

