/**
 * \file DDXSerialComm.h
 * \class DDXSerialComm
 * \brief The SerialComm class implements a seria communication using the rtx_seial interface.
 *
 * \author Navid Nourani-Vatani
 * \version 1.0.20070219
 */

// C libs
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
// C++ libs
#include <iostream>
#include <string>
// CSIRO libs
#include <rtx/serial.h>
#include <rtx/timer.h>

class DDXSerialComm {

 protected:
  std::string device;
  int baud;
  int dataBits;
  int stopBits;
  RtxSerialParity parity;
  RtxSerialFlow flow;
  RtxSerialModem modem;
  int vMin;
  int vTime;

  int fileDesc;
  char syncByte;
  int packetLength;
  bool connected;

  std::string msg;

 public:
  /**
   * \brief Constructor initializes class to default values.
   */
  DDXSerialComm( void );
  /**
   * \brief Constructor initializes class to given device and baud
   */
  DDXSerialComm( const std::string, const int );
  /**
   * \brief Destructor closes the serial connection
   */  
  ~DDXSerialComm( void );

  /**
   * \brief Closes the serial connection.
   * \return bool - true if succesful
   */
  bool closeComm( void );

  /**
   * \brief Returns the name of the device.
   * \return std::string - Name of device
   */
  std::string getDevice( void ) const { return this->device; }
  /**
   * \brief Returns the baud used
   * \return int - Connection speed in bps
   */
  int getBaud( void ) const { return this->baud; }
  /**
   * \brief Returns the number of data bits used.
   * \return int - Number of data bits.
   */
  int getDataBits( void ) const { return this->dataBits; }
  /**
   * \brief Returns the number of stop bits used.
   * \return int - Number of stop bits.
   */  
  int getStopBits( void ) const { return this->stopBits; }
  /**
   * \brief Returns the type of parity used.
   * \return RtxSerialParity - The type of rtx parity.
   */
  RtxSerialParity getParity( void ) const { return this->parity; }
  /**
   * \brief Returns the type of flow control used.
   * \return RtxSerialFlow - The type of rtx flow control.
   */
  RtxSerialFlow getFlow( void ) const { return this->flow; }
  /**
   * \brief Returns the type of modem used.
   * \return RtxSerialModem - The type of rtx modem used.
   */
  RtxSerialModem getModem( void ) const { return this->modem; }
  /**
   * \brief Returns the VMIN.
   * \return int - VMIN.
   */
  int getVmin( void ) const { return this->vMin; }
  /**
   * \brief Returns the VTIME.
   * \return int VTIME.
   */
  int getVtime( void ) const { return this->vTime; }

  /**
   * \brief Flushes the input and output
   * \return bool - true if successful
   */
  bool flush( void ) const ;

  /**
   * \brief Opens the serial connection.
   * \return bool - true if successful
   */
  bool initComm( void );

  /**
   * \brief Receives the next msg. Reads byte for byte from header until next header.
   * \param std::string * - Pointer to string containing the msg.
   * \return bool - true if successful
   */
  bool receive( std::string * );
  bool send( const std::string ) const;

  bool setPacketLength( const int );
  bool setSyncByte( const unsigned char );
  bool setLine( const std::string );
  bool setBaud( const int );
  bool setDataBits( const int );
  bool setStopBits( const int );
  bool setParity( const RtxSerialParity );
  bool setFlow( const RtxSerialFlow );
  bool setModem( const RtxSerialModem );
  bool setVmin( const int );
  bool setVtime( const int );

  void display_bits( unsigned value, unsigned int bits ) {
    unsigned c, displayMask = 1 << bits-1;
    
    /*printf( "%7.0u = ", value );*/
    
    for( c = 1; c <= bits; c++ ) {
      printf( "%c", (value & displayMask ? '1' : '0' ) );
      value <<= 1;
      
      if( c % 8 == 0 ) {
	/*printf( " " );*/
      }
    }
    /*printf( "\n" );*/
  }

};
