#include <assert.h>
#include <signal.h>
#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <strings.h>
#include <pthread.h>
#include <time.h>
#include <math.h>

#include <string>
#include <iostream>
#include <map>
#include <set>

#include <rtx/defines.h>
#include <rtx/main.h>
#include <rtx/message.h>
#include <rtx/time.h>
#include <rtx/error.h>
#include <rtx/command.h>
#include <rtx/parse.h>
#include <rtx/inet.h>
#include <rtx/httpd.h>
#include <rtx/getopt.h>

#include "../src/DDXStore.h"
#include "../laser/DDXLaser.h"
#include "../video/DDXVideo.h"
#include "../video/JpegCodec.h"
#include "DDXWebRender.h"
#include "DDXWebPost.h"

static char rcsid[] RTX_UNUSED = "$Id: DDXWebServer.cpp 3073 2008-05-15 04:55:42Z roy029 $";

using namespace std;

typedef enum {HTML=1,
	TEXT,
	XML,
	XSD,
	SVG,
	JPG,
	JPGSTREAM,
	BIN,
	BINSTREAM,
	FORM
} RequestType; 

int dontwait=0;
double timeout = 3.0;
int done = 0;
int port = 8000;
int quality = 75;
int debug = 0;
int grayimages = 0;
DDXStore store;

struct StoredVariable {
	DDXVariable * variable;
	bool locked;
};

typedef map<string,StoredVariable> VariableMap;
pthread_mutex_t vmaplock = PTHREAD_MUTEX_INITIALIZER;
VariableMap vmap;

const char *help = "CSIRO Video Server Project\nServe video to a browser";
RtxGetopt myOpts[] = {
	{ "timeout", "How long (in s) to wait for next image",
		{ { RTX_GETOPT_DOUBLE, &timeout, "timeout"}, RTX_GETOPT_END_ARG } },
	{ "port", "Select tcp port",
		{ { RTX_GETOPT_INT, &port, "port"}, RTX_GETOPT_END_ARG } },
	{ "gray", "Send images in gray level",
		{ { RTX_GETOPT_SET, &grayimages, "grayimages"}, RTX_GETOPT_END_ARG } },
	{ "dbg", "Select debug level",
		{ { RTX_GETOPT_INT, &debug, "debug"}, RTX_GETOPT_END_ARG } },
	{ "quality", "Select encoding quality, in [50-100]",
		{ { RTX_GETOPT_INT, &quality, "quality"}, RTX_GETOPT_END_ARG } },
	{ "dontwait", "Don't wait for next value to be written",
		{ { RTX_GETOPT_SET, &dontwait, "dontwait"}, RTX_GETOPT_END_ARG } },
	RTX_GETOPT_END 
};

VariableMap::iterator getVideo(const string & name)
{
	// It must exists since video object are too big to be on a remote store
	if (!store.exists(name)) return vmap.end();

	string index("video/"); index += name;
	DDXVideo *video;
	VariableMap::iterator vid = vmap.find(index);
	pthread_mutex_lock(&vmaplock);
	if (vid == vmap.end()) {
		StoredVariable sv;
		sv.variable = video = new DDXVideo;
		if (!video->lookupVariable(store,name)) {
			rtx_error_flush("Could not lookup '%s'",name.c_str());
			delete sv.variable;
			pthread_mutex_unlock(&vmaplock);
			return vmap.end();
		}
		sv.locked = true;
		vid = vmap.insert(pair<string,StoredVariable>(index,sv)).first;
		if (debug) rtx_message("Linked with ddxvariable '%s'",name.c_str());
	} else {
		DDXVideo * video = (DDXVideo*)(vid->second.variable);
		if (video == NULL) return vmap.end();
		
		if (vid->second.locked) {
			rtx_message("Discarding request on '%s'",name.c_str());
			pthread_mutex_unlock(&vmaplock);
			return vmap.end();
		}
		vid->second.locked = true;
	}
	pthread_mutex_unlock(&vmaplock);
	
	return vid;
}

VariableMap::iterator getLaser(const string & name)
{
	string index("laser/"); 
	index += name;
	DDXLaser *laser;
	VariableMap::iterator vid = vmap.find(index);
	pthread_mutex_lock(&vmaplock);
	if (vid == vmap.end()) {
		StoredVariable sv;
		sv.variable = laser = new DDXLaser; 
		if (!laser->lookupVariable(store,name)) {
			rtx_error_flush("Could not lookup '%s'\n",name.c_str());
			delete sv.variable;
			pthread_mutex_unlock(&vmaplock);
			return vmap.end();
		}
		sv.locked = true;
		vid = vmap.insert(pair<string,StoredVariable>(index,sv)).first;
		if (debug) {
			rtx_message("Linked with ddxvariable '%s'\n",name.c_str());
		}
	} else {
		DDXLaser * laser = (DDXLaser*)(vid->second.variable);
		if (laser == NULL) return vmap.end();
		
		if (vid->second.locked) {
			rtx_message("Discarding request on '%s'\n",name.c_str());
			pthread_mutex_unlock(&vmaplock);
			return vmap.end();
		}
		vid->second.locked = true;
	}
	pthread_mutex_unlock(&vmaplock);
	
	return vid;
}

VariableMap::iterator getVariable(const string & type, const string & name)
{
	if (type == "video") return getVideo(name);

	string index = type + "/" + name;
	DDXVariable * var;
	VariableMap::iterator vid = vmap.find(index);
	pthread_mutex_lock(&vmaplock);
	if (vid == vmap.end()) {
		StoredVariable sv;
		sv.variable = var = new DDXVariable;
		if (!store.lookupVariable(*var,name)) {
			rtx_error_flush("Could not lookup '%s'",name.c_str());
			delete sv.variable;
			pthread_mutex_unlock(&vmaplock);
			return vmap.end();
		}
		sv.locked = true;
		vid = vmap.insert(pair<string,StoredVariable>(index,sv)).first;
		if (debug) rtx_message("Linked with ddxvariable '%s'",name.c_str());
	} else {
		if (vid->second.locked) {
			pthread_mutex_unlock(&vmaplock);
			return vmap.end();
		}
		vid->second.locked = true;
	}
	pthread_mutex_unlock(&vmaplock);
	return vid;
}

void print4bytes(unsigned char * b,int n) 
{
	int i;
	for (i=0;i<n;i++) printf("%02X ",(unsigned int)b[i]);
}

#define DOR_ELSE(C,E) if ((C)!=0) {E;vid->second.locked=false; rtx_error_flush("Command '"#C"'failed"); return 0;}
#define DOR(C) if ((C)!=0) {vid->second.locked=false; rtx_error_flush("Command '"#C"'failed"); return 0;}
#define DO(C) if ((C)!=0) {rtx_error_flush("Command '"#C"'failed"); return 0;}

int
ddxweb_handler_video(RtxHttpdReq * r,const string & url) 
{
	int thisquality = quality;
	int thisgray = grayimages;

	int field = 0;
	unsigned int pos;
	string request = url;
	pos = request.find(".jpg");
	if (pos < request.size()) {
		request = request.substr(0,pos);
		rtx_message("Removed .jpg: '%s'",request.c_str());
	}

	string args = request;
	pos = args.find_first_of(':');
	if (pos < args.size()) {
		request = request.substr(0,pos);
	}
	while (pos < args.size()) {
		sscanf(args.c_str()+pos+1,"field=%d",&field);
		sscanf(args.c_str()+pos+1,"gray=%d",&thisgray);
		sscanf(args.c_str()+pos+1,"grey=%d",&thisgray);
		sscanf(args.c_str()+pos+1,"quality=%d",&thisquality);
		args = args.substr(pos+1,args.size()-pos-1);
		pos = args.find_first_of(':');
	}

	JpegCodec jpeg(thisquality);
	if (thisgray) {
		jpeg.setOutputColorSpace(JpegCodec::cmGray);
	} else {
		jpeg.setOutputColorSpace(JpegCodec::cmAuto);
	}
	
	if (debug)
		rtx_message("Connection started for '%s' field %d",request.c_str(),field);

	VariableMap::iterator vid = getVideo(request);
	if (vid == vmap.end()) 
		return rtx_httpd_send_error(r,404,"Failed to lookup variable");
	DDXVideo * video = (DDXVideo*)(vid->second.variable);

	if (dontwait) {
		video->read(1.0,0);
	} else if (!video->read(timeout,1)) { 
		vid->second.locked = false;
		if(rtx_httpd_send_error(r,304,NULL)!=0){
			rtx_error("couldn't send error response");
		}
		if (debug) rtx_error("Failed to read video frame");
		return 0;
	}
	
	if (!jpeg.encode(video,field)) {
		return rtx_httpd_send_error(r,404,"Could not encode the content of this variable");
	}

	DOR(rtx_httpd_reply_send_status(r,200,NULL)); 
	DOR(rtx_httpd_reply_send_header(r,"Content-Type","image/jpeg"));
	DOR(rtx_httpd_reply_send_header(r,"Content-Transfer-Encoding","binary")); 
	DOR(rtx_httpd_reply_send_allbody(r,jpeg.getBuffer(),jpeg.bsize(),3.0));

	if (debug) rtx_message("Connection terminated for '%s'",request.c_str());
	vid->second.locked = false;

    return 0;
}

int
ddxweb_handler_svg(RtxHttpdReq * r,const string & request) 
{
	if (debug)
	{ rtx_message("Connection started for '%s'\n",request.c_str()); }

	VariableMap::iterator vid;
	vid = getLaser(request);
	if (debug) { rtx_message("Connection started for '%s'\n",request.c_str()); }

	if (vid == vmap.end()){ 
		vid->second.locked = false;
		return rtx_httpd_send_error(r,404,"Failed to lookup variable");
	}
	DDXLaser * laser = (DDXLaser*) (vid->second.variable);

	if (dontwait) {
		laser->read(1.0,0);
	} else if (!laser->read(timeout,1)) { 
		vid->second.locked = false;
		if(rtx_httpd_send_error(r,304,NULL)!=0){
			rtx_error("couldn't send error response");
		}
		if (debug) rtx_error("Failed to read variable '%s'",request.c_str());
		return 0;
	}

	string output;
	output = renderVarAsSVG(output, laser);

	DOR(rtx_httpd_reply_send_status(r,200,NULL)); 
	DOR(rtx_httpd_reply_send_header(r,"Content-Type", "image/svg+xml"));
	DOR(rtx_httpd_reply_send_body(r,output.c_str(),output.size()));

	if (debug) {rtx_message("Connection terminated for '%s'\n",request.c_str());}
	vid->second.locked = false;

    return 0;
}

int
ddxweb_handler_binary(RtxHttpdReq * r,const string & request) 
{
	if (debug)
		rtx_message("Connection started for '%s'",request.c_str());
	VariableMap::iterator vid = getVariable("binary",request);
	if (vid == vmap.end()) 
		return rtx_httpd_send_error(r,404,"Failed to lookup variable");
	DDXVariable *var = vid->second.variable;

	if (dontwait) {
		var->read(1.0,0);
	} else if (!var->read(timeout,1)) { 
		vid->second.locked = false;
		if(rtx_httpd_send_error(r,304,NULL)!=0){
			rtx_error("couldn't send error response");
		}
		if (debug) rtx_error("Failed to read variable '%s'",request.c_str());
		return 0;
	}

	DOR(rtx_httpd_reply_send_status(r,200,NULL)); 
	DOR(rtx_httpd_reply_send_header(r,"Content-Type", "binary/octet-stream"));
	unsigned int psize = var->packsize();
	unsigned char *buffer = new unsigned char[psize];
	var->pack(buffer);
	DOR_ELSE(rtx_httpd_reply_send_body(r,buffer,psize), delete [] buffer);

	if (debug) rtx_message("Connection terminated for '%s'",request.c_str());
	vid->second.locked = false;

    return 0;
}

int
ddxweb_handler_binary_stream(RtxHttpdReq * r,const string & request) 
{
	if (debug)
		rtx_message("Connection started for '%s'",request.c_str());
	// We don't use the getVariable mecanism here, because this request is not
	// instantaneous, we keep the variable and we keep reading until the
	// connection is closed.
	// On the other hand, this cost a lookup
	DDXVariable var;
	if (!store.lookupVariable(var,request)) {
		return rtx_httpd_send_error(r,404,"Failed to lookup variable");
	}
	unsigned int psize = var.packsize();
	char length[128];
	sprintf(length,"%d",psize);
	unsigned char *buffer = new unsigned char[psize];

	DO(rtx_httpd_reply_send_status(r,200,NULL));
	//DOR(rtx_httpd_reply_send_header(r,"Content-Length", length));
	DO(rtx_httpd_reply_send_header(r,"Content-Type", "binary/octet-stream"));
	DO(rtx_httpd_reply_send_header(r,"Cache-Control", "no-cache"));
	DO(rtx_httpd_reply_send_header(r,"Connection", "Keep-Alive"));
	DO(rtx_httpd_reply_send_header(r,"Content-type","multipart/x-mixed-replace;boundary=%%boundary%%"));
	DO(rtx_httpd_reply_flush_headers(r));

	char tmp[256];
	while (1) {
		var.read(-1.0,1);
		var.pack(buffer);
		sprintf(tmp,"--%%%%boundary%%%%\r\nContent-Length: %s\r\nX-TimeStamp: %f\r\n\r\n",length,var.timeStamp());
		int tlen = strlen(tmp);
		if (rtx_inet_write(r->sock,tmp,tlen,NULL) != tlen) break;
		if (rtx_inet_write(r->sock,(const char*)buffer,psize,NULL) != (signed int)psize) break;
		if (rtx_inet_write(r->sock,"\r\n",2,NULL) != 2) break;
	}

	delete [] buffer;
	if (debug) rtx_message("Connection terminated for '%s'",request.c_str());

    return 0;
}

int
ddxweb_handler_text(RtxHttpdReq * r,const string & request,int style) 
{
	if (debug)
		rtx_message("Connection started for '%s'",request.c_str());
	VariableMap::iterator vid = getVariable("text",request);
	if (vid == vmap.end()) 
		return rtx_httpd_send_error(r,404,"Failed to lookup variable");
	DDXVariable *var = vid->second.variable;

	if (dontwait) {
		var->read(1.0,0);
	} else if (!var->read(timeout,1)) { 
		vid->second.locked = false;
		if(rtx_httpd_send_error(r,304,NULL)!=0){
			rtx_error("couldn't send error response");
		}
		if (debug) rtx_error("Failed to read variable '%s'",request.c_str());
		return 0;
	}

	string output;
	RtxParseVar * V = var->getParsedVar();
	if (V == NULL) {
		return rtx_httpd_send_error(r,404,"Failed to parse variable");
	}
	switch (style) 
	{
		case HTML :
			output = renderVarAsHTML(output, V,V->dim,V->arrayDim,0,true,var->rawPointer());
			if (debug) cerr << output << endl;
			break;
		case FORM :
			output = renderVarAsHTML(output, V,V->dim,V->arrayDim,0,true,var->rawPointer(),true,request);
			if (debug) cerr << output << endl;
			break;
		case TEXT :
			output = renderVarAsText(output, V,V->dim,V->arrayDim,0,true,var->rawPointer());
			if (debug) cerr << output << endl;
			break;
		case XML :
			output = renderVarAsXML(output, V,V->dim,V->arrayDim,0,true,var->rawPointer(),var->timeStamp());
			if (debug) cerr << output << endl;
			break;
		case XSD :
			output = renderVarAsXSD(output, V,V->dim,V->arrayDim,0,true,var->rawPointer());
			if (debug) cerr << output << endl;
			break;
		default : break;
	}	
			
	rtx_parse_free_var(V);

	DOR(rtx_httpd_reply_send_status(r,200,NULL)); 

	string mime="text/html";
	switch (style)
	{
		case FORM :
		case HTML :
			mime = "text/html";
			break;
		case TEXT :
			mime = "text/plain";
			break;
		case XML :
			mime = "text/xml";
			break;
		case XSD :
			mime = "text/xml";
			break;
		case SVG :
			mime = "image/svg+xml";
		default: break;
	}

	DOR(rtx_httpd_reply_send_header(r,"Content-Type", mime.c_str()));
	DOR(rtx_httpd_reply_send_body(r,output.c_str(),output.size()));

	if (debug) rtx_message("Connection terminated for '%s'",request.c_str());
	vid->second.locked = false;

    return 0;
}

int ddxweb_handler_list(RtxHttpdReq *r)
{
	vector<string> variables;
	if (!store.listVariables(variables)) 
		return rtx_httpd_def_handler(r);

	string resp = "<html><body><h1>Variables available locally</h1><table border=1>\n";
	unsigned int i;
	for (i=0;i<variables.size();i++) {
		const string & V = variables[i];
		resp += "<tr> <td> " + V + ":</td> ";
		resp += "<td><a href=text/"+V+">text</a></td> ";
		resp += "<td><a href=html/"+V+">html</a></td> ";
		resp += "<td><a href=form/"+V+">form</a></td> ";
		resp += "<td><a href=xml/"+V+">xml</a></td> ";
		resp += "<td><a href=xsd/"+V+">xsd</a></td> ";
		resp += "<td><a href=bin/"+V+">bin</a></td> ";
		resp += "<td><a href=binstream/"+V+">bin</a></td> ";
		resp += "<td><a href=svg/"+V+">svg</a></td> ";
		resp += "<td><a href=video/"+V+">video</a></td> ";
		resp += "</tr>\n";
	}
	resp += "</table></body></html>\n";

	DO(rtx_httpd_reply_send_status(r,200,NULL));
	DO(rtx_httpd_reply_send_header(r,"Content-Type","text/html")); 
	DO(rtx_httpd_reply_send_body(r,resp.c_str(),resp.size()));
	return 0;
}

int ddxweb_handler_help(RtxHttpdReq *r)
{
	string respend = "</body></html>\n";
	string resphead = "<html><body><h1> DDXWebServer help</h1>\n";
	string respactions = 
		"<h2> Possible Actions </h2> <ul>\n"
		"<li> <a href=help>Access to this help </a></li>\n"
		"<li> <a href=list>Access to the variable list</a></li>\n"
		"<li> See the text representation of a variable: go to text/&lt;varname&gt; on this server</li>\n"
		"<li> See the html representation of a variable: go to html/&lt;varname&gt; on this server</li>\n"
		"<li> See the video representation of a video variable: go to video/&lt;varname&gt; on this server</li>\n"
		"<li> When requesting a variable, <a href=wait>wait for the next value to be written</a>, "
		"or <a href=dontwait>return with the current value</a></li>\n"
		"</ul>";
	
	string respstatus = "<h2> Current status </h2> <ul>\n";

	if (dontwait) {
		respstatus += "<li> Wait mode: immediate value </li>\n";
	} else {
		respstatus += "<li> Wait mode: wait next value </li>\n";
	}
	respstatus += "</ul>";
		
	string resp = resphead + respactions + respstatus + respend;

	DO(rtx_httpd_reply_send_status(r,200,NULL));
	DO(rtx_httpd_reply_send_header(r,"Content-Type","text/html")); 
	DO(rtx_httpd_reply_send_allbody(r,resp.c_str(),resp.size(),1.0));
	return 0;
}


static void hash_to_map(void *key, void *data,
			int i, int j,
			int *stop_iter,
			void *context) {
	map <string, string> * hashmap = (map <string, string> *) context;
	hashmap->insert(pair<string, string>((char *)key, (char *) data) );
	if (debug)
		rtx_message("name:%s  content:%s", key, (char *)data);
	
}

int ddxweb_handler_input(RtxHttpdReq *r)
{
	rtx_message("in the handler_input");
	/* make a map from the hash table */
	map <string, string> hashMap;
	rtx_hash_iter(r->args, hash_to_map, &hashMap );

	string respend = "</body></html>\n";
	string resphead = "<html><body><h1> DDXWebServer Input</h1>\n";
	
	string resp = resphead + respend;

	DO(rtx_httpd_reply_send_status(r,200,NULL)); 
	DO(rtx_httpd_reply_send_header(r,"Content-Type", "text/html")); 
	DO(rtx_httpd_reply_send_body(r,resp.c_str(),resp.size()));
	return 0;

}


int ddxweb_handler_post(RtxHttpdReq *r) //contains post variable
{
	VariableMap::iterator vid;
	DDXVariable *var;
	string output;
	bool postok = true;
	set < DDXVariable * > storeSet;
	set < DDXVariable * >::iterator storeSetIt;
	pair <set< DDXVariable * >::iterator, bool > insPair;

	if (debug) {
		cout << " in ddxweb post handler " << endl;
	}

	/* make a map from the post data */
	PostMap postMap; 

	map <string, string> hashMap;
	map <string, string>::iterator hashMapIt;

	rtx_hash_iter(r->post, post_hash_to_map, &postMap );

	if (debug)
		rtx_message("content length = %d",r->content_length);

	PostMap::iterator mit = postMap.begin();

	if(debug)
		rtx_message("[post] mapsize  %d",postMap.size());

	while (mit != postMap.end())
	{
		if (debug)
			rtx_message("[post] mit->first2[1] %d", mit->first[0].first.c_str());
		/* look up var in store */	
		vid = getVariable("text", mit->first[0].first);
		if (vid == vmap.end()) {
			rtx_message("vid = vmap.end COULD NOT FIND VAR IN STORE");
			output = "vid = vmap.end COULD NOT FIND VAR IN STORE";
			postok = false;
			break;
		}

		var = vid->second.variable;

		/* check to see if var has already been used */

		insPair = storeSet.insert(var);
		if (insPair.second)
			(*insPair.first)->read(1.0,0);

		/* find the write place in the buffer and write variable value */	
		if ( !(postok =  find_where_to_write((*insPair.first), mit, debug)) ) {
			rtx_message("find_where to write returned false");
			output = "where to write returned false maybe a struct at bottom of tree";
			vid->second.locked = false;
			break;
		}

		mit++;
		vid->second.locked = false;
	}
	/* write variables to the store */
	storeSetIt = storeSet.begin();
	while (storeSetIt != storeSet.end()) {
		if (!(postok = (*storeSetIt)->write()))
		{
			rtx_message("var->write failed");
			output = "var->write failed";
		}
		storeSetIt++;
	}
	storeSet.clear();

	/* Put html stuff together */
	string respend = "</body></html>\n";
	string resphead = "<html><body><h1> DDXWebServer Input</h1>\n";

	if (postok)
	{
		output = "successful post";
		DOR(rtx_httpd_reply_send_status(r,200,NULL));
	} else {
		DOR(rtx_httpd_reply_send_status(r,400,NULL));
	}
	if (debug)
		cerr << output << endl;

	string resp = resphead + output + respend;
	DOR(rtx_httpd_reply_send_header(r,"Content-Type", "text/html"));
	DOR(rtx_httpd_reply_send_body(r,resp.c_str(),resp.size()));
	vid->second.locked = false;
	return 0;
}

int ddxweb_handler_waitornot(RtxHttpdReq *r, bool wait) 
{
	if (wait)
		dontwait = 0;
	else
		dontwait = 1;

	DO(rtx_httpd_reply_send_status(r,302,NULL));
	DO(rtx_httpd_reply_send_header(r,"Content-Type","text/html"));
	DO(rtx_httpd_reply_send_header(r,"Location","/help")); 
	string resp = "<html> Hello, please go to <a href=/help>help</a></html>";
	DO(rtx_httpd_reply_send_body(r,resp.c_str(),resp.size()));
	return 0;
}

int ddxweb_handler_favicon(RtxHttpdReq *r)
{
	DO(rtx_httpd_reply_send_status(r,301,NULL)); 
	DO(rtx_httpd_reply_send_header(r,"Content-Type","text/html"));
	DO(rtx_httpd_reply_send_header(r,"Location","http://www.csiro.au/files/layout/logo.gif"));
	string resp("No favicon here");
	DO(rtx_httpd_reply_send_body(r,resp.c_str(),resp.size()));
	return 0;
}

struct ImageParam {
	unsigned int quality;
	unsigned int field;
	unsigned int gray;
	ImageParam() {
		quality = field = gray = 0;
	}
	ImageParam(unsigned int q, unsigned int f, unsigned int g) {
		quality=q;
		field=f;
		gray=g;
	}
};

int ddxweb_handler_compound(RtxHttpdReq *r, const string & request, unsigned int timeout)
{
	int thisquality = quality;
	int thisgray = grayimages;

	unsigned int i;
	unsigned int pos;
	char tmphead[80];
	const string boundary("--partboundary\015\012");
	string errstr;
	string creq = request;
	vector< VariableMap::iterator > vars;
	map<string, ImageParam> vparams;
	vector< string > names;
	vector< string > types;
	vector< pair<unsigned int, unsigned char *> > data;
	JpegCodec jpeg(thisquality);
	unsigned int global_size = 0;
	if (debug)
		rtx_message("Requested var: '%s'",request.c_str());
	while (!creq.empty()) {
		DDXVariable * v;
		string var, type, varmap;
		pos = creq.find_first_of('&');
		if (pos == string::npos) {
			var = creq;
			creq.clear();
		} else {
			var = creq.substr(0,pos);
			creq = creq.substr(pos+1,request.size()-pos-1);
		}
		if (debug > 1) {
			printf("Var '%s' Req '%s'\n",var.c_str(),creq.c_str());
		}
		pos = var.find_first_of('.');
		if (pos == string::npos) {
			type = "txt";
		} else {
			type = var.substr(pos+1,request.size()-pos-1);
			var = var.substr(0,pos);
		}
		if (debug > 1) {
			printf("Var '%s' Type '%s'\n",var.c_str(),type.c_str()); 
		}
		/** Convert extensions to standard names **/
		varmap = "text";
		if ((type == "video") || (type == "jpg") || (type == "jpeg")) {
			type = "jpeg";
			varmap = "video";
			string args = var;
			pos = args.find_first_of(':');
			if (pos != string::npos) {
				var = var.substr(0,pos);
			}
			if (debug > 1) {
				printf("Var '%s' Args '%s'\n",var.c_str(),args.c_str());
			}
			int field = 0;
			thisquality = quality;
			thisgray = grayimages;
			while (pos != string::npos) {
				sscanf(args.c_str()+pos+1,"field=%d",&field);
				sscanf(args.c_str()+pos+1,"gray=%d",&thisgray);
				sscanf(args.c_str()+pos+1,"grey=%d",&thisgray);
				sscanf(args.c_str()+pos+1,"quality=%d",&thisquality);
				args = args.substr(pos+1,args.size()-pos-1);
				pos = args.find_first_of(':');
				if (debug > 1) {
					printf("Pos %d Args '%s'\n",pos,args.c_str());
				}
			}

			vparams[var] = ImageParam(thisquality,field,thisgray);
		}
		VariableMap::iterator vid = getVariable(varmap,var);
		if (vid == vmap.end()) {
			// This may be because this variable has already bin locked
			// in this request
			unsigned int j;
			for (j=0;j<names.size();j++) {
				if (names[j] == var) {
					vid = vars[j];
					break;
				}
			}
			if (vid == vmap.end()) {
				// if vid is still not resolved, then let it be
				errstr = string("Failed to lookup variable ") + var;
				goto cleanup;
			}
		}
		if (debug > 1) {
			printf("Got Variable (%s)\n",varmap.c_str());
		}
		names.push_back(var);
		types.push_back(type);
		vars.push_back(vid);
		v = vid->second.variable;
		if (debug > 1) {
			printf("Reading Variable (dont wait = %d)\n",dontwait);
		}
		if (dontwait) {
			v->read(1.0,0);
		} else if (!v->read(timeout,1)) { 
			errstr = string("Failed to read variable ") + var;
			goto cleanup;
		}
		if (debug > 1) {
			printf("Read variable\n");
		}
		// Keep the variable locked, and in memory, 
		// we will generate the data later
		// This should make all the variable acquisition as 
		// synchronous as possible
	}

	// Now generate the output
	for (i = 0;i<vars.size();i++) {
		string output, subhead, mime;
		DDXVariable * var = vars[i]->second.variable;
		RtxParseVar * V = var->getParsedVar();
		if (V == NULL) {
			errstr = string("Failed to parse variable ") + var->getName();
			goto cleanup;
		}
		if (debug > 1) {
			printf("Processing variable '%d'\n",i);
		}
		if (types[i] == "jpeg") {
			DDXVideo * video = (DDXVideo*)(var);
			const ImageParam & p = vparams[var->getName()];
			jpeg.setQuality(p.quality);
			if (p.gray) {
				jpeg.setOutputColorSpace(JpegCodec::cmGray);
			} else {
				jpeg.setOutputColorSpace(JpegCodec::cmAuto);
			}
			if (!jpeg.encode(video,p.field)) {
				errstr = string("Could not encode the content of variable") + var->getName();
				goto cleanup;
			}
			sprintf(tmphead,"Content-Length: %d\015\012",jpeg.bsize());
			subhead = boundary + 
				"Content-Description: " + var->getName() + "\015\012" +
				string("Content-Type: image/jpeg\015\012") + 
				tmphead;

			unsigned int bsize = subhead.size() + jpeg.bsize();
			unsigned char * buffer = (unsigned char *) malloc(bsize);
			memcpy(buffer,subhead.c_str(),subhead.size());
			memcpy(buffer+subhead.size(),jpeg.getBuffer(),jpeg.bsize());
			data.push_back(pair<unsigned int,unsigned char*>(bsize,buffer));
			global_size += bsize;
		} else {
			if (types[i] == "html") {
				mime = "text/html";
				output = renderVarAsHTML(output, V,V->dim,V->arrayDim,0,true,var->rawPointer());
			} else if (types[i] == "xml") {
				mime = "text/xml";
				output = renderVarAsXML(output, V,V->dim,V->arrayDim,0,true,var->rawPointer(), var->timeStamp());
			} else if (types[i] == "xsd") {
				mime = "text/xsd";
				output = renderVarAsXSD(output, V,V->dim,V->arrayDim,0,true,var->rawPointer());
			} else if ((types[i] == "txt") || (types[i] == "text")) {
				mime = "text/plain";
				output = renderVarAsText(output, V,V->dim,V->arrayDim,0,true,var->rawPointer());
			}
			sprintf(tmphead,"Content-Length: %d\015\012",output.size());
			subhead = boundary + 
				"Content-Description: " + var->getName() + "\015\012" +
				"Content-Type: " + mime + "\015\012" +
				string(tmphead);
			output = subhead + output;
			data.push_back(pair<unsigned int,unsigned char*>(output.size(),
						(unsigned char*)strdup(output.c_str())));
			global_size += output.size();
		}
	}

	// If we manage to arrive here, we can send every thing, 
	// Yeehaa
	if(rtx_httpd_reply_send_status(r,200,NULL)!=0) {
		rtx_error("Could not set HTTP response status");
		goto cleanup;
	}
	if(rtx_httpd_reply_send_header(r,"Content-Type","multipart/related; boundary=partboundary")!=0){
		rtx_error("Could not set Content Type");
		goto cleanup;
	}
	if(rtx_httpd_reply_send_header(r,"Content-Transfer-Encoding","binary")!=0){
		rtx_error("Could not set Content Transfer Encoding");
		goto cleanup;
	}
	sprintf(tmphead,"%d",global_size);
	if(rtx_httpd_reply_send_header(r,"Content-Length",tmphead)!=0){
		rtx_error("Could not set Content Length");
		goto cleanup;
	}
	if (r->persistent) {
		if(rtx_httpd_reply_send_header(r,"Connection","Keep-Alive")!=0){
			rtx_error("Could not set Connection");
			goto cleanup;
		}
	} else {
		if(rtx_httpd_reply_send_header(r,"Connection","Close")!=0){
			rtx_error("Could not set Connection");
			goto cleanup;
		}
	}
	if(rtx_httpd_reply_flush_headers(r)!=0){
		rtx_error("Could not flush headers");
		goto cleanup;
	}
	// No send the bits of data gathered in all the variables
	for (i = 0;i<data.size();i++) {
		unsigned int written, len;
		unsigned char * bit;
		RtxTime ts0, ts1;
		double t0, t1;
		written = 0;
		len = data[i].first;
		bit = data[i].second;
		rtx_time_get(&ts0); t0 = rtx_time_to_double(&ts0);
		while (written != len) {
			rtx_time_get(&ts1); t1 = rtx_time_to_double(&ts1);
			if ((t1-t0) > timeout) {
				rtx_message("ddxweb_handler_compound: timeout while sending body %d",i);
				goto cleanup;
			}
			int ret = rtx_inet_write(r->sock,(const char*)bit,len-written,NULL);
			if (ret < 0) {
				rtx_message("ddxweb_handler_compound: inet write failed while sending body %d",i);
				goto cleanup;
			}
			written += ret; 
			bit += ret;
		}
	}
	
cleanup:
	for (i = 0;i<data.size();i++) {
		free(data[i].second);
	}
	for (i = 0;i<vars.size();i++) {
		vars[i]->second.locked = false;
	}
	if (!errstr.empty()) {
		rtx_httpd_send_error(r,404,errstr.c_str());
	}
	return 0;
}

int ddxweb_handler(RtxHttpdReq *r)
{
	if (debug)
		rtx_message("DDXWeb: connection request '%s'",r->url);

//  if (!(r->method == RTX_HTTPD_M_GET))
//		return rtx_httpd_send_error(r,404,"unsupported url");
	int retVal = -1;
	string request;
	unsigned int pos;
	string reqtype, vardef;
	if (debug)
		rtx_message("r->content %d",r->content_length);
	switch (r->method)
	{
		case RTX_HTTPD_M_GET :
	
			request = r->url;
			request.erase(0,1);
			if (request.empty()) { 
				retVal = ddxweb_handler_help(r);
				break;
			}
			pos = request.find_first_of('/');
	
			reqtype = request.substr(0,pos);
			vardef = request.substr(pos+1,request.size()-pos-1);

			if (pos < request.size()) {
				if (reqtype == string("video")) {
					retVal = ddxweb_handler_video(r,vardef);
					break;
				}	
				if (reqtype == string("svg")) {
					retVal = ddxweb_handler_svg(r,vardef);
					break;
				}
				if (reqtype == string("html")) {
					retVal = ddxweb_handler_text(r,vardef, HTML);
					break;
				}
				if (reqtype == string("form")) {
					retVal = ddxweb_handler_text(r,vardef, FORM);
					break;
				}
				if (reqtype == string("xml")) {
					retVal = ddxweb_handler_text(r,vardef, XML);
					break;
				}
				if (reqtype == string("xsd")) {
					retVal = ddxweb_handler_text(r,vardef, XSD);
					break;
				}
				if (reqtype == string("text")) {
					retVal = ddxweb_handler_text(r,vardef,TEXT);
					break;
				}
				if (reqtype == string("bin")) {
					retVal = ddxweb_handler_binary(r,vardef);
					break;
				}
				if (reqtype == string("binstream")) {
					retVal = ddxweb_handler_binary_stream(r,vardef);
					break;
				}
				if ((reqtype == string("compound"))||(reqtype == string("var"))) {
					retVal = ddxweb_handler_compound(r,vardef,10);
					break;
				}
			}

			if (reqtype == string("wait")) {
				retVal = ddxweb_handler_waitornot(r,true);
				break;
			}

			if (reqtype == string("dontwait")) {
				retVal = ddxweb_handler_waitornot(r,false);
				break;
			}

			if (reqtype == string("list")) {
				retVal = ddxweb_handler_list(r);
				break;
			}
	
			if (reqtype == string("input")) {
				retVal = ddxweb_handler_input(r);
				break;
			}

			if (reqtype == string("help")) {
				retVal = ddxweb_handler_help(r);
				break;
			}	
			if ((reqtype == string("favicon.ico"))) {
				retVal = ddxweb_handler_favicon(r);
				break;
			}
			if ((reqtype == string("help")) || (reqtype == string("")))  {
				retVal = ddxweb_handler_help(r);
				break;
			}
			break;
		case RTX_HTTPD_M_POST :
			retVal = ddxweb_handler_post(r);
			break;

		default:
			retVal = ddxweb_handler_help(r);
			break;
	}
	if( retVal != 0)
		return rtx_httpd_send_error(r,404,"Invalid URL");
	else 
		return retVal;
}



int main(int argc, char **argv) 
{
	if( RTX_GETOPT_CMD( myOpts, argc, argv, NULL, help) == -1 )
	{ RTX_GETOPT_PRINT( myOpts, argv[0], NULL, help); exit(-1); }

	rtx_main_init("DDXWebServer",RTX_ERROR_STDERR);
    rtx_error_init ("ddxweb", RTX_ERROR_STDERR, NULL);
	if (!store.open()) {
		return rtx_error("Failed to open ddx store");
	}
	

	RtxHttpd * server = NULL;

    rtx_message( "Initialising simple HTTPD server on port %d [debug=%d]",port,debug);
	server = rtx_httpd_server_init(NULL,port,debug,ddxweb_handler);
	
    if (!server) {
        rtx_message("httpd_server_init() failed.");
        exit(1);
    }

    if (debug) rtx_message("HTTPD server running");
    rtx_main_wait_shutdown(0);

	rtx_httpd_server_destroy(server);

    if (debug) rtx_message("HTTPD server terminated");
	for (VariableMap::iterator it = vmap.begin();it!=vmap.end();it++) {
		it->second.variable->terminate();
		delete it->second.variable;
	}

	store.close();
	return 0;
}






