/** \file
  Copyright (c) CSIRO ICT Robotics Centre
  \brief  View Video Stream using SDL - faster for Colour video 
  \author Elliot.Duff@csiro.au
  \warning This program requires XVideo support on the X server
  \todo   Handle more events (including resize)
  */

#include <SDL/SDL.h>
#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>

#include "DDXStore.h"
#include "DDXVideo.h"

#include <rtx/main.h>
#include <rtx/getopt.h>
#include <rtx/error.h>
#include <rtx/message.h>

#include "DDXVPConvert.h"

static char rcsid[] RTX_UNUSED = "$Id: DDXVideoPlay.cpp 3072 2008-05-15 04:48:23Z roy029 $";

int noov=0;
int hwsurf=0;
int field = 0;	/*<! Select field to view (-1 is all) */
const char *name = "video";
const char *help = "CSIRO Video Server Project\nDisplay video using SDL (fast)";

RtxGetopt myOpts[] = {
	{ "field", "Select Video Field to Display",
		{ { RTX_GETOPT_INT, &field, "field"}, RTX_GETOPT_END_ARG } },
	{ "name", "Name of Video in Store",
		{ { RTX_GETOPT_STR, &name, "name"}, RTX_GETOPT_END_ARG } },
	{ "nooverlay", "Use standard SDL surface instead of YUV overlay.",
		{ { RTX_GETOPT_SET, &noov, "noov"}, RTX_GETOPT_END_ARG } },
	{ "hwsurf", "Use HW_SURFACE flag.",
		{ { RTX_GETOPT_SET, &hwsurf, "hwsurf"}, RTX_GETOPT_END_ARG } },
	RTX_GETOPT_END };

void saveCurrentImage(const DDXVideo & video)
{
	char fname[1024];
	unsigned int i = 0;
	for (i=0;i<1000;i++) {
		sprintf(fname,"/tmp/capture.%05d.tga",i);
		FILE * fp =  fopen(fname,"r");
		if (fp == NULL) break;
		fclose(fp);
	}
	if (video.saveAsTga(fname)) {
		printf("Video successfully saved to '%s'\n",fname);
	} else {
		printf("Failed to save '%s'\n",fname);
	}
}

void scale(SDL_Surface* src, SDL_Surface *dest)
{
	int i,j;
	unsigned char *Psrc, *Pdst;
	for (j=0;j<dest->h;j++) {
		Pdst = (unsigned char *)(dest->pixels) + j*dest->pitch;
		for(i=0;i<dest->w;i++) {
			int is,js;
			is = (i * src->w)/dest->w;
			js = (j * src->h)/dest->h;
			Psrc = ((unsigned char*)(src->pixels)) + (js*src->pitch+is*3);
			*Pdst++ = *Psrc++;
			*Pdst++ = *Psrc++;
			*Pdst++ = *Psrc++;
		}
	}
}

int main(int ac, char* av[])
{
	unsigned char *src[3];
	int quit = 0;
	int width,height;
	int depth=24;
	int modes = SDL_ANYFORMAT|SDL_RESIZABLE;
	if (hwsurf)
		modes = SDL_HWSURFACE | modes;
	else
		modes = SDL_SWSURFACE | modes;

	SDL_Surface *screen = NULL;
	SDL_Surface *frame = NULL, *sframe = NULL;
	SDL_Overlay *overlay = NULL;
	SDL_Rect     rect;
	SDL_Event    event;

	rtx_main_init (av[0], 0);
	if( RTX_GETOPT_CMD( myOpts, ac, av, rcsid, help) == -1 )
	{ RTX_GETOPT_PRINT( myOpts, av[0], rcsid, help); exit(-1); }

	DDXStore store;
	DDXVideo video;
	store.open(NULL,0,5);
	video.lookupVariable(store,name,true);

	video.read(10.0,0);

	width  = video.width();
	height = video.height();

	if( field >= (signed)video.fields() ) return rtx_error ("Insufficent Field Depth");

	rect.x = 0;
	rect.y = 0;
	rect.w = width;
	rect.h = height;  

	/* Initialize the SDL library */
	if( SDL_Init(SDL_INIT_VIDEO) < 0 ) 
		return rtx_error("Couldn't initialize SDL: %s\n", SDL_GetError());

	/* Initialize the display */
	screen = SDL_SetVideoMode(width, height, depth, modes);
	if ( screen == NULL ) return rtx_error("Couldn't set video mode: %s\n", SDL_GetError()); 

	SDL_WM_SetCaption("ddxvideo", "ddxvideo");

	if (!noov) {
		unsigned int w=width,h=height;
		if (w%2) w += 1;
		if (h%2) h += 1;
		overlay = SDL_CreateYUVOverlay(w,h,SDL_YV12_OVERLAY,screen);
	} else {
		frame = createSurface(width,height,24);
		sframe = createSurface(width,height,24);
	}


	bool stalled = false;
	while( !quit )
	{
		while( SDL_PollEvent( &event ) )
		{

			switch( event.type )
			{
				case SDL_VIDEORESIZE:
					screen = SDL_SetVideoMode(event.resize.w, event.resize.h, depth, modes);
					rect.w = event.resize.w;
					rect.h = event.resize.h;  
					SDL_FreeSurface(sframe);
					sframe = createSurface(rect.w,rect.h,24);
					//printf("Resized to %dx%d\n",rect.w,rect.h);
					break;


				case SDL_KEYDOWN:
					switch ( event.key.keysym.sym) {
						case SDLK_F2 :
							saveCurrentImage(video);
							break;
						case SDLK_q : 
							quit=1; 
							continue; 
						case SDLK_1 :
							field=0;
							printf("Displaying field %d\n",field);
							break;
						case SDLK_2 :
							field=(1 >= video.fields())?0:1;
							printf("Displaying field %d\n",field);
							break;
						case SDLK_3 :
							field=(2 >= video.fields())?0:2;
							printf("Displaying field %d\n",field);
							break;
						case SDLK_4 :
							field=(3 >= video.fields())?0:3;
							printf("Displaying field %d\n",field);
							break;
						case SDLK_5 :
							field=(4 >= video.fields())?0:4;
							printf("Displaying field %d\n",field);
							break;
						case SDLK_6 :
							field=(5 >= video.fields())?0:5;
							printf("Displaying field %d\n",field);
							break;
						case SDLK_7 :
							field=(6 >= video.fields())?0:6;
							printf("Displaying field %d\n",field);
							break;
						case SDLK_8 :
							field=(7 >= video.fields())?0:7;
							printf("Displaying field %d\n",field);
							break;
						case SDLK_9 :
							field=(8 >= video.fields())?0:8;
							printf("Displaying field %d\n",field);
							break;
						case SDLK_0 :
							field=(9 >= video.fields())?0:9;
							printf("Displaying field %d\n",field);
							break;
						default :
							break;
					}
					break;

					/*
					   case SDL_MOUSEMOTION:
					   printf("Mouse moved by %d,%d to (%d,%d)\n", 
					   event.motion.xrel, event.motion.yrel,
					   event.motion.x, event.motion.y);
					   break;
					   */

				case SDL_MOUSEBUTTONDOWN:
					printf("Mouse button %d pressed at (%d,%d)\n",
							event.button.button, event.button.x, event.button.y);
					break;

				case SDL_QUIT:
					quit = 1;
					continue;
					break;

				default:
					break;
			}
		}

		if (!video.read(1.0,1)) {
			if (!stalled) printf("Stalled video ?\n");
			stalled = true;
		} else {
			stalled = false;
		}

		src[0] = video.YPlane(field);
		src[1] = video.VPlane(field);
		src[2] = video.UPlane(field);


		if (!noov) {
			SDL_LockYUVOverlay(overlay);
			copyYUV420toOverlay(src,width,height,overlay);
			SDL_DisplayYUVOverlay(overlay,&rect);
			SDL_UnlockYUVOverlay(overlay);
		} else {
			copyYUV420toSurface(src,width,height,frame);
			if (SDL_MUSTLOCK(screen)) { SDL_LockSurface(screen); }
			if ((frame->w!=sframe->w) || (frame->h!=sframe->h)) {
				scale(frame,sframe);
				SDL_BlitSurface(sframe,NULL,screen,NULL);
			} else {
				SDL_BlitSurface(frame,NULL,screen,NULL);
			}
			if (SDL_MUSTLOCK(screen)) { SDL_UnlockSurface(screen); }
			SDL_Flip(screen);
		}

		usleep(50000);

	}

	if (!noov)  {
		SDL_FreeYUVOverlay(overlay);
	} else {
		SDL_FreeSurface(frame);
		SDL_FreeSurface(sframe);
	}
	SDL_FreeSurface(screen);
	SDL_Quit();
}
