/** \file
  Copyright (c) CSIRO ICT Robotics Centre
  \brief  Save Video Stream as JPEG 
  \todo	  EXIF should contain real data
  \todo	  Output should default to stdout
 */

#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>

#include <rtx/defines.h>
#include <rtx/timer.h>
#include <rtx/main.h>
#include <rtx/getopt.h>
#include <rtx/error.h>
#include <rtx/message.h>

#include "DDXStore.h"
#include "DDXVideo.h"
#include "JpegCodec.h"

#include "DDXVideoJpeg.h"

static char rcsid[] RTX_UNUSED = "$Id: DDXVideoJpeg.cpp 3072 2008-05-15 04:48:23Z roy029 $";

#define EXIF "\xff\xe2\x00YDataLen: 00007061\x00\nCamTim: 2004-01-01 Thu 10:11:18\r\nFrmRate: 25\r\nTimStamp: 0000542587\r\n"

int sony = 0;
int field = -1;			// Select field to view (-1 is all) 
int quality = 75;
int dontwait = 0;
int direct = 0;
int loop = 0;
int loadfile = 0;
int grayimages = 0;
int frame_start = 0;
char* control = NULL;
double period = 0.0;
const char *name = "video";
const char *filename = "video.jpg";
const char *help = "CSIRO Video Server Project\nSave video as JPEG"; 
const char *stemplate = "%08d.jpg";

RtxGetopt myOpts[] = {
	{ "quality", "Select encoding quality, in [50-100]",
		{ { RTX_GETOPT_INT, &quality, "quality"}, RTX_GETOPT_END_ARG } },
	{ "fstart", "Frame number to start with",
		{ { RTX_GETOPT_INT, &frame_start, "frame_start"}, RTX_GETOPT_END_ARG } },
	{ "field", "Select Video Field to Display (-1 for all)",
		{ { RTX_GETOPT_INT, &field, "field"}, RTX_GETOPT_END_ARG } },
	{ "gray", "Save images in gray level",
		{ { RTX_GETOPT_SET, &grayimages, "grayimages"}, RTX_GETOPT_END_ARG } },
	{ "sony", "Generate Sony EXIF header",
		{ { RTX_GETOPT_SET, &sony, "sony"}, RTX_GETOPT_END_ARG } },
	{ "dontwait", "Select if read current image, without waiting",
		{ { RTX_GETOPT_SET, &dontwait, "dontwait"}, RTX_GETOPT_END_ARG } },
	{ "name", "Name of Video in Store",
		{ { RTX_GETOPT_STR, &name, "name"}, RTX_GETOPT_END_ARG } },
	{ "filename", "Name of input/output file name",
		{ { RTX_GETOPT_STR, &filename, "filename"}, RTX_GETOPT_END_ARG } },
	{ "loop", "Select if a stream of image must be recorded/read",
		{ { RTX_GETOPT_SET, &loop, "loop"}, RTX_GETOPT_END_ARG } },
	{ "load", "Select if the jpeg must be loaded to the store",
		{ { RTX_GETOPT_SET, &loadfile, "load"}, RTX_GETOPT_END_ARG } },
	{ "direct", "Select direct access to shared memory. Baad",
		{ { RTX_GETOPT_SET, &direct, "direct"}, RTX_GETOPT_END_ARG } },
	{ "period", "Recording period when recording a stream",
		{ { RTX_GETOPT_DOUBLE, &period, "period"}, RTX_GETOPT_END_ARG } },
	{ "template", "Template (sprintf) for output/input file name",
		{ { RTX_GETOPT_STR, &stemplate, "output"}, RTX_GETOPT_END_ARG } },
	{ "control", "Select a variable name to control the recording",
		{ { RTX_GETOPT_STR, &control, "control"}, RTX_GETOPT_END_ARG } },
	RTX_GETOPT_END };

DDXStore store;
DDXVideo video;
DDXVariable controlVar;

void * recording_thread(void * dummy) 
{
	bool verbose = rtx_getopt_get_verbose(0);
	unsigned int frame_num = frame_start;
	JpegCodec * codec = (JpegCodec*)dummy;
	while (1) {
		if (control && controlVar.isRegistered()) {
			DDX_VIDEO_JPEG_CONTROL ctrl;
			controlVar.t_readto(ctrl,1,0);
			if (!ctrl.enable) {
				rtx_timer_sleep(0.01);
				continue;
			}
			codec->setQuality(ctrl.quality);
			period = ctrl.period;
		}
		if (verbose) {
			rtx_message("%s frame %d",loadfile?"loading":"saving",frame_num);
		}

		if (loadfile) {
			char ftname[1024];
			snprintf(ftname,1023,stemplate,frame_num);
			if (!video.isRegistered()) {
				if (!codec->decodeHeader(ftname)) {
					rtx_error_flush("Failed to load header '%s'",ftname);
					goto terminate;
				}
				if (!video.registerAdvancedVariable(store,name,DDXVideo::YUV420,codec->getWidth(),codec->getHeight(),1,direct)) {
					rtx_error_flush("Failed to lookup '%s', trying to lookup",name);
					if (!video.lookupVariable(store,name,direct)) {
						rtx_error_flush("Failed to lookup '%s'",name);
						goto terminate;
					}
					if (!video.setSize(codec->getWidth(),codec->getHeight())) {
						rtx_error_flush("Failed to set video object size %dx%d",codec->getWidth(),codec->getHeight());
						goto terminate;
					}
				}
			}
			if (!codec->decode(ftname, &video, field)) {
				rtx_error_flush("Failed to load '%s'",ftname);
				goto terminate;
			} else {
				video.write();
			}
		} else {
			if (!video.isRegistered()) {
				if (!video.lookupVariable(store,name,direct)) {
					rtx_error_flush("Failed to lookup '%s'",name);
					goto terminate;
				}
				if( field >= (signed)video.fields() )  {
					rtx_error_flush("Insufficent Field Depth");
					goto terminate;
				}
			}
			if (dontwait) {
				video.read(10.0,0);
			} else {
				if (!video.read(10.0,1)) {
					rtx_error_flush("Failed to read '%s'",name);
					goto terminate;
				}
			}

			char ftname[1024];
			snprintf(ftname,1023,stemplate,frame_num);

			if( sony ) {
				unsigned char *p;
				if (!codec->encode(&video,field)) {
					rtx_error_flush("Failed to encode");
					goto terminate;
				}


				p = codec->getBuffer();
				size_t size = codec->bsize();

				FILE * fp = fopen(ftname,"w");
				fwrite( p, 2, 1, fp);
				fwrite( EXIF, 91, 1, fp);
				fwrite( p+2, size-2, 1, fp);
				fclose(fp);
			} else {
				if (!codec->encode(&video,ftname,field)) {
					rtx_error_flush("Failed to save '%s'",ftname);
					goto terminate;
				}
			}
		}
		frame_num += 1;
		rtx_timer_sleep(period);
	}
terminate:
	if (verbose) rtx_message("Working thread shutting down");
	rtx_main_signal_shutdown();
	return NULL;
}

int main(int ac, char* av[]) 
{
	int width,height;
	int verbose;

	if( RTX_GETOPT_CMD( myOpts, ac, av, rcsid, help) == -1 )
	{ RTX_GETOPT_PRINT( myOpts, av[0], rcsid, help); exit(-1); }
	rtx_main_init("DDXVideoJpeg",RTX_ERROR_STDERR|RTX_ERROR_MESSAGE);
	verbose = rtx_getopt_get_verbose(0);
	if (sony) { loadfile = 0; }
	if (loadfile) { 
		dontwait = 0; 
		if (field<0) field = 0;
	}
	if (verbose) rtx_message("Loading file");

	if (loop && (dontwait||loadfile) && (period <= 1e-3)) {
		rtx_message("Warning, period cannot be zero with loop and dontwait/load, set to 10ms");
		period = 0.01;
	}

	if (!store.open(NULL,0,5)) {
		return rtx_error("Failed to open store");
	}

	if (control) {
		if (DDX_STORE_REGISTER_TYPE(store.getId(),DDX_VIDEO_JPEG_CONTROL)) {
			return rtx_error("Failed to register type '%s'","DDX_VIDEO_JPEG_CONTROL");
		}
		if (!store.registerVariable(controlVar,control,"DDX_VIDEO_JPEG_CONTROL")) {
			return rtx_error("Failed to register variable '%s'",control);
		}
		DDX_VIDEO_JPEG_CONTROL ctrl;
		ctrl.enable = 0;
		ctrl.quality = quality;
		ctrl.period = period;
		controlVar.t_writefrom(ctrl);
	}

	JpegCodec codec(quality);
	if (grayimages) {
		codec.setOutputColorSpace(JpegCodec::cmGray);
	} else {
		codec.setOutputColorSpace(JpegCodec::cmAuto);
	}

	if (loop) {
		RtxThread * wthread = rtx_thread_create("recording thread",(verbose > 1), 
				RTX_THREAD_SCHED_OTHER,0,0,
				RTX_THREAD_CANCEL_DEFERRED,recording_thread,&codec,NULL,NULL);
		if (!wthread) return rtx_error("Failed to start recording thread");
		rtx_main_wait_shutdown(0);
		if (verbose) rtx_message("Shutdown received");
		rtx_thread_destroy_sync(wthread);
	} else if( sony ) {
		if (!video.lookupVariable(store,name,direct)) {
			return rtx_error("Failed to lookup '%s'",name);
		}
		if (dontwait) {
			video.read(10.0,0);
		} else {
			if (!video.read(10.0,1)) {
				return rtx_error("Failed to read '%s'",name);
			}
		}
		width  = video.width();
		height = video.height();
		if( field >= (signed)video.fields() ) 
			return rtx_error ("Insufficent Field Depth");


		unsigned char *p;
		if (!codec.encode(&video,field)) 
			return rtx_error("Failed to encode");

		p = codec.getBuffer();
		size_t size = codec.bsize();

		fwrite( p, 2, 1, stdout);
		fwrite( EXIF, 91, 1, stdout);
		fwrite( p+2, size-2, 1, stdout);
	} else {
		if (loadfile) {
			if (!codec.decodeHeader(filename)) {
				return rtx_error("Failed to load header '%s'",filename);
			}
			if (!video.registerAdvancedVariable(store,name,DDXVideo::YUV420,codec.getWidth(),codec.getHeight(),1,direct)) {
				rtx_error_flush("Failed to lookup '%s', trying to lookup",name);
				if (!video.lookupVariable(store,name,direct)) {
					return rtx_error("Failed to lookup '%s'",name);
				}
				if (!video.setSize(codec.getWidth(),codec.getHeight())) {
					return rtx_error("Failed to set video object size %dx%d",codec.getWidth(),codec.getHeight());
				}
			}
			if (!codec.decode(filename, &video, field)) {
				return rtx_error("Failed to load '%s'",filename);
			}
			video.write();
		} else {
			if (!video.lookupVariable(store,name,direct)) {
				return rtx_error("Failed to lookup '%s'",name);
			}
			if (dontwait) {
				video.read(10.0,0);
			} else {
				if (!video.read(10.0,1)) {
					return rtx_error("Failed to read '%s'",name);
				}
			}
			width  = video.width();
			height = video.height();
			if( field >= (signed)video.fields() ) 
				return rtx_error ("Insufficent Field Depth");
			if (!codec.encode(&video,filename,field)) {
				return rtx_error("Failed to save '%s'",filename);
			}
		}
	}

	if (control) {
		controlVar.terminate();
	}
	video.terminate();
	store.close();
}
