#ifndef TASK_DESCRIPTION_H
#define TASK_DESCRIPTION_H


#include <string.h>
#include <rtx/error.h>
#include <rtx/param.h>
#include <rtx/byte.h>
#include <ddx.h>

#ifdef __cplusplus
extern "C" {
#endif

/**
 * Status report for running tasks
 * */
typedef enum {
	TSTATUS_RUNNING=0,      /* task still needs to run */
	TSTATUS_COMPLETED=1,    /* task has completed */
	TSTATUS_TIMEDOUT=2,     /* task has timed out */
	TSTATUS_INTERRUPTED=3,  /* task has been interrupted (FIXME: clear semantic) */
	TSTATUS_ERROR=4         /* task has failed */
} TaskStatus;
	
/** 
 * Render a TaskStatus object as a string
 * */
extern 
const char * task_status_to_string(TaskStatus mt);

DDX_STORE_TYPE(DDX_TASK_DEMAND,
		struct {
			int seqno;  /* the sequence number of the request, Used to distinguish between two task with same id, requested successively */
			int taskId; /* the id of the required task */
			double maxTime; /* maximum allowed time for this task to complete */
			int argc;   /* number of parameters attached to the request */
			double argv[400]; /* the parameters */
		}
);

DDX_STORE_TYPE(DDX_TASK_STATUS,
		struct {
			double runTime; /* the time for which the task has been running */
			int seqno;  /* the sequence number of the running task */
			int taskId; /* the id of the running task */
			int taskStatus; /* the status exported by the running task, must map to TaskStatus */
			char statusStr[128]; /* storage place for a status-related message */
		}
);


/** 
 * Set the task status and status string, and check for the length
 * */
extern
void task_set_status(DDX_TASK_STATUS *status, 
		TaskStatus ts, const char *text);
		

/**
 * Description of a task 
 **/
typedef struct {
	char * name;   /* the name of the task */
	char * help;   /* a help string, useful for the command line control */
	unsigned int minArgs;   /* the minimum number of parameter that this task will require */
	unsigned int maxArgs;   /* the maximum number of parameter that this task will accept, often equal to minArgs */
	/* a function that should get task specific parameters from a parameter file */
	int (*readParams)(void * context, RtxParamStream * pf);
	/* the function called when the task is started */
	int (*init)(void * context, unsigned int argc, const double *argv);
	/* the function called periodically by the task controller */
	int (*run)(void * context, DDX_TASK_STATUS *status, 
			unsigned int argc, const double *argv);
	/* the function called when the task is stopped. */
	int (*terminate)(void * context);
	/* a function that is called to load any dependencies of the task */
	int (*loadDependencies)(const char * plugindir);
	/** 
	 * All the above function are expected to return 0 on success, something else on failure. 
	 * The context variable is a provision for user to pass some system specific
	 * context to the task execution function. The interpretation of this context
	 * is the responsibility of the task implementation. The task scheduler libray
	 * provides a way to pass this context.
	 *
	 * The couple (argc,argv) contains the parameters of the task (speed,
	 * waypoints, ...). argc is the number of provided argument, and argv contains
	 * them. Only double can be used as argument (for a task request to be sent via
	 * the store).
	 *
	 * The status (DDX_TASK_STATUS) can be modified by the task run function to
	 * tell how it goes (see function task_set_status), especially report
	 * completion, timeout or failure. 
	 * */
} TaskDescription;


#define TASK_DECLARATION_END {NULL, NULL, 0,0, NULL,NULL,NULL,NULL,NULL}

/**
 * Register a task plugin for a dynamically loaded task
 * */
#define REGISTER_TASK_PLUGIN(name,help,mina,maxa,readparams,init,run,terminate,loaddep) \
	TaskDescription TASK_PLUGIN = {name,help,mina,maxa,readparams,init,run,terminate,loaddep}

/** 
 * This macro should be used when declaring the context for dynamically loaded
 * modules. It will create a struct with everything put in def, name it id, and
 * record its string description.
 * This string description can then be checked against to verify that the
 * task context has not been changed since compilation
 * Example:
 * REGISTER_TASK_CONTEXT(Context, DDXStore store; double weight;);
 * */
#define REGISTER_TASK_CONTEXT(id, def) \
	static char* task_checksum_string_##id = #def; \
	typedef struct {int task_checksum_##id;def} id

/**
 * Declare a context variable, usually in the main of the task controller. The
 * variable is named var, of type id, and initialised with anything in init.
 * A checksum for the context declaration is also included in the
 * initialisation
 * Example:
 * DECLARE_TASK_CONTEXT(Context, context, store, 1.0);
 * */
#define DECLARE_TASK_CONTEXT(id,var,init...) id var = {\
	rtx_byte_crc16((unsigned char*)task_checksum_string_##id,strlen(task_checksum_string_##id)),\
	init}

/**
 * Check that a pointer on a context variable has the same type as the one used
 * in this tasks. This relies on the context to be declared with
 * DECLARE_TASK_CONTEXT
 * Example:
 * if (CHECK_CONTEXT_CHECKSUM(Context,ctxtp)) { 
 *		return rtx_error("Invalid context");
 * }
 * **/
#define CHECK_CONTEXT_CHECKSUM(id,ctxt) (rtx_byte_crc16((unsigned char*)task_checksum_string_##id,strlen(task_checksum_string_##id)) == ((id*)(ctxt))->task_checksum_##id)


/**
 * Load a task description from a plugin file. Returns a pointer to the task
 * description. The corresponding plugin is opened and kept open forever, so
 * the pointer life should also be "forever"
 * Returns NULL on failure.
 * */
extern
TaskDescription* load_task(const char * pluginfile);

/**
 * Load a all the valid task in directory dirname. Returns a pointer to a newly
 * allocated (malloc) array of task description, terminated by a
 * TASK_DECLARATION_END. Plugins are loaded using the loadTask function, so
 * anything that would prevent this function from succeeding will cause the
 * corresponding task to be ignored.
 * File template provides a way to filter the file to load, it is ignored if
 * NULL. A shell-like pattern is expected otherwise, e.g. "*.so"
 * Returns NULL on failure.
 * */
extern
TaskDescription* load_task_directory(const char * dirname, const char * filetemplate);


/** 
 * Call the readParam function on the task described in dmd. Argument context
 * and pf are passed to the readParam function. The task corresponding to
 * dmd->taskId is looked for in tasks 
 * */
extern 
int task_read_params(void * context, TaskDescription * tasks, DDX_TASK_DEMAND *dmd, RtxParamStream * pf);

/** 
 * Call the init function on the task described in dmd. Argument context
 * is passed to the init function. The task corresponding to
 * dmd->taskId is looked for in tasks 
 * */
extern 
int task_init(void * context, TaskDescription * tasks, DDX_TASK_DEMAND *dmd);

/** 
 * Call the terminate function on the task described in dmd. Argument context
 * is passed to the terminate function. The task corresponding to
 * dmd->taskId is looked for in tasks 
 * */
extern 
int task_terminate(void * context, TaskDescription * tasks, DDX_TASK_DEMAND *dmd);

/**
 * returns the name of the task described in dmd. 
 * The task corresponding to dmd->taskId is looked for in tasks 
 * */
extern 
const char* task_name(TaskDescription * tasks, DDX_TASK_DEMAND *dmd);

/** 
 * Call the terminate function on the task described in dmd. Argument context
 * and status are passed to the run function. The task corresponding to
 * dmd->taskId is looked for in tasks 
 * */
extern 
int task_run(void * context, TaskDescription * tasks, 
		DDX_TASK_DEMAND *dmd, DDX_TASK_STATUS *status);

/**
 * Check if argc is a correct argument number for the task described in dmd.
 * The task corresponding to dmd->taskId is looked for in tasks 
 * */
extern 
int task_check_args_number(TaskDescription * tasks,
		DDX_TASK_DEMAND *dmd, unsigned int argc);

/**
 * Lookup the index of the task whose name is 'name' in the list of tasks
 * 'tasks'. If numTasks is positive, it represents the number of tasks in
 * 'tasks', if not 'tasks' is assumed to be terminated with a task with NULL
 * name.
 * Return the index on succes, negative on failure
 * */
extern
int task_get_index(const TaskDescription * tasks, int numTasks, const char * name);

#ifdef __cplusplus
}
#endif

#endif // TASK_DESCRIPTION_H
