#include <stdlib.h>
#include <stdio.h>

#include <rtx/error.h>
#include <rtx/message.h>

#include <string.h>
#include "task.h"

void task_set_status(DDX_TASK_STATUS *status, 
		TaskStatus ts, const char *text)
{
	status->taskStatus = ts;
	strncpy(status->statusStr,text,127);
	status->statusStr[127]=0;
}
		
const char * task_status_to_string(TaskStatus mt)
{
	switch (mt) {
		case TSTATUS_RUNNING: return "Running";
		case TSTATUS_COMPLETED: return "Completed";
		case TSTATUS_ERROR: return "Error";
		case TSTATUS_TIMEDOUT: return "Timed-out";
		case TSTATUS_INTERRUPTED: return "Interrupted";
		default: break;
	}
	return NULL;
}

int task_get_index(const TaskDescription * tasks, int numTasks, const char * name)
{
	unsigned int i = 0;
	while (1) {
		if ((numTasks>=0) && (i>=numTasks)) break;
		if (tasks[i].name == NULL) break;
		if (strcasecmp(tasks[i].name,name)==0)
			return i;
		i += 1;
	}
	return rtx_error("task_get_index: can't find task '%s'",name);
}

const char* task_name(TaskDescription * tasks, DDX_TASK_DEMAND *dmd)
{
	return tasks[dmd->taskId].name;
}


int task_init(void * context, TaskDescription * tasks, DDX_TASK_DEMAND *dmd)
{
	/* no check of arg length here since it has been done 
	 * at initialisation */
	
	if (task_check_args_number(tasks,dmd,dmd->argc)) 
		return rtx_error("Invalid number of argument %d for task %d:%s\n",
				dmd->argc,dmd->seqno,task_name(tasks,dmd));

	if (tasks[dmd->taskId].init)
		return tasks[dmd->taskId].init(context, dmd->argc,dmd->argv);
	return 0;
}


int task_read_params(void * context, TaskDescription * tasks, DDX_TASK_DEMAND *dmd,
		RtxParamStream * pf)
{
	if (tasks[dmd->taskId].readParams)
		return tasks[dmd->taskId].readParams(context, pf);
	return 0;
}

int task_terminate(void * context, TaskDescription * tasks, DDX_TASK_DEMAND *dmd)
{
	/* no check of arg length here since it has been done 
	 * at initialisation */
	
	if (tasks[dmd->taskId].terminate)
		return tasks[dmd->taskId].terminate(context);
	return 0;
}

int task_run(void * context, TaskDescription * tasks, 
		DDX_TASK_DEMAND *dmd, DDX_TASK_STATUS *status)
{
	/* no check of arg length here since it has been done 
	 * at initialisation */
	
	if (tasks[dmd->taskId].run==NULL)
		return rtx_error("Invalid NULL run function for task %d:%s\n",
				dmd->seqno,task_name(tasks,dmd));
	return tasks[dmd->taskId].run(context,status,dmd->argc,dmd->argv);
}

int task_check_args_number(TaskDescription * tasks,
		DDX_TASK_DEMAND *dmd, unsigned int argc)
{
	if ((argc < tasks[dmd->taskId].minArgs) ||
			(argc > tasks[dmd->taskId].maxArgs))
		return 1;
	return 0;
}


/******************** PLUGIN *************************/

#include <dlfcn.h>

TaskDescription* load_task(const char * pluginfile)
{
	void * handle = dlopen(pluginfile,RTLD_NOW|RTLD_LOCAL|RTLD_NODELETE);
	TaskDescription *result = NULL; 
	if (!handle) {
		return rtx_error_null("Cannot load task description in '%s': %s",pluginfile,dlerror());
	}
	result = (TaskDescription*)dlsym(handle,"TASK_PLUGIN");
	if (!result) {
		return rtx_error_null("Cannot find task description in '%s': %s",pluginfile,dlerror());
	}
	dlclose(handle);
	rtx_message("Loaded task plugin '%s'",result->name);
	return result;
}

#include <sys/types.h>
#include <dirent.h>
#include <fnmatch.h>


TaskDescription* load_task_directory(const char * dirname, const char * filetemplate)
{
	unsigned int ntasks = 0, sizetasks = 20, errcnt = 0;
	TaskDescription end = TASK_DECLARATION_END;
	TaskDescription *tasks;
	DIR *dd; /* dir descriptor */
	struct dirent  *dirp; /* dir pointer */
	unsigned int fnamesize=0,dlen;
	char * filename = NULL;

	dlen = strlen(dirname);
	
	dd = opendir(dirname);
	if (dd == NULL) {
		return rtx_error_null("Cannot load directory '%s'", dirname);
	}

	tasks = (TaskDescription*)malloc(sizetasks * sizeof(TaskDescription));
	if (!tasks) {
		return rtx_error_null("Error allocating memory for the task array");
	}


	/* iterate through all items in the directory */
	while((dirp = readdir(dd)) != NULL) {
		TaskDescription *thistask;
		unsigned int len;

		if (filetemplate && fnmatch(filetemplate,dirp->d_name,FNM_PATHNAME)) {
			continue;
		}

		len = dlen + 1 + strlen(dirp->d_name) + 1;
		if (len > fnamesize) {
			filename = (char*)realloc(filename,len);
			if (!filename) {
				free(tasks);
				return rtx_error_null("Error reallocating memory for the plugin file name");
			}
		}
		sprintf(filename,"%s/%s",dirname,dirp->d_name);

		thistask = load_task(filename);
		if (thistask) {
			if (thistask->loadDependencies && thistask->loadDependencies(dirname)) {
				rtx_error("Failed to load dependencies for task '%s' in directory '%s'",
						thistask->name, dirname);
				errcnt += 1;
				continue;
			} 
			if (ntasks >= sizetasks) {
				sizetasks += 20;
				tasks = (TaskDescription*)realloc(tasks,sizetasks * sizeof(TaskDescription));
				if (!tasks) {
					free(filename);
					return rtx_error_null("Error reallocating memory for the task array");
				}
			}
			memcpy(tasks+ntasks,thistask,sizeof(TaskDescription));
			ntasks += 1;
		} else { 
			errcnt += 1;
		}
	}
	closedir(dd);
	free(filename);

	/* if errcnt is non null, some of the loadTask call failed. There may be
	 * some error message in the pipe, just flush them */
	if (errcnt) {
		rtx_error_flush("Warning some errors detected when loading task directory");
	}

	/* resize the array to be exactly the right size. This is most likely to
	 * have no effect, but... */
	tasks = (TaskDescription*)realloc(tasks,(ntasks+1) * sizeof(TaskDescription));
	if (!tasks) {
		return rtx_error_null("Error reallocating memory for the task array");
	}
	/* terminate by putting a null class at the end of the list */
	memcpy(tasks+ntasks,&end,sizeof(TaskDescription));

	return tasks;
}















