#ifndef TASK_SCHEDULER_H
#define TASK_SCHEDULER_H


#include <rtx/inet.h>
#include <rtx/timer.h>
#include <rtx/thread.h>
#include <rtx/param.h>
#include <rtx/sync.h>
#include <ddx.h>
#include "task.h"

#ifdef __cplusplus
extern "C" {
#endif

/**
 * Specification of a function to be called on watchdog trigger 
 * */
typedef void (*TaskSchedulerWatchdog)(void * scheduler);

/**
 * Specification of a function that will be called at every
 * iteration to check for system health. Must return 0 on 
 * success, !0 on failure (e.g. rtx_error)
 * */
typedef int (*TaskSchedulerSystemCheck)(void * context);

/**
 * private types used in the scheduler state machine 
 * */
typedef enum {TS_INIT,TS_RUN,TS_TERM} SCHED_STATE;
typedef enum {TSM_RUNNING,TSM_IDLE,TSM_FAILED} SCHED_MODE;

typedef void (*UserDataGenerator)(RtxInetConn*,void*); 

/**
 * All the data required by the task scheduler 
 * */
typedef struct {
	/* private */
	int mallocd;                          /* has this object been malloc'ed */
	int verbose;                          /* verbosity level */
	int timeoutPrinted;                   /* internal flag to avoid clogging the display with repeated messages */
	double lastInitTime,                  /* internal time to record when a request was last received, and when a */
		   lastDemandTime,                /* verbose message was last printed */
		   lastPrintTime;  
									      
	SCHED_STATE state;                    /* internal state of the scheduler state machine (init,run,term) */
	SCHED_MODE mode;                      /* internal mode of the scheduler state machine (idle, run, failed) */
	DDX_TASK_STATUS status;               /* internal record of the last status of a running task */

	/* protected */
	DDX_STORE_ITEM * demandItem;          /* pointer on the store variable for the task requests */
	DDX_STORE_ITEM * statusItem;          /* pointer on the store variable for the task status */

	
	double period;                        /* task running period */
	double watchdogPeriod;                /* watchdog checking period, maximum allowed age of a command */
	unsigned int prio;                    /* task thread priority */
	int watchdogCounter, lastWatchdogCounter; /* watchdog counter and last one seen, to detect change */
	
	TaskSchedulerWatchdog wdfunc;         /* function to be called when watchdog triggers */
	RtxSync * sync;                       /* synchronisation object for task running thread */
	RtxTimerThread * clock;               /* clock thread descriptor */
	RtxThread * scheduler;                /* scheduler thread descriptor */
	RtxTimerThread * watchdog;            /* watchdog thread descriptor */
	TaskSchedulerSystemCheck checkfunc;   /* function called to check system healt */
	/* generator for user data in xml form, must write to the socket
	 * passed as first argument, received context as its second arg*/
	UserDataGenerator userdata_gen; 
	RtxInet * server;

	unsigned int runningTask;             /* id of task currently executed */
	unsigned int commandedTask;           /* id of task currently commanded (running can be idle) */
	int runningSeqNo;                     /* seq number of task currently executed */
	int commandedSeqNo;                   /* seq number of task currently commanded (running can be idle) */
	unsigned int failedTask;              /* id of last task having failed */
	int failedSeqNo;                      /* seq number of last task having failed */

	unsigned int numTasks;                /* number of task managed by the scheduler */
	unsigned int idleTask;                /* id of the idle task (executed when nothing else to do) */
	unsigned int failureTask;             /* id of the failure task (executed when an error condition has occured) */
	TaskDescription * tasks;              /* list of tasks managed by the scheduler */

	/* public */
	void * context;                       /* user data: used in task function and in watchdog function */ 
} TaskScheduler;


/**
 * Initialise the task scheduler internal data, but do not start it
 * All data passed is copied internally except the context and store
 * */
extern
TaskScheduler * task_scheduler_init(
		TaskScheduler * ts,             /* if we don't want to malloc the structure */
		int verbose,                    /* augment tracing info */
		void * context,                 /* the context pointer */
		DDX_STORE_ID * store,           /* the store (DDX_STORE_ID*) pointer */
		const char * command,           /* the name of the store variable for task request */
		const char * status,            /* the name of the store variable for task status */
		const TaskDescription * tasks,  /* the list of all registered task */ 
		unsigned int idleTask,          /* the index of the task used when idling */
		unsigned int failureTask        /* the index of the task used after a failure */
);

/**
 * Register a system checking function. This function will be called at each
 * iteration, with context as argument. If it returns a failure (!0, rtx_error)
 * then the current task will be declared failed.
 * */
extern
int task_scheduler_register_syscheck(
		TaskScheduler *ts,            /* the scheduler object */
		TaskSchedulerSystemCheck sc   /* the system checking function, may be null */
);

/**
 * Start the scheduler server on port port. This scheduler will answer
 * connection by outputing a task description
 * */
extern
int task_scheduler_start_server(
		TaskScheduler * sched,    /* the scheduler object */
		unsigned int port,         /* the TCP port to listen onto */
		UserDataGenerator user_data_gen /* a user data generator */
);                                

/**
 * Start the scheduler threads, using the idle task specified above 
 * */
extern
int task_scheduler_start(
		TaskScheduler * sched,    /* the scheduler object */
		double period,            /* the task run period */
		int prio,                 /* the priority of the task scheduler thread */
		double watchdog_period,   /* the watchdog timeout */
		TaskSchedulerWatchdog wdf /* a function called if the watchdog get triggered, 
									 rtx_main_signal_shutdown if NULL */
);                                


/**
 * Pass the parameter file pf to all the task in sched 
 * */
extern
int task_scheduler_set_tasks_params(TaskScheduler * sched,
		RtxParamStream * pf);

/**
 * Stop the scheduler but do not release memory.
 * */
extern
int task_scheduler_stop(TaskScheduler * sched);

/**
 * Stop the scheduler and release memory (whenever appropriate).
 * */
extern
int task_scheduler_terminate(TaskScheduler * sched);

#ifdef __cplusplus
}
#endif


#endif // TASK_SCHEDULER_H
