#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <ctype.h>

#include <sys/socket.h>
#include <netinet/tcp.h>
#include <netdb.h>



#include <rtx/error.h>
#include <rtx/message.h>
#include <rtx/main.h>

#include "task-scheduler.h"

#ifndef RTX_THREAD_PRIO_MIN
#define RTX_THREAD_PRIO_MIN 0
#endif


/**
 * Initialise the task scheduler internal data, but do not start it
 * All data passed is copied internally except the context and store
 * */
TaskScheduler * task_scheduler_init(
		TaskScheduler * ts,             /* if we don't want to malloc the structure */
		int verbose,                    /* augment tracing info */
		void * context,                 /* the context pointer */
		DDX_STORE_ID * store,           /* the store (DDX_STORE_ID*) pointer */
		const char * command,           /* the name of the store variable for task request */
		const char * status,            /* the name of the store variable for task status */
		const TaskDescription * tasks,  /* the list of all registered task */ 
		unsigned int idleTask,          /* the index of the task used when idling */
		unsigned int failureTask        /* the index of the task used after a failure */
)
{
	DDX_TASK_DEMAND dmd;

	dmd.seqno = -1;
	dmd.taskId = idleTask;
	dmd.argc = 0;

	TaskScheduler * res = ts;
	unsigned int i;
	if (res) {
		res->mallocd = 0;
	} else {
		res = (TaskScheduler*)malloc(sizeof(TaskScheduler));
		if (!res) return rtx_error_null("task_scheduler_init: malloc failed");
		res->mallocd = 1;
	}
	res->server = NULL;
	res->userdata_gen = NULL;
	res->checkfunc = NULL;
	res->lastWatchdogCounter = -1;
	res->lastInitTime = -1;
	res->lastPrintTime = -1;
	res->timeoutPrinted = 1;
	res->context = context;
	res->verbose = verbose;
	res->demandItem = NULL;
	res->statusItem = NULL;
	res->tasks = NULL;
	res->commandedTask = idleTask;
	res->commandedSeqNo = -1;
	res->runningTask = idleTask;
	res->runningSeqNo = -1;
	res->state = TS_INIT;
	res->mode = TSM_IDLE;

	res->lastDemandTime = -1;
	res->period = 0;
	res->prio = 0;
	res->watchdogPeriod = 0;
	res->watchdogCounter = 0;
	res->wdfunc = NULL;

	res->clock = NULL;
	res->scheduler = NULL;
	res->watchdog = NULL;

	res->sync = rtx_sync_init(NULL,0,0,0);
	if (!res->sync) {
		task_scheduler_terminate(res);
		return rtx_error_null("task_scheduler_init: sync init failed");
	}

	res->idleTask = idleTask;
	res->failureTask = failureTask;
	res->numTasks = 0;
	while (tasks[res->numTasks].name != NULL) {
		res->numTasks += 1;
	}
	res->tasks = (TaskDescription*)malloc(sizeof(TaskDescription)*res->numTasks);
	if (!res->tasks) {
		task_scheduler_terminate(res);
		return rtx_error_null("task_scheduler_init: task malloc failed");
	}

	for (i=0;i<res->numTasks;i++) {
		memcpy(res->tasks+i,tasks+i,sizeof(TaskDescription));
		res->tasks[i].name = strdup(tasks[i].name);
		res->tasks[i].help = strdup(tasks[i].help);
	}

	if (DDX_STORE_REGISTER_TYPE(store,DDX_TASK_DEMAND)) {
		task_scheduler_terminate(res);
		return rtx_error_null("task_scheduler_init: failed to register command type");
	}
	if (DDX_STORE_REGISTER_TYPE(store,DDX_TASK_STATUS)) {
		task_scheduler_terminate(res);
		return rtx_error_null("task_scheduler_init: failed to register status type");
	}
		
	res->demandItem = ddx_store_lookup_item(store,command,"DDX_TASK_DEMAND",0);
	if (!res->demandItem) {
		task_scheduler_terminate(res);
		return rtx_error_null("task_scheduler_init: failed to register command variable");
	}

	res->statusItem = ddx_store_lookup_item(store,status,"DDX_TASK_STATUS",0);
	if (!res->statusItem) {
		task_scheduler_terminate(res);
		return rtx_error_null("task_scheduler_init: failed to register status variable");
	}

	res->status.seqno = -1;
	res->status.runTime = -1;
	res->status.taskId = idleTask;
	res->status.taskStatus = TSTATUS_RUNNING;
	res->status.statusStr[0] = 0;


	//ddx_store_write(res->demandItem, &dmd,NULL);
	ddx_store_write(res->statusItem, &res->status,NULL);

	return res;

}

/**
 * Internal function whose only role is to trigger a sync (POSIC condition)
 * */
static
void clock_func(void * arg) {
	TaskScheduler * ts = (TaskScheduler*)arg;
	rtx_sync_broadcast(ts->sync);
}

/**
 * Actual implementation of the task watch dog. Each time the task scheduler is
 * run, it increments the watchdogCounter variable in the TaskScheduler object.
 * The watchdog just check that this value changed between each call
 * */
static 
void watchdog_func(void * arg) {
	TaskScheduler * ts = (TaskScheduler*)arg;
	if (ts->lastWatchdogCounter == ts->watchdogCounter) {
		if (ts->wdfunc) {
			rtx_message("task scheduler thread seems stopped, running user function");
			ts->wdfunc(ts);
		} else {
			rtx_message("task scheduler thread seems stopped, signalling shutdown");
			rtx_main_signal_shutdown();
		}
	}
	ts->lastWatchdogCounter = ts->watchdogCounter;
}

/**
 * Internal function, real meaty implementation of the task scheduling. The
 * function is quite complex but the principle can be described as a cartesian
 * product of 2 state automatons:
 * 1: for any given task, there is a three state automation:
 * Init -> Run -> Terminate
 * The init state is instantaneous, where as the automaton stays in the run
 * state until the task fails or report itself completed
 * The Terminate state is also instantaneous
 * 2: The system can be in three states: 
 * Idle <-> Normal <-> Failed
 * We go from Idle to Normal on reception of a new task request
 *            Normal to Idle on completion of the current task
 *            Normal to Failed if anything fails in the current task
 *            Failed to Normal on reception of a new task after failure (this
 *            is semantically arguable).
 * We then have to handle funky situation where a new task is requested while
 * running the current one.
 * Each task is identified with a task id and a sequence number, to identify
 * receiving two time the same task but with possibly different parameters
 * **/
static 
int task_scheduler_thread_once(TaskScheduler *ts)
{
	DDX_TASK_STATUS sts;
	DDX_TASK_DEMAND dmd;
	RtxTime dts;
	int waitNextCycle = 0;
	int ret,dmdTimeout,newDemand;
	double tDemand;

	/* reading demand */
	ret = ddx_store_read(ts->demandItem,&dmd,&dts,ts->watchdogPeriod,0);
	if (ret < 0)
		return rtx_error("ddx_store_read(demand) failed");

	/** If the demand is too old, we switch to the Idle state. This is a
	 * semantic choice for the scheduler, but this seems to be a sensible
	 * choice for an experimental UAV
	 */
	dmdTimeout = (ret > 0);
	if (dmdTimeout) {
		if (!ts->timeoutPrinted && ts->verbose && (ts->mode != TSM_FAILED))  {
			rtx_message("Timeout while reading demand, switching to idle");
		}
		ts->timeoutPrinted = 1;
		ts->commandedSeqNo = -1;
	} else {
		ts->timeoutPrinted = 0;
	}

	/** We test if we have a new demand: different from current one, 
	 * or simply new in Idle mode */
	tDemand = rtx_time_to_double(&dts);
	newDemand = (tDemand > ts->lastDemandTime) 
		&& ((dmd.taskId != ts->commandedTask) || (dmd.seqno != ts->commandedSeqNo));
	ts->lastDemandTime = tDemand;




	/** The outer loop here provides a way to do all possible computation in
	 * one time step. For instance, if a task reports its completion, we need
	 * to terminate it, init the Idle task and start running the idle task.
	 * There is no need to wait between this steps. On the other hand, once we
	 * are in a Run state, then we have to wait the next clock tick before
	 * running again, then waitNextCycle is set to 1
	 * */
	waitNextCycle = 0;
	while (!waitNextCycle) {
		RtxTime currtime;
		if (ts->verbose>1) {
			rtx_message("TS: new %d timout %d seq %d",newDemand,dmdTimeout,ts->runningSeqNo);
			rtx_message("TS: mode %d state %d dsq %d",ts->mode,ts->state,dmd.seqno);
		}
		rtx_time_get(&currtime);
		/** ts->state is the state if the automaton 1: init,run,terminate */
		switch (ts->state) {
			case TS_INIT:
				ts->lastInitTime = rtx_time_to_double(&currtime);
				/** ts->mode is the state of automaton 2:
				 * idle,normal/running,failure*/
				switch (ts->mode) {
					/* In idle or failed mode, we don't check the returns of
					 * the init functions. We can't really do anything about a
					 * failure in this mode
					 * */
					case TSM_IDLE:
						if (ts->tasks[ts->idleTask].init)
							if (ts->tasks[ts->idleTask].init(ts->context,0,NULL)) {
								rtx_error_flush("task idle init failed, this should not happen!");
							}
						ts->state = TS_RUN;
						break;
					case TSM_FAILED:
						if (ts->tasks[ts->failureTask].init)
							if (ts->tasks[ts->failureTask].init(ts->context,0,NULL)) {
								rtx_error_flush("task failure init failed, this should not happen!");
							}
						ts->state = TS_RUN;
						break;
					/* In running mode, then we have first to validate that the
					 * task id is correct, and switch to failure mode if not */
					case TSM_RUNNING:
						ts->commandedTask = dmd.taskId;
						ts->commandedSeqNo = dmd.seqno;
						if (dmd.taskId >= ts->numTasks) {
							rtx_message("TS: invalid task id");
							ts->failedTask = dmd.taskId;
							ts->failedSeqNo = dmd.seqno;
							ts->mode = TSM_FAILED;

							sts.taskId = dmd.taskId;
							sts.seqno = dmd.seqno;
							sts.runTime = -1;
							task_set_status(&sts,TSTATUS_ERROR,"Invalid task id");
							ts->status = sts;
							break;
						}

						if (ts->verbose) {
							rtx_message("TS: starting task %d : %s",
									dmd.seqno,task_name(ts->tasks,&dmd));
						}
						/** The task is valid so we can call its init function
						 */
						ret = task_init(ts->context,ts->tasks,&dmd);
						/** Again the error is handled by switching to the
						 * failure mode */
						if (ret) {
							rtx_error_flush("TS: init error detected");
							rtx_message("TS: task %d:%s failed to initialise",
									dmd.taskId,task_name(ts->tasks,&dmd));
							ts->failedTask = dmd.taskId;
							ts->failedSeqNo = dmd.seqno;
							ts->mode = TSM_FAILED;

							sts.taskId = dmd.taskId;
							sts.seqno = dmd.seqno;
							sts.runTime = -1;
							task_set_status(&sts,TSTATUS_ERROR,"Init failure");
							ts->status = sts;
							break;
						} else {
							rtx_message("Successfully started task %d:%s",dmd.taskId,task_name(ts->tasks,&dmd));
						}
						ts->runningTask = dmd.taskId;
						ts->runningSeqNo = dmd.seqno;
						ts->state = TS_RUN;
						break;
					default:
						return rtx_error("task_scheduler_thread_once: invalid mode");
				}
				break;

			case TS_RUN:
				/* set the status of the task to something correct
				 * so the run function of the task can rely on it */
				sts.taskId = dmd.taskId;
				sts.seqno = dmd.seqno;
				sts.taskStatus = TSTATUS_RUNNING;
				sts.statusStr[0] = 0;
				sts.runTime = rtx_time_to_double(&currtime) - ts->lastInitTime;
				switch (ts->mode) {
					/** Running the idle task: no checking of failure, but we
					 * need to check that we don't have a new task to run
					 * */
					case TSM_IDLE:
						ts->runningTask = ts->idleTask;
						ts->runningSeqNo = -1;
						if (!dmdTimeout && newDemand) { 
							ts->mode = TSM_RUNNING;
							ts->state = TS_TERM;
							rtx_message("New demand in idle mode");
							break;
						}
						if (ts->tasks[ts->idleTask].run)
							ts->tasks[ts->idleTask].run(ts->context,&sts,0,NULL);
						waitNextCycle = 1;
						break;
					/** Running the failure task, similar to the idle task,
					 * Currently it is possible to ge out of failure mode. 
					 * TODO: discuss if this is the correct semantic
					 * */
					case TSM_FAILED:
						ts->runningTask = ts->failureTask;
						ts->runningSeqNo = -1;
#if 1 // set to 0 to prevent exit from failure mode
						if (!dmdTimeout && newDemand &&
								((dmd.taskId != ts->failedTask)||(dmd.seqno != ts->failedSeqNo))) { 
							ts->mode = TSM_RUNNING;
							ts->state = TS_TERM;
							rtx_message("New demand in idle mode");
							break;
						}
#endif
						if (ts->tasks[ts->failureTask].run)
							ts->tasks[ts->failureTask].run(ts->context,&sts,0,NULL);
						waitNextCycle = 1;
						break;
					/**
					 * Running a user-requested task. This case is big, mainly
					 * due to error checking/handling
					 * */
					case TSM_RUNNING:
						/** Check that the client is still requesting this task
						 * */
						if (dmdTimeout) {
							if (ts->commandedTask == ts->failureTask) {
								ts->mode = TSM_FAILED;
							} else {
								ts->mode = TSM_IDLE;
								ts->commandedTask = ts->idleTask;
							}
							ts->state = TS_TERM;
							ts->commandedSeqNo = -1;
							rtx_message("Demand timeout in run mode");
							break;
						}
						/**
						 * Check is there was a timeout associated with this
						 * task
						 * */
						if ((dmd.maxTime>0) && (sts.runTime > dmd.maxTime)) {
							rtx_message("TS: task %d:%s timedout ",
									dmd.taskId,task_name(ts->tasks,&dmd));
							task_set_status(&sts,TSTATUS_TIMEDOUT,
									"timedout due to dmd.maxTime");
							ts->status = sts;
							ts->mode = TSM_FAILED;
							ts->state = TS_TERM;
							break;
						}
						
						/** Check if a new task request has been received */
						if ((dmd.taskId != ts->runningTask) || (dmd.seqno != ts->runningSeqNo)) {
							rtx_message("TS: task %d:%s premature completion ",
									ts->runningSeqNo,ts->tasks[ts->runningTask].name);
							ts->state = TS_TERM;
							break;
						}

						/** Run the system checking function, if it has been
						 * defined */
						ret = 0;
						if (ts->checkfunc) {
							ret = ts->checkfunc(ts->context);
						}
						if (ret) {
							/* system checker found a fault, report and fail */
							task_set_status(&sts,TSTATUS_ERROR,"System error detected by user function");
						} else {
							/** Here we run the proper task function */
							ret = task_run(ts->context,ts->tasks,&dmd,&sts);
						}

						/* just in case these have been tampered */
						sts.taskId = dmd.taskId;
						sts.seqno = dmd.seqno;
						sts.runTime = rtx_time_to_double(&currtime) - ts->lastInitTime;
						ts->status = sts;

						if (ret) {
							rtx_error_flush("TS: error detected");
							if (ts->status.taskStatus == TSTATUS_RUNNING) {
								task_set_status(&ts->status,TSTATUS_ERROR,"unreported error");
							}
							rtx_message("TS: task %d:%s failed ",
									dmd.taskId,task_name(ts->tasks,&dmd));
							ts->mode = TSM_FAILED;
							ts->state = TS_TERM;
							break;
						}

						/** Handle various task status */
						switch (sts.taskStatus) {
							/** These are "normal" terminations. I know it
							 * sounds strange to have "INTERRUPTED" as a normal
							 * termination. This is arguable, but it suited
							 * me... */
							case TSTATUS_INTERRUPTED:
							case TSTATUS_COMPLETED:
								ts->mode = TSM_IDLE;
								ts->state = TS_TERM;
								if (ts->verbose)
									rtx_message("Task %d:%s: %s",dmd.seqno,
											ts->tasks[dmd.taskId].name,task_status_to_string(sts.taskStatus));
								break;

							/** These are failure termination */
							case TSTATUS_ERROR:
							case TSTATUS_TIMEDOUT:
								ts->mode = TSM_FAILED;
								ts->state = TS_TERM;
								if (ts->verbose)
									rtx_message("Task %d:%s: %s",dmd.seqno,
											ts->tasks[dmd.taskId].name,task_status_to_string(sts.taskStatus));
								break;
							/** Otherwise, the task did not terminate, keep
							 * going */
							default:
								waitNextCycle = 1;
								if (ts->verbose>1)
									rtx_message("Task %d:%s: %s",dmd.seqno,
											ts->tasks[dmd.taskId].name,task_status_to_string(sts.taskStatus));
						}
						break;
					default:
						return rtx_error("task_scheduler_thread_once: invalid mode");
				}
				break;

			/** The terminate case is simple because independent of the state
			 * of automation 2. In any case, we terminate the current task, and
			 * there is no real point for error checking since we can't do
			 * anything about it.
			 * TODO: discuss if this is the correct semantic.
			 * */
			case TS_TERM:
				if (ts->tasks[ts->runningTask].terminate)
					ts->tasks[ts->runningTask].terminate(ts->context);
				ts->state = TS_INIT;
				break;


			default:
				return rtx_error("task_scheduler_thread_once: invalid state");

		}

		newDemand = 0;
	}
	/** At this point, we ran all the possible state transition in the
	 * automatons. We can write the task status to the store */
	ddx_store_write(ts->statusItem,&ts->status,NULL);

	/** This may be incremented twice, but as long as it moves, the watchdog
	 * will be happy. */
	ts->watchdogCounter += 1;

	if (ts->verbose>1) {
		RtxTime rts;
		double t;
		rtx_time_get(&rts);
		t = rtx_time_to_double(&rts);
		if (t - ts->lastPrintTime > 1.0) {
			if (ts->verbose) {
				rtx_message("Scheduler: task %03d:%s : %s\n",ts->status.seqno,
						ts->tasks[ts->runningTask].name,
						task_status_to_string(ts->status.taskStatus));
			}
			ts->lastPrintTime = t;
		}
	}

	return 0;
}

/** 
 * Implementation of the task scheduling thread. The principle is simple:
 * forever, wait for a tick from the clock timer, then run the
 * task_scheduler_thread_once function and increment the watchdog timer
 * **/
static
void * task_scheduler_thread(void * arg)
{
	TaskScheduler * ts = (TaskScheduler*)arg;

	while (1) {
		if (rtx_sync_wait(ts->sync)<0) {
			rtx_message("task_scheduler_thread: failure in rtx_sync_wait");
			rtx_main_signal_shutdown();
			return NULL;
		}

		if (task_scheduler_thread_once(ts)) {
			rtx_message("task_scheduler_thread: failure detected in main loop");
			rtx_main_signal_shutdown();
			return NULL;
		}
		
		ts->watchdogCounter += 1;
	}
	return NULL;
}

/**
 * Start the scheduler threads, using the idle task specified above in
 * task_scheduler_init 
 * */
int task_scheduler_start(
		TaskScheduler * sched,    /* the scheduler object */
		double period,            /* the task run period */
		int prio,                 /* the priority of the task scheduler thread */
		double watchdog_period,   /* the watchdog timeout */
		TaskSchedulerWatchdog wdf /* a function called if the watchdog get triggered, 
									 rtx_main_signal_shutdown if NULL */
)                                
{
	if (!sched) 
		return rtx_error("task_scheduler_start: Invalid NULL scheduler object");
	
	if (sched->scheduler) 
		return rtx_error("task_scheduler_start: Already started");

	sched->period = period;
	sched->prio = prio;
	sched->watchdogPeriod = watchdog_period;
	sched->wdfunc = wdf;

	sched->scheduler = NULL;
	sched->clock = NULL;
	sched->watchdog = NULL;

	/** Create the main thread **/
	sched->scheduler = rtx_thread_create("Task Scheduler", 0,
			RTX_THREAD_SCHED_OTHER, RTX_THREAD_PRIO_MIN, 0,
			RTX_THREAD_CANCEL_DEFERRED, task_scheduler_thread,
			sched, NULL, NULL);
	if (!sched->scheduler) {
		task_scheduler_stop(sched);
		return rtx_error("task_scheduler_start: failed to start main thread");
	}

	/** Start the timer thread that will trigger the main thread **/
	sched->clock = rtx_timer_create_thread(period,period,RTX_THREAD_PRIO_MIN,
			clock_func,sched,RTX_TIMER_CLOCK_REALTIME);
	if (!sched->clock) {
		task_scheduler_stop(sched);
		return rtx_error("task_scheduler_start: failed to start clock thread");
	}

	/** Start the watchdog timer thread. To let the system start at its own
	 * speed, the watchdog wait 3 main thread period before starting
	 * */
	sched->watchdog = rtx_timer_create_thread(3*period,watchdog_period,
			RTX_THREAD_PRIO_MIN,
			watchdog_func,sched,RTX_TIMER_CLOCK_REALTIME);
	if (!sched->watchdog) {
		task_scheduler_stop(sched);
		return rtx_error("task_scheduler_start: failed to start watchdog thread");
	}

	return 0;
}

/**
 * Pass the parameter file pf to all the task in sched 
 * */
int task_scheduler_set_tasks_params(TaskScheduler * sched,
		RtxParamStream * pf)
{
	unsigned int i;
	for (i=0;i<sched->numTasks;i++) {
		if (!sched->tasks[i].readParams) 
			continue;
		if (sched->tasks[i].readParams(sched->context,pf))
			return rtx_error("Error while reading parameters for task %s",
					sched->tasks[i].name);
	}
	return 0;
}


/**
 * Stop the scheduler but do not release memory.
 * */
int task_scheduler_stop(TaskScheduler * sched)
{
	if (sched->watchdog)
		rtx_timer_destroy_thread(sched->watchdog);
	sched->watchdog = NULL;
	if (sched->clock)
		rtx_timer_destroy_thread(sched->clock);
	sched->clock = NULL;
	if (sched->scheduler)
		rtx_thread_destroy_sync(sched->scheduler);
	sched->scheduler = NULL;
	if (sched->sync)
		rtx_sync_unlock(sched->sync);
	if (sched->tasks[sched->runningTask].terminate) 
		sched->tasks[sched->runningTask].terminate(sched->context);
	return 0;
}

/**
 * Stop the scheduler and release memory (whenever appropriate).
 * */
int task_scheduler_terminate(TaskScheduler * sched)
{
	/* Stop the scheduler in case it is required */
	if (task_scheduler_stop(sched))
		rtx_message("task_scheduler_terminate: task_scheduler_stop failed");

	if (sched->server) {
		rtx_message("Terminating inet server");
		rtx_inet_done(sched->server);
	}

	if (sched->demandItem)
		ddx_store_done_item(sched->demandItem);
	if (sched->statusItem)
		ddx_store_done_item(sched->statusItem);
	if (sched->tasks) {
		unsigned int i;
		for (i=0;i<sched->numTasks;i++) {
			free(sched->tasks[i].name);
			free(sched->tasks[i].help);
		}
		free(sched->tasks);
	}
	if (sched->sync)
		rtx_sync_destroy(sched->sync);
	if (sched->mallocd) {
		free(sched);
	} else {
		bzero(sched,sizeof(TaskScheduler));
	}
	return 0;
}

/**
 * Internal function to remove all < and > from the content of an string
 * to be included in an xml file. This function ASSUMES enough memory for
 * in place operations. This is UNSAFE outside of the context of this file.
 * **/
static 
void sanitize_xml(const char * src, char * dest)
{
	while (*src) {
		switch (*src) {
			case '<':
				*dest = 0;
				dest = strcat(dest,"&lt;");
				dest += 4;
				break;
			case '>':
				*dest = 0;
				strcat(dest,"&gt;");
				dest+=4;
				break;
			default:
				*dest = *src;
				dest ++;
		}
		src++;
	}
	*dest = 0;
}

/**
 * Internal function to convert all task names to "word", meaning a sequence of
 * alpha-numeric character separated by '_'. This is specially important if the
 * task names are to be used as commands or python functions.
 * **/
static 
void convert_name_to_under(char * buffer)
{
	while (*buffer) {
		if (isspace(*buffer)) *buffer = '_';
		if (!isprint(*buffer)) *buffer= '_';
		buffer ++;
	}
	*buffer = 0;
}

/**
 * Internal function called each time a connection is received on the 
 * task scheduler internal server. Its goal is to generate an XML description
 * of the task scheduler state. It does not wait for any request, but just
 * write the XML stream to the connection then close the connection.
 * **/
static 
void * ts_server_function(void * arg1, void * arg2)
{
	char convbf[1024];
	char buffer[1024];
    RtxInetTcpClient * clnt = (RtxInetTcpClient *) arg1;
	TaskScheduler * sched = (TaskScheduler*)arg2;
	/* Some clients seems to appreciate to be given a bit of time before
	 * receiving the data */
	rtx_timer_sleep(0.1);
	sprintf(buffer,"<?xml version=\"1.0\" encoding=\"iso-8859-1\"?>\n");
	rtx_inet_write(clnt->sock,buffer,strlen(buffer),NULL);
	sprintf(buffer,"<TASKLIST>\n\t<NUMTASKS>%d</NUMTASKS>\n",sched->numTasks);
	rtx_inet_write(clnt->sock,buffer,strlen(buffer),NULL);
	sprintf(buffer,"\t<IDLE>%d</IDLE>\n\t<FAIL>%d</FAIL>\n",
			sched->idleTask,sched->failureTask);
	rtx_inet_write(clnt->sock,buffer,strlen(buffer),NULL);
	sprintf(buffer,"\t<COMMAND>%s</COMMAND>\n\t<STATE>%s</STATE>\n",
			sched->demandItem->varName,sched->statusItem->varName);
	rtx_inet_write(clnt->sock,buffer,strlen(buffer),NULL);
	if (sched->tasks) {
		unsigned int i;
		for (i=0;i<sched->numTasks;i++) {
			sprintf(buffer,"\t<TASK>\n");
			rtx_inet_write(clnt->sock,buffer,strlen(buffer),NULL);
			sanitize_xml(sched->tasks[i].name,convbf);
			convert_name_to_under(convbf);
			sprintf(buffer,"\t\t<NAME>%s</NAME>\n",convbf);
			rtx_inet_write(clnt->sock,buffer,strlen(buffer),NULL);
			sanitize_xml(sched->tasks[i].help,convbf);
			sprintf(buffer,"\t\t<HELP>%s</HELP>\n",convbf);
			rtx_inet_write(clnt->sock,buffer,strlen(buffer),NULL);
			sprintf(buffer,"\t\t<MINARG>%d</MINARG>\n",sched->tasks[i].minArgs);
			rtx_inet_write(clnt->sock,buffer,strlen(buffer),NULL);
			sprintf(buffer,"\t\t<MAXARG>%d</MAXARG>\n",sched->tasks[i].maxArgs);
			rtx_inet_write(clnt->sock,buffer,strlen(buffer),NULL);
			sprintf(buffer,"\t</TASK>\n");
			rtx_inet_write(clnt->sock,buffer,strlen(buffer),NULL);
		}
	}
	/* The user data is generated by the user data generation function that the
	 * user may have provided. This is useful to pass some mission specific
	 * parameters to the client */
	sprintf(buffer,"<USERDATA>\n");
	rtx_inet_write(clnt->sock,buffer,strlen(buffer),NULL);
	if (sched->userdata_gen) {
		sched->userdata_gen(clnt->sock,sched->context);
	}
	sprintf(buffer,"</USERDATA>\n");
	rtx_inet_write(clnt->sock,buffer,strlen(buffer),NULL);
	sprintf(buffer,"</TASKLIST>\n");
	rtx_inet_write(clnt->sock,buffer,strlen(buffer),NULL);
	/* wait this long, to avoid the socket to go into TIME_WAIT */
	rtx_timer_sleep(0.5);
	return NULL;
}

/**
 * Start the scheduler server on port port. This scheduler will answer
 * connection by outputing a task description, see ts_server_function
 * It is useful to have a non null user_data_gen if the system needs to 
 * pass some mission specific parameters to the mission controllers
 * */
int task_scheduler_start_server(
		TaskScheduler * sched,    /* the scheduler object */
		unsigned int port,         /* the TCP port to listen onto */
		UserDataGenerator user_data_gen /* a user data generator */
)
{
	sched->userdata_gen = user_data_gen;
	sched->server = rtx_inet_init(RTX_INET_TCP_SERVER,
			NULL, port, NULL, 0, 
			ts_server_function, NULL, sched);
	if (!sched->server) {
		return rtx_error("Failed to start task-scheduler server");
	}


	return 0;
}

/**
 * Register a system checking function. This function will be called at each
 * iteration, with context as argument. If it returns a failure (!0, rtx_error)
 * then the current task will be declared failed.
 * */
int task_scheduler_register_syscheck(
		TaskScheduler *ts,            /* the scheduler object */
		TaskSchedulerSystemCheck sc   /* the system checking function, may be null */
)
{
	ts->checkfunc = sc;
	return 0;
}

