#ifndef MISSION_SCHEDULER_H
#define MISSION_SCHEDULER_H

#include <rtx/thread.h>
#include "task.h"

#ifdef __cplusplus
extern "C" {
#endif

/** 
 * All the data required by the mission scheduler 
 * */
typedef struct {
	/* private */
	int mallocd;                   /* has this object been malloc'ed */
	int verbose;                   /* verbosity level */
	DDX_STORE_ITEM * commandItem;  /* pointer on the store variable for the task requests */
	DDX_STORE_ITEM * statusItem;   /* pointer on the store variable for the task status */
	
	char * missionFile;            /* file containing the mission description */
	int missionIsAScript;          /* flag if we try running a python script */
	int abortMission;              /* flag set to 1 when the mission needs to be aborted */
	int scriptPid;                 /* pid of the script process */
	RtxThread * missionThread;     /* thread responsible for running a mission file */
	unsigned int missionIndex;     /* index of task currently running loaded mission */
	unsigned int missionLength;    /* length of the loaded mission */
	DDX_TASK_DEMAND * mission;     /* description of the loaded mission */

	unsigned int numTasks;         /* number of task managed by the scheduler */
	unsigned int idleTask;         /* id of the idle task (executed when nothing else to do) */
	unsigned int failureTask;      /* id of the failure task (executed when an error condition has occured) */
	TaskDescription * tasks;       /* list of tasks managed by the scheduler */
} MissionScheduler;

/**
 * Initialise the mission scheduler data
 * */
extern
MissionScheduler * mission_scheduler_init(
		MissionScheduler * ms,         /* if we don't want to malloc the structure */
		int verbose,                   /* augment tracing info */
		DDX_STORE_ID * store,          /* the store (DDX_STORE_ID*) pointer */
		const char * command,          /* the name of the store variable for task request */
		const char * status,           /* the name of the store variable for task status */
		const TaskDescription * tasks, /* the list of all registered task */ 
		unsigned int idleTask,         /* the index of the task used when idling */
		unsigned int failureTask       /* the index of the task used after a failure */
		);

#ifdef HAS_XML
/**
 * Request the task description list from the task controller
 * The returned array has been created with malloc, and must be free by the
 * caller. name and help string must also be freed by the caller.
 * Only the task name, help and args are read.
 * Returns a array terminated by TASK_DECLARATION_END on success
 * Returns NULL on failure
 * */
extern
TaskDescription* request_task_description(const char *server, unsigned int port);
#endif


/**
 * Load the mission described in filename 
 * A mission filename is made of a sequence of line:
 * <task name> <arg1> <arg2> <arg3> ...
 * and terminates by the keyword endfile.
 * Space are not relevant, comment line are defined by #
 * */
extern 
int mission_scheduler_load_file(MissionScheduler * ms, 
		const char * filename, int isScript);

/** 
 * Run a mission previously loaded. The mission is aborted after timeout (if
 * non zero), or if endsignal is non null, and becomes non zero.
 * */
extern 
int mission_scheduler_run(MissionScheduler *ms, 
		double timeout, int * endsignal);

/** 
 * Run a mission previously loaded. The mission is aborted after timeout (if
 * non zero), or if endsignal is non null, and becomes non zero.
 * The function starts a thread and returns immediately
 * */
extern 
int mission_scheduler_run_thread(MissionScheduler *ms);

/** 
 * Stop a mission currently running
 * */
extern 
int mission_scheduler_stop_thread(MissionScheduler *ms);

/**
 * Just write 'task' to the store 
 * */
extern 
int mission_scheduler_run_task(MissionScheduler *ms, 
		DDX_TASK_DEMAND * task);

/**
 * Send 'task' to the task scheduler and wait for completion by monitoring
 * output status. Upon completion, if status is non null, it is filled with the
 * last task status.
 * The task is aborted after timeout (if non zero), or if endsignal is 
 * non null, and becomes non zero.
 * */
extern 
int mission_scheduler_run_task_and_wait(MissionScheduler *ms, 
		DDX_TASK_DEMAND * task, DDX_TASK_STATUS * status,
		double timeout, int * endsignal);

/**
 * Get the next sequence number to use, by watching the one currently 
 * used in the store task status 
 * */
extern 
int mission_scheduler_get_next_seqno(MissionScheduler *ms);

/**
 * Send the idle task, returns immediately
 * */
extern
int mission_scheduler_idle(MissionScheduler *ms);

/**
 * Send the failure task, returns immediately
 * */
extern
int mission_scheduler_failure(MissionScheduler *ms);

/**
 * Destroy the mission scheduler data, and free the object, if it 
 * has been allocated in the first place 
 * */
extern
int mission_scheduler_destroy(MissionScheduler *ms);

#ifdef __cplusplus
}
#endif


#endif // MISSION_SCHEDULER_H
