#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <unistd.h>

#include <rtx/error.h>
#include <rtx/message.h>
#include <rtx/timer.h>
#include <rtx/longstring.h>

#include "mission-scheduler.h"


/**
 * Initialise the mission scheduler data
 * */
extern
MissionScheduler * mission_scheduler_init(
		MissionScheduler * ms,         /* if we don't want to malloc the structure */
		int verbose,                   /* augment tracing info */
		DDX_STORE_ID * store,          /* the store (DDX_STORE_ID*) pointer */
		const char * command,          /* the name of the store variable for task request */
		const char * status,           /* the name of the store variable for task status */
		const TaskDescription * tasks, /* the list of all registered task */ 
		unsigned int idleTask,         /* the index of the task used when idling */
		unsigned int failureTask       /* the index of the task used after a failure */
		)
{
	MissionScheduler * res = ms;
	unsigned int i;
	if (res) {
		res->mallocd = 0;
	} else {
		res = (MissionScheduler*)malloc(sizeof(MissionScheduler));
		if (!res) return rtx_error_null("mission_scheduler_init: malloc failed");
		res->mallocd = 1;
	}
	res->verbose = verbose;
	res->mission = NULL;
	res->missionFile = NULL;
	res->missionLength = 0;
	res->missionIndex = -1;
	res->abortMission = 0;
	res->missionIsAScript = 0;
	res->commandItem = NULL;
	res->statusItem = NULL;
	res->tasks = NULL;
	res->missionThread = NULL;

	res->idleTask = idleTask;
	res->failureTask = failureTask;
	res->numTasks = 0;
	while (tasks[res->numTasks].name != NULL) {
		res->numTasks += 1;
	}
	res->tasks = (TaskDescription*)malloc(sizeof(TaskDescription)*res->numTasks);
	if (!res->tasks) {
		mission_scheduler_destroy(res);
		return rtx_error_null("mission_scheduler_init: mission malloc failed");
	}

	for (i=0;i<res->numTasks;i++) {
		memcpy(res->tasks+i,tasks+i,sizeof(TaskDescription));
		res->tasks[i].name = strdup(tasks[i].name);
		res->tasks[i].help = strdup(tasks[i].help);
	}

	if (DDX_STORE_REGISTER_TYPE(store,DDX_TASK_DEMAND)) {
		mission_scheduler_destroy(res);
		return rtx_error_null("mission_scheduler_init: failed to register command type");
	}
	if (DDX_STORE_REGISTER_TYPE(store,DDX_TASK_STATUS)) {
		mission_scheduler_destroy(res);
		return rtx_error_null("mission_scheduler_init: failed to register status type");
	}
		
	res->commandItem = ddx_store_lookup_item(store,command,"DDX_TASK_DEMAND",0);
	if (!res->commandItem) {
		mission_scheduler_destroy(res);
		return rtx_error_null("mission_scheduler_init: failed to register command variable");
	}

	res->statusItem = ddx_store_lookup_item(store,status,"DDX_TASK_STATUS",0);
	if (!res->statusItem) {
		mission_scheduler_destroy(res);
		return rtx_error_null("mission_scheduler_init: failed to register status variable");
	}

	return res;
}

/************ Mission parsing and scheduling ****************/

/** 
 * In place trimming of string s. Terminate the string at the last non white
 * character, and return a pointer to the first non white
 * */
static 
char * trim(char * s) {
	signed int len = strlen(s);
	signed int i;
	char * res = s;
	for (i=len-1;i>=0;i--) {
		if (isspace(s[i]))
			s[i]=0; 
		else break;
	}
	while (*res && isspace(*res)) res++;
	return res;
}

/**
 * Read double from string s, and check their number against the task
 * description.
 * **/
static int parseNdoubles(TaskDescription *tasks,
		const char * s, DDX_TASK_DEMAND *it)
{
	int pos=0, ret;
	unsigned int maxArgs,minArgs;
	maxArgs = tasks[it->taskId].maxArgs;
	minArgs = tasks[it->taskId].minArgs;
	

	for (it->argc=0;it->argc<maxArgs;it->argc++) {
		pos = 0;
		ret = sscanf(s," %le%n",it->argv+it->argc,&pos);
		if (ret != 1) {
			if (it->argc >= minArgs) return 0;
			return rtx_error("Parsing mission item %d:%s, failed to parse arg %d: '%s'\n",
					it->seqno,tasks[it->taskId].name,it->argc,s);
		}
		s += pos;
	}
	return 0;	
}


/**
 * Load the mission described in filename 
 * */
int mission_scheduler_load_file(MissionScheduler * ms, 
		const char * filename, int isScript)
{
	unsigned int mt,endl;
	int pos=0,completed=0, found = 0;;
	char line[1024],taskName[1024];
	char *lbuffer,*tline;
	unsigned int seqno = 0;
	DDX_TASK_DEMAND it;
	struct stat sstat;

	FILE *fp;

	mission_scheduler_stop_thread(ms);
	
	if (ms->mission) free(ms->mission);
	ms->mission = NULL;
	ms->missionLength = 0;
	ms->missionIndex = -1;
	ms->abortMission = 0;
	ms->missionIsAScript = isScript;
	if (ms->missionFile) free(ms->missionFile);
	ms->missionFile = strdup(filename);
	
	if (isScript) {
		return 0;
	}


	fp = fopen(filename,"r");
	if (fp == NULL) return rtx_error("parseMissionFile: failed to open '%s'\n",filename);

	stat(filename,&sstat);
	lbuffer = (char *)calloc(1,sstat.st_size);
	if (lbuffer == NULL) return rtx_error("parseMissionFile: failed to malloc line");
	endl = sstat.st_size - 1;
	
	while (!feof(fp) && !completed) {
		/* get a line and trim it */
		int llen;
		lbuffer[0] = 0;
		while (!feof(fp)) {
			line[1023] = 0;
			fgets(line,1023,fp);
			tline = trim(line);

			/* remove comment line */
			if ((tline[0] == '#')||(tline[0] == '%')||(tline[0] == 0)) 
				continue;

			llen = strlen(tline)-1;
			if (tline[llen] != '\\') {
				strcat(lbuffer,tline);
				break;
			} else {
				tline[llen] = 0;
				strcat(lbuffer,tline);
			}
		} 
		
		tline = lbuffer;
		/* Parse command name */
		if (sscanf(tline,"%1023s%n",taskName,&pos) != 1) continue;
		tline += pos;
		
		/* structure initialisation */
		it.taskId = ms->idleTask;
		it.argc = 0;
		it.seqno = seqno;
		it.maxTime = -1;

		/* task specific functions */
		found = 0;
		for (mt=0;mt<ms->numTasks;mt++) {
			if (strcasecmp(ms->tasks[mt].name,taskName) == 0) {
				it.taskId = mt;
				found = 1;
				break;
			}		
		}
		/* end of file */
		if (strcasecmp("endfile",taskName) == 0) {
			it.taskId = ms->idleTask;
			completed = 1;
			found = 1;
		}		
		if (!found)
			return rtx_error("Unrecognised task id '%s'",taskName);

		if (!completed) {
			/* structure filling */
			if (parseNdoubles(ms->tasks,tline,&it)) {
				return rtx_error("mission_scheduler_load_file: failed to parse '%s'\n",filename);
			}
			if (seqno >= ms->missionLength) {
				ms->missionLength += 10;
				ms->mission = (DDX_TASK_DEMAND*)realloc(ms->mission,
						ms->missionLength*sizeof(DDX_TASK_DEMAND));
				if (!ms->mission)
					return rtx_error("mission_scheduler_load_file: realloc failed");
			}
			memcpy(ms->mission+seqno,&it,sizeof(DDX_TASK_DEMAND));
			seqno ++; 
		}
	}
		
	fclose (fp);
	ms->missionLength = seqno;
	return 0;
	
}

/**
 * Just write 'task' to the store 
 * */
int mission_scheduler_run_task(MissionScheduler *ms,
		DDX_TASK_DEMAND * task)
{
	return ddx_store_write(ms->commandItem,task,NULL);
}


/**
 * Thread implementation of mission running. This is a thread because, the
 * mission may be a mission script (in python for instance)
 * */
static
void * mission_thread(void * m)
{
	MissionScheduler *ms = (MissionScheduler*)m;

	if (ms->missionIsAScript) {
		system(ms->missionFile);
	} else {
		mission_scheduler_run(ms, 0, &(ms->abortMission));
	}

	mission_scheduler_idle(ms);

	return NULL;
}

/** 
 * Stop a mission currently running
 * */
int mission_scheduler_stop_thread(MissionScheduler *ms)
{
	ms->abortMission = 1;
	if (!ms->missionThread) {
		return 0;
	}
	if (ms->missionIsAScript) {
		rtx_thread_destroy_sync(ms->missionThread);
	} else {
		rtx_thread_join(ms->missionThread);
		rtx_thread_destroy(ms->missionThread);
	}
	mission_scheduler_idle(ms);
	ms->missionThread = NULL;

	return 0;
}

/** 
 * Run a mission previously loaded. The mission is aborted after timeout (if
 * non zero), or if endsignal is non null, and becomes non zero.
 * The function starts a thread and returns immediately
 * */
int mission_scheduler_run_thread(MissionScheduler *ms)
{
	mission_scheduler_stop_thread(ms);

	if (!(ms->missionIsAScript || (ms->mission != NULL))) {
		return rtx_error("run_thread: mission must be loaded before run_thread");
	}
	ms->abortMission = 0;

	ms->missionThread = rtx_thread_create("mission", 0, RTX_THREAD_SCHED_OTHER,
			RTX_THREAD_PRIO_MIN, 0, RTX_THREAD_CANCEL_DEFERRED,
			mission_thread, ms, NULL, NULL);
	if (!ms->missionThread) {
		return rtx_error("run_thread: failed to start mission thread");
	}


	return 0;
}

/** 
 * Run a mission previously loaded. The mission is aborted after timeout (if
 * non zero), or if endsignal is non null, and becomes non zero.
 * */
int mission_scheduler_run(MissionScheduler *ms,
		double timeout, int * endsignal)
{
	RtxTime rts;
	double t1,t2;
	unsigned int i;
	DDX_TASK_STATUS status;

	for (i=0;i<ms->missionLength;i++) {
		rtx_time_get(&rts);
		t1 = rtx_time_to_double(&rts);
		rtx_message("mission_scheduler_run: starting task %d:%s",
					i,task_name(ms->tasks,ms->mission+i));
		ms->missionIndex = i;
		if (mission_scheduler_run_task_and_wait(ms,
					ms->mission+i,&status,timeout,endsignal)) {
			return rtx_error("mission_scheduler_run: task %d:%s failed",
					i,task_name(ms->tasks,ms->mission+i));
		}
		rtx_time_get(&rts);
		t2 = rtx_time_to_double(&rts);
		if (timeout > 0) {
			timeout -= (t2-t1);
			if (timeout <= 0) {
				return rtx_error("mission_scheduler_run: timeout after task %d:%s",
						i,task_name(ms->tasks,ms->mission+i));
			}
		}
	}
	ms->missionIndex = -1;
	return 0;
}

/**
 * Send 'task' to the task scheduler and wait for completion by monitoring
 * output status. Upon completion, if status is non null, it is filled with the
 * last task status.
 * The task is aborted after timeout (if non zero), or if endsignal is 
 * non null, and becomes non zero.
 * */
int mission_scheduler_run_task_and_wait(MissionScheduler *ms, 
		DDX_TASK_DEMAND * task, DDX_TASK_STATUS * status,
		double timeout, int * endsignal)
{
	int ret;
	/* three states in this function:
	 * 0:WaitingAck : the task has been written but is not running yet
	 * 1:Running    : the task is running, but not completed
	 * 2:Completed  : the task is terminated 
	 * 3:Error      : something unexpected has happen
	 */
	int state = 0;
	int waitNextCycle;
	double ptime = -1;
	DDX_TASK_STATUS lstatus;
	RtxTime ts;
	double t0;
	rtx_time_get(&ts);
	t0 = rtx_time_to_double(&ts);
	ptime = t0;

	while (state < 3) {
		double t;
		if (endsignal && *endsignal) {
			rtx_message("mission_scheduler_run_task_and_wait: interrupted");
			if (status) 
				task_set_status(status,TSTATUS_INTERRUPTED,
					"User interrupt");
			return 1;
		}
		
		rtx_time_get(&ts);
		t = rtx_time_to_double(&ts);
		if ((timeout > 0) && (t - t0 > timeout)) {
			rtx_message("mission_scheduler_run_task_and_wait: timeout");
			if (status) 
				task_set_status(status,TSTATUS_TIMEDOUT,
					"Local timeout");
			return 1;
		}

		if (ddx_store_write(ms->commandItem,task,NULL)) 
			return rtx_error("mission_scheduler_run_task_and_wait: ddx_store_write failed");

		ret = ddx_store_read(ms->statusItem,&lstatus,NULL,1.0,1);
		if (ret < 0) 
			return rtx_error("mission_scheduler_run_task_and_wait: fatal error in ddx_store_read(status)");
		if (ret > 0)
			return rtx_error("mission_scheduler_run_task_and_wait: task status seems stalled");
		//printf("status = %d\n",lstatus.taskStatus);


		if (ms->verbose) {
			if (t - ptime > 1.0) {
				rtx_message("M %d task status: %d:%s : %s (%s)\n",
						state, lstatus.seqno,
						ms->tasks[lstatus.taskId].name,
						task_status_to_string(lstatus.taskStatus),
						lstatus.statusStr);
				ptime = t;
			}
		}

		waitNextCycle = 0;
		while (!waitNextCycle && (state<3)) {
			//printf("state %d id %d seqno %d\n",state,
			//		lstatus.taskId,lstatus.seqno);
			switch (state) {
				case 0:
					// WaitingAck
					if (lstatus.seqno != task->seqno) {
						waitNextCycle = 1;
						continue;
					}
					if (lstatus.taskId == task->taskId) {
						state = 1;
						break;
					} else {
						waitNextCycle = 1;
					}
					break;
				case 1:
					/* Running */
					if (lstatus.seqno != task->seqno) 
						return rtx_error("mission_scheduler_run_task_and_wait: Running: inconsistent seqno\n");
					if (lstatus.taskId != task->taskId) 
						return rtx_error("mission_scheduler_run_task_and_wait: Running: inconsistent taskId\n");
					if (lstatus.taskStatus != TSTATUS_RUNNING)
						state = 2;
					else 
						waitNextCycle = 1;
					break;
				case 2:
					/* Completion */
					if (lstatus.seqno != task->seqno) 
						return rtx_error("mission_scheduler_run_task_and_wait: Completed: inconsistent seqno\n");
					if (lstatus.taskId != task->taskId) 
						return rtx_error("mission_scheduler_run_task_and_wait: Completed: inconsistent taskId\n");
					if (lstatus.taskStatus == TSTATUS_COMPLETED) {
						if (status) memcpy(status,&lstatus,sizeof(DDX_TASK_STATUS));
					} else {
						lstatus.statusStr[127] = 0;
						return rtx_error("mission_scheduler_run_task_and_wait: task %d:%s failed with error '%s'\n",
								task->seqno,ms->tasks[task->taskId].name,
								lstatus.statusStr);
					}
					state = 3;
					break;
				default:
					return rtx_error("mission_scheduler_run_task_and_wait: invalid state %d\n",state);
			}
		}
				
	}
	if (ms->verbose) {
		rtx_message("M %d task status: %d:%s : %s (%s)\n",
			state, lstatus.seqno,
			ms->tasks[lstatus.taskId].name,
			task_status_to_string(lstatus.taskStatus),
			lstatus.statusStr);
	}
	return 0;
}

/**
 * Destroy the mission scheduler data, and free the object, if it 
 * has been allocated in the first place 
 * */
int mission_scheduler_destroy(MissionScheduler *ms)
{
	if (!ms) return rtx_error("mission_scheduler_destroy: null object");
	if (ms->mission) free(ms->mission);
	if (ms->commandItem) ddx_store_done_item(ms->commandItem);
	if (ms->statusItem) ddx_store_done_item(ms->statusItem);
	if (ms->tasks) {
		unsigned int i;
		for (i=0;i<ms->numTasks;i++) {
			free(ms->tasks[i].name);
			free(ms->tasks[i].help);
		}
		free(ms->tasks);
	}
	if (ms->mallocd) 
		free(ms);
	else
		bzero(ms,sizeof(MissionScheduler));
	return 0;
}

/**
 * Send the idle task, returns immediately
 * */
int mission_scheduler_idle(MissionScheduler *ms)
{
	DDX_TASK_DEMAND dmd;
	dmd.argc = 0;
	dmd.seqno = -1;
	dmd.maxTime = -1;
	dmd.taskId = ms->idleTask;
	return mission_scheduler_run_task(ms,&dmd);
}


/**
 * Send the failure task, returns immediately
 * */
int mission_scheduler_failure(MissionScheduler *ms)
{
	DDX_TASK_DEMAND dmd;
	dmd.argc = 0;
	dmd.seqno = -1;
	dmd.taskId = ms->failureTask;
	dmd.maxTime = -1;
	return mission_scheduler_run_task(ms,&dmd);
}


/**
 * Get the next sequence number to use, by watching the one currently 
 * used in the store task status 
 * */
int mission_scheduler_get_next_seqno(MissionScheduler *ms)
{
	int seqno;
	DDX_TASK_STATUS sts;
	DDX_TASK_DEMAND dmd;
	dmd.seqno = sts.seqno = 0;
	ddx_store_read(ms->statusItem,&sts,NULL,0,0);
	seqno = sts.seqno + 1;
#if 0
	ddx_store_read(ms->commandItem,&dmd,NULL,0,0);
	if (dmd.seqno >= seqno) seqno = dmd.seqno + 1;
#endif
	return seqno;
}

/******************* PARSE From Network *****************/

#ifdef HAS_XML 

#include <libxml/xmlmemory.h>
#include <libxml/parser.h>


TaskDescription* request_task_description(const char *server, unsigned int port)
{
	xmlDocPtr doc;
	xmlNodePtr cur,tlist;
	unsigned int ntasks = 0, sizetasks = 0;
	TaskDescription end = TASK_DECLARATION_END;
	TaskDescription *tasks = NULL;

	RtxInet *conn = NULL;
	RtxLongString *buffer = rtx_longstring_init();

	conn = rtx_inet_init(RTX_INET_TCP_CLIENT,NULL,0,server,port,NULL,NULL,NULL);
	if (!conn) {
		return rtx_error_null("Failed to connect to %s:%d",server,port);
	}


	while (1) {
		char line[1024]="";

		rtx_inet_readline(conn->sock,line,1023,NULL);

		rtx_longstring_concat(buffer,line);

		if (strstr(line,"</TASKLIST>")) {
			break;
		}
	}

	rtx_inet_done(conn);

	//printf("Received task descriptions: \n%s\n",rtx_longstring_content(buffer));

	doc = xmlParseMemory(rtx_longstring_content(buffer),rtx_longstring_length(buffer));
	if (doc == NULL) {
		rtx_error("Failed to load task description (%s)",rtx_longstring_content(buffer));
	}

	rtx_longstring_destroy(buffer);

	if (doc == NULL) {
		return NULL;
	}

	 /*
     * Check the document is of the right kind
     */
    
    cur = xmlDocGetRootElement(doc);
	if (cur == NULL) {
		xmlFreeDoc(doc);
		return rtx_error_null("Invalid task description (empty)");
	}
	if (xmlStrcmp(cur->name, (const xmlChar *) "TASKLIST")) {
		xmlFreeDoc(doc);
		return rtx_error_null("document of the wrong type, root node != TASKLIST");
	}

	/*
     * Now, walk the tree.
     */
	/* First level we expect just Jobs */
	
	ntasks = 0;
	for (cur = cur->xmlChildrenNode;cur != NULL; cur = cur->next) {
		if (cur == NULL) break;
		if (xmlIsBlankNode(cur)) continue;
		if (xmlStrcmp(cur->name, (const xmlChar *) "NUMTASKS")==0) {
			// ignore for now
		}
		if (xmlStrcmp(cur->name, (const xmlChar *) "FAIL")==0) {
			// ignore for now
		}
		if (xmlStrcmp(cur->name, (const xmlChar *) "IDLE")==0) {
			// ignore for now
		}
		if (xmlStrcmp(cur->name, (const xmlChar *) "TASK")==0) {
			TaskDescription thistask = TASK_DECLARATION_END;
			if (ntasks >= sizetasks) {
				sizetasks += 20;
				tasks = (TaskDescription*)realloc(tasks,sizetasks * sizeof(TaskDescription));
				if (!tasks) {
					xmlFree(doc);
					return rtx_error_null("Error reallocating memory for the task array");
				}
			}

			for (tlist = cur->xmlChildrenNode;tlist != NULL; tlist = tlist->next) {
				if (tlist == NULL) break;
				if (xmlIsBlankNode(tlist)) continue;
				if (xmlStrcmp(tlist->name, (const xmlChar *) "NAME")==0) {
					thistask.name = (char*)xmlNodeListGetString(doc,tlist->xmlChildrenNode,1);
				}
				if (xmlStrcmp(tlist->name, (const xmlChar *) "HELP")==0) {
					thistask.help = (char*)xmlNodeListGetString(doc,tlist->xmlChildrenNode,1);
				}
				if (xmlStrcmp(tlist->name, (const xmlChar *) "MINARG")==0) {
					xmlChar *s = xmlNodeListGetString(doc,tlist->xmlChildrenNode, 1);
					sscanf((char*)s," %d ",&thistask.minArgs);
					xmlFree(s);
				}
				if (xmlStrcmp(tlist->name, (const xmlChar *) "MAXARG")==0) {
					xmlChar *s = xmlNodeListGetString(doc,tlist->xmlChildrenNode, 1);
					sscanf((char*)s," %d ",&thistask.maxArgs);
					xmlFree(s);
				}
			}

			memcpy(tasks+ntasks,&thistask,sizeof(TaskDescription));
			ntasks += 1;
		}
	}

	xmlFreeDoc(doc);

	/* resize the array to be exactly the right size. This is most likely to
	 * have no effect, but... */
	tasks = (TaskDescription*)realloc(tasks,(ntasks+1) * sizeof(TaskDescription));
	if (!tasks) {
		return rtx_error_null("Error reallocating memory for the task array");
	}
	/* terminate by putting a null class at the end of the list */
	memcpy(tasks+ntasks,&end,sizeof(TaskDescription));

	return tasks;
}
#endif

