#include <stdlib.h>
#include <string.h>

#include <rtx/error.h>
#include <rtx/message.h>
#include <rtx/main.h>

#include "mission-command.h"

/**
 * This makes this API non-reentrant, but much easier to use.
 * It makes sense, because there can only be one instance running of the
 * command line interface
 * */
typedef struct {
	int mallocd;
	int verbose;
	int seqno;
	char * missionFile;
	char * prompt;
	MissionScheduler * scheduler;
	RtxCommandMenu * menu;
	RtxThread * thread;
} MissionCommand;


static MissionCommand * mc = NULL;

/**
 * Parse and check the argument of a task
 * */
static 
int mission_command_parse_task(MissionScheduler * ms, int argc, const char *argv[], 
		DDX_TASK_DEMAND *it)
{
	unsigned int i;
	if (task_check_args_number(ms->tasks, it, argc-1)) {
		return rtx_error("Command '%s': invalid number of argument %d\n",
				task_name(ms->tasks,it), argc-1);
	}
	for (i=1;i<argc;i++) {
		if (sscanf(argv[i],"%le",it->argv+i-1)!=1) {
			return rtx_error("Command %s, failed to parse arg %d: '%s'\n",
					task_name(ms->tasks,it),i,argv[i]);
		}
		if (mc->verbose)
			printf("%s: %d : %s -> %f\n",argv[0],i,argv[i],it->argv[i-1]);
	}
	it->argc = argc - 1;
	return 0;	
}


/**
 * Execute one task: parse the argument, send the task and wait for the results
 * */
static
int mission_command_exec(MissionScheduler *ms, int argc,const char *argv[],int taskId){ 
	DDX_TASK_DEMAND it;
	int ret;

	it.maxTime = -1;
	it.seqno = mc->seqno++;
	it.taskId = taskId;
	if (mission_command_parse_task(ms,argc,argv,&it)) {
		rtx_error_flush("Fail to parse command '%s'\n",argv[0]);
		return -1;
	}
	if (mc->verbose) {
		rtx_message("Scheduling task %d:%s with %d args [%.2f %.2f ... ]\n",
				it.seqno,ms->tasks[taskId].name,it.argc,it.argv[0],it.argv[1]);
	}
	ret = mission_scheduler_run_task_and_wait(ms,&it,NULL,-1,NULL);
	return ret;
}

/**
 * Internal implementation for all tasks. Find its name from argv, and look for
 * it in the list of tasks
 * */
static 
int mission_command_handler(int argc, const char * argv[])
{
	unsigned int  tt;
	tt = task_get_index(mc->scheduler->tasks,mc->scheduler->numTasks,argv[0]);
	if (tt >= 0)
		return mission_command_exec(mc->scheduler,argc,argv,tt);
	return rtx_error("Failed to recognise command '%s'\n",argv[0]);
}


/**
 * Internal implementation for the load command
 * */
static
int mission_command_load(int argc, const char * argv[])
{
	if (argc < 2) {
		rtx_error_flush("Command load expect 2 argument, the filename, thread flag\n");
		return -1;
	}
	if (mission_scheduler_load_file(mc->scheduler,argv[1],0)) {
		rtx_error_flush("Failed to parse mission file\n");
		return -1;
	}

	int thread = 0;
	if (argc >=2 ) {
		thread = atoi(argv[2]);
	}
	if (thread) {
		if (mission_scheduler_run_thread(mc->scheduler)) {
			rtx_error_flush("Error while starting mission\n");
			return -1;
		} else {
			if (mc->verbose) rtx_message("Mission started\n");
		}
	} else {
		if (mission_scheduler_run(mc->scheduler,-1,NULL)) {
			rtx_error_flush("Error while running mission\n");
			return -1;
		} else {
			if (mc->verbose) rtx_message("Mission completed\n");
		}
	}
	return 0;
}


/**
 * Internal implementation for the run command
 * */
static
int mission_command_run(int argc, const char * argv[])
{
	if (argc < 2) {
		rtx_error_flush("Command load expect 1 argument, the filename\n");
		return -1;
	}
	if (mission_scheduler_load_file(mc->scheduler,argv[1],1)) {
		rtx_error_flush("Failed to parse mission file\n");
		return -1;
	}

	if (mission_scheduler_run_thread(mc->scheduler)) {
		rtx_error_flush("Error while running mission\n");
		return -1;
	} else {
		if (mc->verbose) rtx_message("Mission started\n");
	}
	return 0;
}

/**
 * Internal implementation for the stop command
 * */
static
int mission_command_stop(int argc, const char * argv[])
{
	if (mission_scheduler_stop_thread(mc->scheduler)) {
		rtx_error_flush("Error while stopping mission\n");
		return -1;
	} else {
		if (mc->verbose) rtx_message("Mission stopped\n");
	}
	return 0;
}



static
void * mission_command_worker_thread(void * arg)
{
	MissionCommand *mc = (MissionCommand*)arg;
	if (mc->missionFile != NULL) {
		if (mc->verbose)
			rtx_message("Executing mission '%s'\n",mc->missionFile);
		const char *argv[2] = {"Load",mc->missionFile};
		mission_command_load(2,argv);
	} else {
		/* if we're going interactive, be sure that starbug is idle */
		rtx_command_handler(mc->menu,mc->prompt);
	}
	rtx_main_signal_shutdown();
	return NULL;
}


/**
 * Start a command line interface for the tasks in sched.
 * If missionFile is non null, then the mission is executed and the function 
 * triggers a rtx_main_signal_shutdown.
 * In both case, the function starts a threads and returns immediately.
 * This function is non-reentrant. Don't call it twice or it will fail 
 * */
int mission_command_start(MissionScheduler * sched, 
		const char * prompt, const char * missionFile,
		int verbose)
{
	unsigned int tt;

	if (mc)
		return rtx_error("mission_command_start: this function is NOT reentrant");

	mc = (MissionCommand*)calloc(1,sizeof(MissionCommand));
	if (!mc) return rtx_error("mission_command_start: malloc failed");

	mc->scheduler = sched;
	mc->missionFile = missionFile?strdup(missionFile):NULL;
	mc->prompt = prompt?strdup(prompt):strdup("> ");
	mc->seqno = mission_scheduler_get_next_seqno(mc->scheduler);

	mc->menu = (RtxCommandMenu*)malloc((mc->scheduler->numTasks+4) * 
			sizeof (RtxCommandMenu));
	if (!mc->menu) {
		mission_command_terminate();
		return rtx_error("mission_command_start: menu malloc failed");
	}

	for (tt = 0;tt < mc->scheduler->numTasks; tt++) {
		mc->menu[tt].name = (char*)mc->scheduler->tasks[tt].name;
		mc->menu[tt].handler = (RtxCommandHandler)mission_command_handler;
		mc->menu[tt].help = (char*)mc->scheduler->tasks[tt].help;
		if (!mc->menu[tt].help)
			mc->menu[tt].help = "No help yet, please update it";
	}
	mc->menu[tt].name = "run";
	mc->menu[tt].handler = (RtxCommandHandler)mission_command_load;
	mc->menu[tt].help = "Load and run a mission file (1 arg: filename)";

	mc->menu[tt+1].name = "exec";
	mc->menu[tt+1].handler = (RtxCommandHandler)mission_command_run;
	mc->menu[tt+1].help = "Run a mission script (1 arg: filename)";

	mc->menu[tt+2].name = "stop";
	mc->menu[tt+2].handler = (RtxCommandHandler)mission_command_stop;
	mc->menu[tt+2].help = "Stop current mission";

	mc->menu[tt+3].name = NULL;
	mc->menu[tt+3].handler = NULL;
	mc->menu[tt+3].help = NULL;

	mc->thread = rtx_thread_create("mission thread",0, 
			RTX_THREAD_SCHED_OTHER,0,0, RTX_THREAD_CANCEL_DEFERRED,
			mission_command_worker_thread,mc,NULL,NULL);

	if (!mc->thread) {
		mission_command_terminate();
		return rtx_error("mission_command_start: failed to create thread");
	}

	return 0;
}

/**
 * Terminates the command line interface. Stop the corresponding thread.
 * */
int mission_command_terminate()
{
	if (!mc) return 0;

	if (mc->thread) {
		rtx_thread_destroy_sync(mc->thread);
		/* TODO: do something about readline */
	}
	if (mc->prompt) free(mc->prompt);
	if (mc->missionFile) free(mc->missionFile);
	if (mc->menu) free(mc->menu);
	if (mc->mallocd) free(mc);
	mc = NULL;
	return 0;
}



