#ifndef SENSORI_MOTOR_MODULE_H
#define SENSORI_MOTOR_MODULE_H

#include <vector>
#include <set>
#include <string>

#include <rtx/param.h>
#include <rtx/message.h>

#include <DDXStore.h>
#include <DDXVariable.h>

/** Sensori motor module without specified output 
 *  Maybe used as a sink: pure controller or data exporter
 * **/
class GenericSensoriMotorModule {
	public:
		typedef std::set<GenericSensoriMotorModule*, 
		std::less<GenericSensoriMotorModule*> > DependencySet;
	protected:
		std::string name;
		DependencySet dependencies;
		double lastRunTime; // Duration of the last run execution

		/** Adds other to the set of dependencies of this module **/
		void dependsOn(GenericSensoriMotorModule* other){
			if (other) {
				dependencies.insert(other);
			} else {
				rtx_message("Dependency of '%s': can't accept NULL dependency",
						name.c_str());
			}
		}

	public:
		GenericSensoriMotorModule(const std::string & _name) : name(_name) {
			lastRunTime = -1;
		}
		virtual ~GenericSensoriMotorModule() {
			//rtx_message("Terminating module %s",name.c_str());
			this->stop();
			this->terminate();
			//rtx_message("Terminated module %s",name.c_str());
		}

		/** Called once to initialise internal paramters **/
		virtual int init(DDXStore &store, RtxParamStream *pf) {return 0;}

		/** Called on startup or after reset **/
		virtual int start() {return 0;}

		/** Called when a watchdog failure has been detected **/
		virtual int reset() {return 0;}

		/** Called before a reset **/
		virtual int stop() {return 0;}

		/** Called before shutdown, once **/
		virtual int terminate() {return 0;}

		/** Called in normal operation, every cycle **/
		virtual int run() {return 0;}

		const std::string & getName() {return name;}

		const DependencySet & getDependencies() const{return dependencies;}

		int run_and_time() {
			RtxTime t1,t2;
			rtx_time_get(&t1);
			int r = this->run();
			rtx_time_get(&t2);
			lastRunTime = rtx_time_subtract_to_double(&t2,&t1);
			return r;
		}

		double getRunTime() const {return lastRunTime;}
};


/** Sensori motor module with specified output **/
template <class OutputClass>
class SensoriMotorModule : public GenericSensoriMotorModule {
	protected:
		OutputClass moduleState;
	public:
		SensoriMotorModule(const std::string &name) : 
			GenericSensoriMotorModule(name) {}
		virtual ~SensoriMotorModule() {}

		const OutputClass & getState() const {
			return moduleState;
		}
};






#endif // SENSORI_MOTOR_MODULE_H
