#ifndef LOW_LEVEL_H
#define LOW_LEVEL_H

#include <list>
#include <vector>
#include <rtx/param.h>
#include <rtx/sync.h>
#include <rtx/thread.h>
#include <rtx/time.h>
#include <rtx/timer.h>
#include <rtx/error.h>
#include <rtx/message.h>
#include <rtx/main.h>
#include "SensoriMotorModule.h"

class LowLevelLoop
{
	public:
		LowLevelLoop(bool restartOnLock); 
		virtual ~LowLevelLoop();

		
		void addModule(GenericSensoriMotorModule *module) {
			modules.push_back(module);
			sorted = false;
		}
		
		// Get the mutex locked before each "run session"
		RtxMutex * getMutex() {return runMutex;}

		int init(DDXStore &store,
				RtxParamStream *pf);

		int start(double period, double wdog_period);
		int terminate();

		int generateVCGModuleGraph(const std::string & filename) ;
		int generateDotModuleGraph(const std::string & filename) ;

		int validateModuleNetwork() ;
	protected:
		std::list<GenericSensoriMotorModule*> modules;
		void insertModule(std::list<GenericSensoriMotorModule*> &list,
				GenericSensoriMotorModule* module);
		int validateModuleTree(GenericSensoriMotorModule* mod);
		void sortModules();
		bool sorted;

	protected:
		/** All the thread management stuff **/
		unsigned int lastSelfMonWatchdog, watchdog;
		bool end, verbose, selfMonRestarting, terminated;
		double main_period;
		double watchdog_period;
		double restart_sleep_length;
		RtxSync *mainSync, *terminationSync;
		RtxTimerThread *mainClockTimer, *selfMonTimer;
		RtxThread *mainThread, *targetThread;
		RtxMutex *runMutex;
		bool restartOnDeadlock;

		int pause_main_thread();
		int resume_main_thread();

		void * main_thread_func();
		void   selfmon_thread_func();
		void * termination_thread_func();
		void   restarting_thread_func();

		friend void *main_thread_ext_func(void *);
		friend void main_clock_ext_func(void *);
		friend void selfmon_thread_ext_func(void *);
		friend void *termination_thread_ext_func(void *);
		friend void *restarting_thread_ext_func(void *);
};


#if 0
template <class DemandType, class StateType>
class LowLevelLoop : public LowLevelLoop_Common
{
	public:
		LowLevelLoop() {enable_control_local=false;control_enabled=false;}
		~LowLevelLoop() {}

		
		void addSensor(SensingModule<StateType> *module) {
			LowLevelLoop_Common::addSensor(module);
		}
		void setControl(ControlModule<DemandType,StateType> *module) {
			LowLevelLoop_Common::setControl(module);
		}

	protected:
		StateType state;
		DemandType target;
		bool enable_control_local, control_enabled;

		int target_function() {
			DemandType demand;
			ControlModule<DemandType,StateType> *ctrl=
				dynamic_cast<ControlModule<DemandType,StateType> *>(control);

			int r = demandVar->t_readto(demand,max_command_age,1);
			if (r<0) {
				enable_control_local = false;
				return rtx_error("Fatal error while reading 'hmcDemand'");
			} else if (r>0) {
				/* timeout: the controlling program may be dead, just go into safe
				 * mode */
				enable_control_local = false;
			} else {
				enable_control_local = ctrl->checkDemand(demand);
			}

			if (enable_control_local) {
				ctrl->copySanitizedDemand(target,demand);
			} else {
				/* does write the demand back to the store, should it? */
				ctrl->setDefaultDemand(&demand);
			}
			return 0;
		}

		int sensing_function(){
			unsigned int i,err=0;
			bzero(&state, sizeof(StateType));
			for (i=0;i<sensors.size();i++){
				SensingModule<StateType> *ssr=
					dynamic_cast<SensingModule<StateType> *>(sensors[i]);
				if (ssr->sense(state)){
					rtx_message("Sensor %d failed",i);
					err += 1;
				}
			}
			return stateVar->t_writefrom(state) || err;
		}

		int control_function(){
			ControlModule<DemandType,StateType> *ctrl=
				dynamic_cast<ControlModule<DemandType,StateType> *>(control);
			if (enable_control_local) 
			{
				if (!control_enabled) {
					rtx_message("Autonomous control enabled");
					control_enabled = true;
				}
				ctrl->control(state,target);
			} else {
				if (control_enabled) {
					rtx_message("Autonomous control disabled");
					control_enabled = false;
				}
				ctrl->setDefaultControl(state);
			}
		}
};
#endif 



#endif // LOW_LEVEL_H
