#include <algorithm>

#include <math.h>
#include <stdlib.h>
#include <stdio.h>

#include "LowLevelLoop.h"

#define VERBOSE_THREAD 0
#define DEBUG_TERMINATION 0

LowLevelLoop::LowLevelLoop(bool restartOnLock) {
	watchdog = 1;
	watchdog_period = -1;
	restart_sleep_length = 3.0;
	mainThread = NULL;
	mainClockTimer = NULL;
	selfMonTimer = NULL;
	terminationSync = NULL;
	restartOnDeadlock = restartOnLock;
	sorted = false;
	end = false;
	runMutex = NULL;
	mainSync = NULL;
	terminationSync = NULL;
	terminated = true;
	selfMonRestarting = false;
	lastSelfMonWatchdog = 0;
}

LowLevelLoop::~LowLevelLoop() {
#if DEBUG_TERMINATION
	rtx_message("LowLevelLoop::destructor");
#endif
	/** Clearing the module list here. It is likely that the modules don't 
	 * exist anymore when this destructor is called, so we don't need to
	 * terminate them**/
	modules.clear();
	terminate();
#if DEBUG_TERMINATION
	rtx_message("LowLevelLoop::destructed");
#endif
}

int LowLevelLoop::init(DDXStore &store,
		RtxParamStream *pf)
{

	end = true;
	/** initialising individual modules **/
	unsigned int err = 0;;
	std::list<GenericSensoriMotorModule*>::iterator it;
	for (it=modules.begin();it!=modules.end();it++) {
		if ((*it)->init(store, pf)){
			rtx_message("LowLevelLoop::init: module(%s)]->init failed",
					(*it)->getName().c_str());
			err ++;
		}
	}
	return err;
}

/*************************************************
 *
 * Start threads  
 *
 * **********************************************/

/*************************************************
 *
 * SELF-MONITORING threads
 *
 * **********************************************/
void *restarting_thread_ext_func(void * that)
{
	LowLevelLoop * lll = (LowLevelLoop*)that;
	lll->restarting_thread_func();
	return NULL;
}

void LowLevelLoop::restarting_thread_func()
{

	rtx_message("Restarting task started");
	if (pause_main_thread()){
		rtx_message("LowLevelLoop::restart_thread_func: pause_main_thread failed");
		rtx_main_signal_shutdown();
	}

	rtx_message("Restarting task: main thread paused");

	std::list<GenericSensoriMotorModule*>::iterator it;
	for (it=modules.begin();it!=modules.end();it++) {
		if ((*it)->stop()){
			rtx_message("LowLevelLoop::restarting_thread_func: module(%s)->stop failed",
					(*it)->getName().c_str());
		}
	}

	rtx_message("Restarting task: modules stopped");
	fflush(NULL);
	rtx_timer_sleep(restart_sleep_length);

	rtx_message("Restarting task: restarting");
	for (it=modules.begin();it!=modules.end();it++) {
		if ((*it)->reset()){
			rtx_message("LowLevelLoop::restarting_thread_func: module(%s)->reset failed",
					(*it)->getName().c_str());
		}
	}
	rtx_message("Restarting task: modules reset");

	for (it=modules.begin();it!=modules.end();it++) {
		if ((*it)->start()){
			rtx_message("LowLevelLoop::restarting_thread_func: module(%s)->start failed",
					(*it)->getName().c_str());
		}
	}
	rtx_message("Restarting task: modules started");

	if (resume_main_thread()){
		rtx_message("LowLevelLoop::restart_thread_func: resume_main_thread failed");
		rtx_main_signal_shutdown();
	}

	rtx_timer_sleep(2*watchdog_period);
	rtx_message("Restarting task completed");

	selfMonRestarting = false;
}

void selfmon_thread_ext_func(void * that)
{
	LowLevelLoop * lll = (LowLevelLoop*)that;
	lll->selfmon_thread_func();
}

void LowLevelLoop::selfmon_thread_func()
{
	bool needsShutdown = false;

	/* don't try to check system state while we are restarting */
	if (selfMonRestarting) return;
	// rtx_message("Self mon: %d %d", watchdog, lastSelfMonWatchdog);

	if (lastSelfMonWatchdog == watchdog) {
		if (restartOnDeadlock) {
			rtx_message("Maing thread seems stopped. Resetting sensors");
			selfMonRestarting = true;
			RtxThread * th = 
				rtx_thread_create("Restarting Thread", VERBOSE_THREAD, RTX_THREAD_SCHED_OTHER,
						RTX_THREAD_PRIO_MIN, 0, RTX_THREAD_CANCEL_DEFERRED, 
						restarting_thread_ext_func, this, NULL, NULL);
			if (th == NULL) {
				rtx_message("Failed to create restarting thread");
				needsShutdown = true;
			} else {
				rtx_thread_detach(th);
			}
		} else {
			needsShutdown = true;
		}
		lastSelfMonWatchdog = 0;
	} else {
		lastSelfMonWatchdog = watchdog;
	}
	if (needsShutdown) {
		rtx_message("All else failed, sending shutdown");
		rtx_main_signal_shutdown();
	}

}

/*************************************************
 *
 * MAIN SENSING/CONTROL thread
 *
 * **********************************************/
void *main_thread_ext_func(void * that)
{
	LowLevelLoop * lll = (LowLevelLoop*)that;
	return lll->main_thread_func();
}

void *LowLevelLoop::main_thread_func()
{
	unsigned int i;
	RtxTime ts;
	double tsd,dts,lasttime = -1;
	while (!end) {
		std::list<GenericSensoriMotorModule*>::iterator it;
		if (rtx_sync_wait(mainSync)<0) {
			rtx_message("main_thread: fatal error while waiting for synchronisation. Shutting down");
			rtx_main_signal_shutdown();
			return NULL;
		}
		/* checking for timing */
		rtx_time_get(&ts);
		tsd = rtx_time_to_double(&ts);
		if (lasttime > 0) {
			dts = tsd - lasttime;
			if (fabs((dts - main_period)/main_period) > 0.25) {
				rtx_message("main_thread: warning, main period varied by more than 25%%: %f instead of %f", dts,main_period);
				for (it=modules.begin();it!=modules.end();it++) {
					rtx_message("\t%s: %.4fs",(*it)->getName().c_str(),
							(*it)->getRunTime());
				}
			} 
		}
		lasttime = tsd;

		bool err = false;
		rtx_mutex_lock(runMutex);
		for (it=modules.begin();it!=modules.end();it++) {
			//printf("running %s\n",(*it)->getName().c_str());
			if ((*it)->run_and_time()){
				rtx_message("LowLevelLoop::main_thread_func: module(%s)[%d]->run failed",
						(*it)->getName().c_str(), i);
				err = true;
				break;
			}
		}
		rtx_mutex_unlock(runMutex);
		if (err) {
			rtx_error_flush("Error detected in the run loop");
			return NULL;
		} else {
			watchdog += 1;
		}
#if 0
		if ((watchdog%20)==0) {
			rtx_message("Wd %d",watchdog);
		}
#endif
	}
	rtx_message("Main thread terminated");
	return NULL;

}

void main_clock_ext_func(void *that)
{
	LowLevelLoop * lll = (LowLevelLoop*)that;
	rtx_sync_broadcast(lll->mainSync);
}
		

/*************************************************
 *
 * Start 
 *
 * **********************************************/
int LowLevelLoop::start(double period, double wdog_period)
{
	unsigned int i;
	int prio = (geteuid()==0)?99:0;
	main_period = period;
	watchdog_period = wdog_period;

	end = false;
	runMutex = rtx_mutex_init(NULL,0,0);

	/** Order the module correctly for the run function in main_thread **/
	sortModules();

	/** First start the sensori-motor modules **/
	std::list<GenericSensoriMotorModule*>::iterator it;
	for (it=modules.begin();it!=modules.end();it++) {
		if ((*it)->start()){
			return rtx_error("LowLevelLoop::init: modules(%s)[%d]->start failed",
					(*it)->getName().c_str(), i);
		}
	}


	mainSync = rtx_sync_init(NULL,0,0,0);
	
	mainClockTimer = rtx_timer_create_thread(main_period,main_period,
			prio, main_clock_ext_func, this, RTX_TIMER_CLOCK_REALTIME);

	if (mainClockTimer == NULL) {
		rtx_sync_destroy(mainSync);
		rtx_mutex_destroy(runMutex);
		return rtx_error_errno("Failed to create main thread clock");
	}


	mainThread = rtx_thread_create("Main Thread", VERBOSE_THREAD, 
			(prio>0)?RTX_THREAD_SCHED_RR:RTX_THREAD_SCHED_OTHER,
			prio, 0, RTX_THREAD_CANCEL_DEFERRED, 
			main_thread_ext_func, this, NULL, NULL);

	if (mainThread == NULL) {
		rtx_timer_destroy_thread(mainClockTimer);
		rtx_sync_destroy(mainSync);
		rtx_mutex_destroy(runMutex);
		return rtx_error("Failed to create main thread");
	}

	/* We add a 2 second delay here, to be sure that the control thread is running 
	 * when we start checking for watchdog */
	selfMonTimer = rtx_timer_create_thread(2*watchdog_period,watchdog_period,
		   	0, selfmon_thread_ext_func, this, RTX_TIMER_CLOCK_REALTIME);
	rtx_message("started self mon timer, period = %f",watchdog_period);

	if (selfMonTimer == NULL) {
		rtx_timer_destroy_thread(mainClockTimer);
		rtx_sync_destroy(mainSync);
		rtx_mutex_destroy(runMutex);
		return rtx_error_errno("Failed to create self-monitoring thread");
	}

	terminated = false;


	return 0;
}

/*************************************************
 *
 * Pause 
 *
 * **********************************************/
int LowLevelLoop::pause_main_thread()
{
	end = true;
	rtx_timer_sleep(2*main_period);
	if (mainThread) rtx_thread_destroy_sync(mainThread);
	// rtx_sync_unlock(mainSync);
	rtx_mutex_unlock (runMutex);
	mainThread = NULL;
	return 0;
}

/*************************************************
 *
 * Resume 
 *
 * **********************************************/
int LowLevelLoop::resume_main_thread()
{
	int prio = (geteuid()==0)?99:0;
	if (mainThread) 
		return rtx_error("main_resume: main thread must be paused before resumed");

	end = false;
	mainThread = rtx_thread_create("Main Thread", VERBOSE_THREAD, 
			(prio>0)?RTX_THREAD_SCHED_RR:RTX_THREAD_SCHED_OTHER,
			prio, 0, RTX_THREAD_CANCEL_DEFERRED, 
			main_thread_ext_func, this, NULL, NULL);

	if (mainThread == NULL) {
		return rtx_error("Failed to resume main thread");
	}

	return 0;
}

/*************************************************
 *
 * Termination 
 *
 * **********************************************/
void *termination_thread_ext_func(void *that)
{
	LowLevelLoop * lll = (LowLevelLoop*)that;
	return lll->termination_thread_func();

}

void* LowLevelLoop::termination_thread_func()
{
	rtx_message("Starting termination thread");
	if (selfMonTimer) {
		rtx_timer_destroy_thread_sync(selfMonTimer);
		selfMonTimer = NULL;
	}
#if DEBUG_TERMINATION
	rtx_message("Selfmon timer terminated");
#endif
	pause_main_thread();
#if DEBUG_TERMINATION
	rtx_message("main thread terminated");
#endif

	if (mainClockTimer) {
		rtx_timer_destroy_thread_sync(mainClockTimer);
		mainClockTimer = NULL;
	}
	
#if DEBUG_TERMINATION
	rtx_message("Main timer terminated");
#endif
	
	//rtx_error_flush("Stopping modules");
	/** Finalize the sensori motor drivers **/
	std::list<GenericSensoriMotorModule*>::iterator it;
	for (it=modules.begin();it!=modules.end();it++) {
		if ((*it)->stop()){
			rtx_message("LowLevelLoop::init: modules(%s)->stop failed",
					(*it)->getName().c_str());
		}
		//rtx_error_flush("Stopped module '%s'",(*it)->getName().c_str());
	}
#if DEBUG_TERMINATION
	rtx_message("Modules stopped");
#endif
			
	/** Finalize the sensori motor drivers **/
	for (it=modules.begin();it!=modules.end();it++) {
		if ((*it)->terminate()){
			rtx_message("LowLevelLoop::init: modules(%s)->terminate failed",
					(*it)->getName().c_str());
		}
		//rtx_error_flush("Terminated module '%s'",(*it)->getName().c_str());
	}
#if DEBUG_TERMINATION
	rtx_message("Modules terminated");
#endif

	rtx_sync_broadcast(terminationSync);

	return NULL;
}

int LowLevelLoop::terminate()
{
	if (terminated) return 0;

	terminationSync = rtx_sync_init(NULL,0,0,0);

	//rtx_error_flush("Terminate threads");

	RtxThread *tthread = rtx_thread_create("termination thread",VERBOSE_THREAD,RTX_THREAD_SCHED_OTHER,
			RTX_THREAD_PRIO_MIN,0, RTX_THREAD_CANCEL_DEFERRED,
			termination_thread_ext_func,this,NULL,NULL);
	
	/*Give 2 seconds for the thread to terminate */
	switch (rtx_sync_timedwait(terminationSync, 12*main_period)){
		case +1:
			rtx_thread_destroy_sync(tthread);
			tthread = NULL;
			if (verbose) rtx_message("LowLevelLoop: termination sync timed out");
			if (mainThread) rtx_thread_destroy(mainThread);
			if (selfMonTimer) rtx_timer_destroy_thread(selfMonTimer);
			if (mainClockTimer) rtx_timer_destroy_thread(mainClockTimer);
			break;
		case 0:
			if (verbose) rtx_message("LowLevelLoop: termination condition signalled");
			rtx_thread_join(tthread);
			break;
		default:
			rtx_message("LowLevelLoop::terminate: rtx_sync_timed_wait failed");
	}
	if (tthread)
		rtx_thread_destroy_sync(tthread);
	
	
	//rtx_error_flush("Stopping modules");
	/** Finalize the sensori motor drivers **/
	std::list<GenericSensoriMotorModule*>::iterator it;
	for (it=modules.begin();it!=modules.end();it++) {
		if ((*it)->stop()){
			rtx_message("LowLevelLoop::init: modules(%s)->stop failed",
					(*it)->getName().c_str());
		}
		//rtx_error_flush("Stopped module '%s'",(*it)->getName().c_str());
	}
	
	//rtx_error_flush("Stopping syncs");
	if (mainSync) {
		rtx_sync_unlock(mainSync);
		rtx_sync_destroy(mainSync);
	}
	if (terminationSync) {
		rtx_sync_unlock(terminationSync);
		rtx_sync_destroy(terminationSync);
	}
	if (runMutex) {
		rtx_mutex_destroy(runMutex);
	}

	mainSync = NULL;
	terminationSync = NULL;
	runMutex = NULL;
	mainThread = NULL;
	mainClockTimer = NULL;
	selfMonTimer = NULL;

	//rtx_error_flush("Terminate Modules");
	//rtx_error_flush("Terminate End");
	terminated = true;

	//rtx_error_flush("Terminate End");
	terminated = true;

	return 0;
}


/***********************************************************
 *
 * Module sorting
 *
 * *******************************************************/

void LowLevelLoop::insertModule(std::list<GenericSensoriMotorModule*> &list,
		GenericSensoriMotorModule* module)
{
	if (find(list.begin(),list.end(),module) != list.end()) {
		return;
	} else if (module->getDependencies().empty()) {
		list.push_front(module);
	} else {
		GenericSensoriMotorModule::DependencySet::iterator it;
		it = module->getDependencies().begin();
		while (it != module->getDependencies().end()) {
			insertModule(list,*it);
			it++;
		}
		list.push_back(module);
	}
	return;
}


void LowLevelLoop::sortModules()
{
	std::list<GenericSensoriMotorModule*> new_modules;
	if (sorted) return;

	std::list<GenericSensoriMotorModule*>::iterator it;
	for (it=modules.begin();it!=modules.end();it++) {
		insertModule(new_modules, (*it));
	}

	modules = new_modules;
	sorted = true;
}

/***********************************************************
 *
 * Module validation
 *
 * *******************************************************/

int LowLevelLoop::validateModuleTree(GenericSensoriMotorModule* mod)
{
		const GenericSensoriMotorModule::DependencySet & deps = mod->getDependencies();
		GenericSensoriMotorModule::DependencySet::const_iterator dit = deps.begin();
		/** not the most efficient way of doing it, but this will do for now...
		 * **/
		while (dit != deps.end()) {
			if (find(modules.begin(),modules.end(),*dit) == modules.end()) {
				return rtx_error("In dependency for module '%s': module '%s' is not registered",
						mod->getName().c_str(), (*dit)->getName().c_str());
			}
			if (validateModuleTree(*dit)) {
				return rtx_error("Validation of dependency tree of module '%s' failed",
						(*dit)->getName().c_str());
			}
			dit++;
		}
		return 0;

}

int LowLevelLoop::validateModuleNetwork()
{
	std::list<GenericSensoriMotorModule*>::iterator it;
	for (it=modules.begin();it!=modules.end();it++) {
		if (validateModuleTree(*it)) {
			rtx_error_flush("Dependency tree for module '%s' is invalid",(*it)->getName().c_str());
			return -1;
		}
	}
	return 0;
}

/***********************************************************
 *
 * Module printing
 *
 * *******************************************************/

int LowLevelLoop::generateVCGModuleGraph(const std::string & filename) 
{
	FILE * fp = NULL;
	rtx_message("generateVCGModuleGraph");
	if (!filename.empty()) {
		fp = fopen(filename.c_str(),"w");
		if (fp == NULL) {
			return rtx_error("LowLevelLoop::generateVCGModuleGraph: can't open output file '%s'",
				filename.c_str());
		}
	}
	/** sort modules if not sorted already */
	sortModules();

	fprintf(fp,"graph:{\n");
	fprintf(fp,"title:\"Image Processing\"\n");
	fprintf(fp,"node.shape: box\n");
	fprintf(fp,"display_edge_labels: yes\n");
	fprintf(fp,"manhatten_edges: yes\n");
	fprintf(fp,"late_edge_labels: no\n");
	fprintf(fp,"dirty_edge_labels: no\n");
	fprintf(fp,"finetuning: yes\n");
	fprintf(fp,"nearedges: no\n");
	fprintf(fp,"splines: yes\n");
	fprintf(fp,"port_sharing: no\n");
	fprintf(fp,"crossingphase2: no\n");
	fprintf(fp,"crossingoptimization: yes\n");
	fprintf(fp,"yspace: 15\n");
	fprintf(fp,"xspace: 50\n");

	unsigned int i;
	// printing nodes
	std::list<GenericSensoriMotorModule*>::iterator it,jt;
	for (it=modules.begin(),i=0;it!=modules.end();it++,i++) {
		fprintf(fp,"node: {title: \"%03d\" label: \"%d:%s\"}\n",i,i,
				(*it)->getName().c_str());
	}
	// printing edges
	for (it=modules.begin(),i=0;it!=modules.end();it++,i++) {
		const GenericSensoriMotorModule::DependencySet & deps = (*it)->getDependencies();
		GenericSensoriMotorModule::DependencySet::const_iterator dit = deps.begin();
		/** not the most efficient way of doing it, but this will do for now...
		 * **/
		while (dit != deps.end()) {
			unsigned int j;
			signed int jm=-1;
			for (jt=modules.begin(),j=0;jt!=modules.end();jt++,j++) {
				if ((*jt) == *dit) {
					jm = j;
				}
			}
			if (jm < 0) {
				rtx_message("generateVCGModuleGraph Warning: Module %d:%s depends on module %s, not registered",
						i,(*it)->getName().c_str(),(*dit)->getName().c_str());
			} else {
				fprintf(fp,"edge: {sourcename: \"%03d\" targetname: \"%03d\" }\n",jm,i);
			}
			dit++;
		}
	}
	fprintf(fp,"\n}");
	if (fp != stdout) fclose(fp);
	rtx_message("generateVCGModuleGraph done");

	return 0;
}

int LowLevelLoop::generateDotModuleGraph(const std::string & filename) 
{
	FILE * fp = NULL;
	rtx_message("generateDotModuleGraph");
	if (!filename.empty()) {
		fp = fopen(filename.c_str(),"w");
		if (fp == NULL) {
			return rtx_error("LowLevelLoop::generateDotModuleGraph: can't open output file '%s'",
				filename.c_str());
		}
	}
	/** sort modules if not sorted already */
	sortModules();

	fprintf(fp,"digraph \"Low Level Loop\" {\n");
	fprintf(fp,"node [shape=box]\n");

	unsigned int i;
	// printing nodes
	std::list<GenericSensoriMotorModule*>::iterator it,jt;
	for (it=modules.begin(),i=0;it!=modules.end();it++,i++) {
		fprintf(fp,"%03d [label=\"%d:%s\"]\n",i,i,(*it)->getName().c_str());
	}
	// printing edges
	for (it=modules.begin(),i=0;it!=modules.end();it++,i++) {
		const GenericSensoriMotorModule::DependencySet & deps = (*it)->getDependencies();
		GenericSensoriMotorModule::DependencySet::const_iterator dit = deps.begin();
		/** not the most efficient way of doing it, but this will do for now...
		 * **/
		while (dit != deps.end()) {
			unsigned int j;
			signed int jm=-1;
			for (jt=modules.begin(),j=0;jt!=modules.end();jt++,j++) {
				if ((*jt) == *dit) {
					jm = j;
				}
			}
			if (jm < 0) {
				rtx_message("generateDotModuleGraph: Warning: Module %d:%s depends on module %s, not registered",
						i,(*it)->getName().c_str(),(*it)->getName().c_str());
			} else {
				fprintf(fp,"%03d -> %03d \n",jm,i);
			}
			dit++;
		}
	}
	fprintf(fp,"\n}");
	if (fp != stdout) fclose(fp);
	rtx_message("generateDotModuleGraph done");

	return 0;
}

