#ifndef DEMAND_MODULE_H
#define DEMAND_MODULE_H

#include <rtx/timer.h>
#include "SensoriMotorModule.h"
#include "MonitorThread.h"

template <class DemandClass>
class DemandModule : public SensoriMotorModule<DemandClass>, DDXVariableChecker
{
	protected:
		double maxCommandAge;
		MonitorThread mt;
		DDXVariable *demandVar;
	public:
		DemandModule(const std::string &name, DDXVariable *var, double maxVarAge=1.0) : 
			SensoriMotorModule<DemandClass>(name),
			mt(var,this,maxVarAge),
			demandVar(var)
		{
			maxCommandAge = maxVarAge;
			this->setDefaultDemand();
			demandVar->t_writefrom(this->moduleState);
			rtx_timer_sleep(0.02);
		}

		virtual ~DemandModule() {mt.stop();}

		/** Called on startup or after reset **/
		virtual int start() {return mt.start();}
		
		/** Called before a reset **/
		virtual int stop() {return mt.stop();}

		/** Set the data mutex for the monitoring thread 
		 * usually, from LowLevelLoop::getMutex() **/
		void setMutex(RtxMutex *mtx) {
			mt.setMutex(mtx);
		}
	protected:
		/*** Set of functions which may be declared by a 
		 * Demand module, expected to be redefined by 
		 * real implementation ***/

		/** Returns true if the demand can be used for control **/
		virtual bool checkDemand(const DemandClass & dmd) const {
			return true;
		}

		/** Copy the demand, eventually clipping the input **/
		virtual int copySanitizedDemand(const DemandClass & src) {
			memcpy(&this->moduleState,&src,sizeof(DemandClass));
			return 0;
		}

		/** Below are the functions required to implement the DDXVariableChecker
		 * interface **/

		/** Fill the demand with a sane default value **/
		virtual int setDefaultDemand() {
			bzero(&this->moduleState,sizeof(DemandClass));
			return 0;
		}

	private:
		/** The three next functions are also required, but must not be 
		 * overloaded */

		/** Returns the maximum allowed age of a variable for it 
		 * to be accepted **/
		double maxVariableAge() const {
			return this->maxCommandAge;
		}

		/** Returns true if the demand can be used for control **/
		bool checkDemand(DDXVariable &dmdvar) const {
			const DemandClass & dmd = dmdvar.value<DemandClass>();
			return this->checkDemand(dmd);
		}

		/** Copy the demand, eventually clipping the input **/
		int copySanitizedDemand(DDXVariable &dmdvar) {
			const DemandClass & dmd = dmdvar.value<DemandClass>();
			return this->copySanitizedDemand(dmd);
		}

};




#endif // DEMAND_MODULE_H
