#ifndef DEMAND_MODULE_H
#define DEMAND_MODULE_H

#include <rtx/timer.h>
#include "SensoriMotorModule.h"

template <class DemandClass>
class DemandModule : public SensoriMotorModule<DemandClass>
{
	protected:
		bool demand_accepted;
		double maxCommandAge;
		DDXVariable *demandVar;
	public:
		DemandModule(const std::string &name, DDXVariable *var, double maxVarAge=1.0) : 
			SensoriMotorModule<DemandClass>(name),
			demandVar(var)
		{
			maxCommandAge = maxVarAge;
			this->setDefaultDemand();
			demandVar->t_writefrom(this->moduleState);
			demand_accepted = false;
		}

		virtual ~DemandModule() {}

		virtual int run() {
			bool accept_variable = false;
			double age;
			if (demandVar->read(maxCommandAge,0)) {
				accept_variable = this->checkDemand(*demandVar);
			} else {
				RtxTime ts;
				rtx_time_get(&ts);
				age = rtx_time_to_double(&ts) - 
					demandVar->timeStamp();
				
				if (age > maxCommandAge) {
					/* timeout: the controlling program may be dead, just go into safe
					 * mode */
					accept_variable = false;
				} else {
					accept_variable = false;
					rtx_message("%s: Fatal error while reading 'demand'", 
							demandVar->getName().c_str());
					return -1;
				}
			}

			/** Logging status change **/
			if (accept_variable && !demand_accepted) {
				rtx_message("%s: demand accepted", 
						demandVar->getName().c_str());
			} else if (!accept_variable && demand_accepted) {
				rtx_message("%s: demand ignored (age %fs/%fs accept %d)", 
						demandVar->getName().c_str(), age, maxCommandAge,accept_variable);
			}
			demand_accepted = accept_variable;
			/** Updating the state **/
			if (demand_accepted) {
				this->copySanitizedDemand(*demandVar);
			} else {
				/* may write the demand back to the store, should it? */
				this->setDefaultDemand();
			}
			return 0;
		}


	protected:
		/*** Set of functions which may be declared by a 
		 * Demand module, expected to be redefined by 
		 * real implementation ***/

		/** Returns true if the demand can be used for control **/
		virtual bool checkDemand(const DemandClass & dmd) const {
			return true;
		}

		/** Copy the demand, eventually clipping the input **/
		virtual int copySanitizedDemand(const DemandClass & src) {
			memcpy(&this->moduleState,&src,sizeof(DemandClass));
			return 0;
		}

		/** Below are the functions required to implement the DDXVariableChecker
		 * interface **/

		/** Fill the demand with a sane default value **/
		virtual int setDefaultDemand() {
			bzero(&this->moduleState,sizeof(DemandClass));
			return 0;
		}

	private:
		/** The three next functions are also required, but must not be 
		 * overloaded */

		/** Returns the maximum allowed age of a variable for it 
		 * to be accepted **/
		double maxVariableAge() const {
			return this->maxCommandAge;
		}

		/** Returns true if the demand can be used for control **/
		bool checkDemand(DDXVariable &dmdvar) const {
			const DemandClass & dmd = dmdvar.value<DemandClass>();
			return this->checkDemand(dmd);
		}

		/** Copy the demand, eventually clipping the input **/
		int copySanitizedDemand(DDXVariable &dmdvar) {
			const DemandClass & dmd = dmdvar.value<DemandClass>();
			return this->copySanitizedDemand(dmd);
		}

};




#endif // DEMAND_MODULE_H
